//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  ibsha18.C - SHA-iButton utility functions.  
//              Requires: ibsha18o.c, crcutil.c, and ibsha18.h
//
//  Version:   2.00
//  History:   
//
//  Debug code designated with      //\\//\\//\\//\\//


#include "ownet.h"   
#include "ibsha18.h"

// exportable functions
int CreateAuthSecret(uchar *, TransState *);
int CreateMoneySecret(uchar *, TransState *);
int SelectCoProcessor(TransState *);
int FindSHA(uchar *, TransState *);
int WritePage(int, uchar *, int, TransState *);
int SelectRoving(TransState *);
int CreateUniqueDeviceSecret(uchar *, TransState *);
int ReadAuth(eCertificate *, TransState *);
int WriteMoney(eCertificate *, TransState *);
int FindCoprocessor(TransState *);
int VerifyMoney(eCertificate *, TransState *);
int FindRoving(TransState *);
int ReadBackAuth(eCertificate *,TransState *);
int RovingIsPresent(TransState *ts);
int WriteFileSHA(uchar *, uchar *, int, uchar *, TransState *);
int ReadFileSHA(int, uchar *, int *, uchar *, int *, int);
static int bitacc(int, int, int, uchar *);

// step constants
enum { ST_CHECK_COPROC=0, ST_CHECK_ROVING, ST_ERASE_SCRATCH, 
       ST_VERIFY, ST_FINISH, 
       ST_READ_SCRATCH, ST_WRITE_SCRATCH, ST_COPY_SCRATCH, ST_READ_AUTH_PAGE,
       ST_TARGET_C_MMS, ST_TARGET_C_AMS, ST_TARGET_C_UDS, ST_TARGET_R_UDS,
       ST_TARGET_T_UDS, ST_TARGET_PAGE, ST_TARGET_SECRET, ST_SET_BUFF_ACCNT,
       ST_SET_BUFF_FF, ST_COMPUTE_SHA, ST_SHA_COMPUTE_FIRST_SCRT,
       ST_SHA_COMPUTE_NEXT_SCRT, ST_SHA_VALIDATE_PAGE, ST_SET_BUFF_MONEY,
       ST_SHA_SIGN_PAGE, ST_SHA_COMPUTE_CHALLENGE, ST_INC_COUNTER,
       ST_SHA_AUTHENTICATE_HOST, ST_SET_BUFF_ROM, ST_SELECT_ROVING,
       ST_SELECT_COPR, ST_SET_BUFF_00, ST_SET_BUFF_RAND, ST_SET_RAND_NULL,
       ST_GET_NEW_RAND, ST_SET_BUFF_VAL, ST_SAVE_MONEY_MAC, ST_SAVE_AUTH_MAC,
       ST_SET_BUFF_ACCNT_FXT, ST_MATCH_SCRATCH_MONEY, ST_MATCH_SCRATCH_AUTH,
       ST_PRESENT,ST_MAKE_ECERT, ST_BUF_INPUT, ST_TARGET_DIRECTORY, 
       ST_FORMAT_UDP, ST_SEARCH_TARGET, ST_FIND_SHA };

// status contants
enum { STATUS_STEP_COMPLETE, STATUS_COMPLETE, STATUS_INPROGRESS,
       STATUS_ERROR_HALT, STATUS_ERROR_TRANSIENT };

// search to find a roving token
Script S_FindRoving[] = 
    {{ ST_SELECT_ROVING,"Select ROVING port" },
     { ST_SEARCH_TARGET,"Target search for SHA" },
     { ST_FIND_SHA,     "Search for SHA" },
     { ST_CHECK_ROVING, "Check if device found is roving token" },
     { ST_FIND_SHA,     "Search for SHA" },
     { ST_CHECK_ROVING, "Check if device found is roving token" },
     { ST_FINISH,       "Finished"}}; 

// search to find the first SHA device
Script S_FindSHA[] = 
    {{ ST_SEARCH_TARGET,"Target search for SHA" },
     { ST_FIND_SHA,     "Search for SHA" },
     { ST_FINISH,       "Finished"}}; 

// verify presence of roving 
Script S_RovingIsPresent[] = 
    {{ ST_SELECT_ROVING,"Select ROVING port" },
     { ST_PRESENT,      "Verify device present" },
     { ST_FINISH,       "Finished"}}; 

// search to find a token
Script S_FindCoprocessor[] = 
    {{ ST_SELECT_COPR,  "Select CO-PROCESSOR port" },
     { ST_SEARCH_TARGET,"Target search for SHA" },
     { ST_FIND_SHA,     "Search for SHA" },
     { ST_CHECK_COPROC, "Check if device found is co-processor token" },
     { ST_FIND_SHA,     "Search for SHA" },
     { ST_CHECK_COPROC, "Check if device found is co-processor token" },
     { ST_FINISH,       "Finished"}}; 

// create the auth secret for co-processor
Script S_CreateAuthSecret[] = 
    {{ ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_BUF_INPUT,      "C_Put auth secret in temp buffer" },
     // write auth secret to page
     { ST_TARGET_C_AMS,   "C_Target co-processor auth master secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad of the co-processor device" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" }, 
     // write FF's to scratchpad 
     { ST_SET_BUFF_FF,    "C_Set buffer to FF's" },
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad of the co-processor device" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     // compute first secret and copy to secret location
     { ST_SHA_COMPUTE_FIRST_SCRT, "C_Set SHA to compute first secret" },
     { ST_COMPUTE_SHA,    "C_Compute SHA" },
     { ST_TARGET_C_AMS,   "C_Target co-processor auth master secret" },
     { ST_TARGET_SECRET,  "C_Target secret" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad (set address)" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" }, 
     { ST_FINISH,         "Finished"}}; 

// create the money secret for co-processor
Script S_CreateMoneySecret[] = 
    {{ ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_BUF_INPUT,      "C_Put money secret in temp buffer" },
     // write money secret to page
     { ST_TARGET_C_MMS,   "C_Target co-processor money master secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad of the roving device" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" }, 
     // write FF's into scratchpad
     { ST_SET_BUFF_FF,    "C_Set buffer to FF's" },
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad of the co-processor device" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },
     // compute first secret and copy to secret location  
     { ST_SHA_COMPUTE_FIRST_SCRT, "C_Set SHA to compute first secret" },
     { ST_COMPUTE_SHA,    "C_Compute SHA" },
     { ST_TARGET_C_MMS,   "C_Target co-processor money master secret" },
     { ST_TARGET_SECRET,  "C_Target secret" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad (set address)" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" },
     { ST_FINISH,         "Finished"}}; 

// create the unique device secret for roaming ibutton
Script S_CreateUniqueDeviceSecret[] = 
    {{ ST_SELECT_ROVING,  "R_Select ROVING port" },
     // write auth secret to page
     { ST_BUF_INPUT,      "R_Put auth secret in temp buffer" },
     { ST_TARGET_R_UDS,   "R_Target roving unique secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_ERASE_SCRATCH,  "R_Erase the scratchpad of the roving device" }, 
     { ST_WRITE_SCRATCH,  "R_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "R_Copy the scratchpad" }, 
     // put constant (FF's) in the scratchpad
     { ST_SET_BUFF_FF,    "R_Set buffer to FF's" },
     { ST_ERASE_SCRATCH,  "R_Erase the scratchpad of the roving device" }, 
     { ST_WRITE_SCRATCH,  "R_Write the scratchpad" },  
     // compute the first secret and copy to secret location (auth secret)
     { ST_SHA_COMPUTE_FIRST_SCRT, "R_Set SHA to compute first secret" },
     { ST_COMPUTE_SHA,    "R_Compute SHA" },
     { ST_TARGET_R_UDS,   "R_Target roving unique secret" },
     { ST_TARGET_SECRET,  "R_Target secret" }, 
     { ST_WRITE_SCRATCH,  "R_Write the scratchpad (set address)" },  
     { ST_COPY_SCRATCH,   "R_Copy the scratchpad" }, 
     // put contant (FF's) in page
     { ST_SET_BUFF_FF,    "R_Set buffer to FF's" },
     { ST_TARGET_R_UDS,   "R_Target roving unique secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_ERASE_SCRATCH,  "R_Erase the scratchpad of the roving device" }, 
     { ST_WRITE_SCRATCH,  "R_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "R_Copy the scratchpad" }, 
     // put ROM in scratchpad
     { ST_TARGET_R_UDS,   "R_Target roving unique secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_ERASE_SCRATCH,  "R_Erase the scratchpad of the roving device" }, 
     { ST_SET_BUFF_ROM,   "R_Move ROM to temporary buffer" },
     { ST_WRITE_SCRATCH,  "R_Write the scratchpad" },
     // compute the next secret
     { ST_SHA_COMPUTE_NEXT_SCRT, "R_Set SHA to compute next secret" },
     { ST_COMPUTE_SHA,    "R_Compute SHA" },
     // copy to unique data secret
     { ST_SET_BUFF_FF,    "R_Set buffer to FF's" },
     { ST_TARGET_R_UDS,   "R_Target roving unique device secret" },
     { ST_TARGET_SECRET,  "R_Target secret" }, 
     { ST_WRITE_SCRATCH,  "R_Write the scratchpad (set address)" },  
     { ST_COPY_SCRATCH,   "R_Copy the scratchpad" }, 
     { ST_FINISH,         "Finished"}}; 

// Select the coprocessor 
Script S_SelectCoProcessor[] =
    {{ ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_FINISH,         "Finished"}}; 
     
// Select the roving
Script S_SelectRoving[] =
    {{ ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_FINISH,         "Finished"}}; 

// write a page specified to the current device
Script S_WritePage[] =
    {{ ST_BUF_INPUT,      "Put data to write in temp buffer" },
     { ST_FORMAT_UDP,     "Format data in temp buf to UDP" },
     { ST_ERASE_SCRATCH,  "Erase the scratchpad of the co-processor device" }, 
     { ST_WRITE_SCRATCH,  "Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "Copy the scratchpad" },
     { ST_FINISH,         "Finished"}}; 
      
// do a authenticated read of the roving monitary page
Script S_ReadAuth[] = 
    {// write 00 to co-processor auth master secret page
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad" }, 
     { ST_SET_BUFF_FF,    "C_Set buffer to FF's" },
     { ST_TARGET_C_AMS,   "C_Target co-processor auth master secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" }, 
     // write token ROM to coprocessor scratch
     { ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_SET_BUFF_ROM,   "R_Move ROM to temporary buffer" },
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     // compute device secret and save in co-processor unique device secret
     { ST_TARGET_C_AMS,   "C_Target co-processor auth master secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_SHA_COMPUTE_NEXT_SCRT, "C_Set SHA to compute next secret" },
     { ST_COMPUTE_SHA,    "C_Compute SHA" },
     { ST_TARGET_C_UDS,   "C_Target co-processor unique device secret" },
     { ST_TARGET_SECRET,  "C_Target secret" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad (set address)" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" }, 
     // write random challenge to scratch of target
     { ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_ERASE_SCRATCH,  "R_Erase the scratchpad" }, 
     { ST_TARGET_R_UDS,   "R_Target roving unique device secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_GET_NEW_RAND,   "R_Get new random value" },
     { ST_SET_BUFF_RAND,  "R_Move random to temporary buffer" },
     { ST_WRITE_SCRATCH,  "R_Write the scratchpad" },  
     // read data page to check       
     { ST_TARGET_R_UDS,   "R_Target roving unique device secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_READ_AUTH_PAGE, "R_Read Authenticate page" },
     { ST_MAKE_ECERT,     "R_Make eCert from read auth data" },
     // write data page to (unique secret) page of coprocessor
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad" }, 
     { ST_TARGET_C_UDS,   "C_Target co-processor unique device secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" }, 
     // build the temp buffer with counter / rom / challenge
     { ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_TARGET_R_UDS,   "R_Target roving unique device secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_SET_BUFF_VAL,   "R_Create the validate data in temporary buffer" },
     // write validate to scratch of coprocessor
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_TARGET_C_UDS,   "C_Target co-processor unique device secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     { ST_SHA_VALIDATE_PAGE, "C_Set SHA to validate page" },
     { ST_COMPUTE_SHA,    "C_Compute SHA" },
     // read scratch from token
     { ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_READ_SCRATCH,   "R_Read the scratchpad" },  
     { ST_SAVE_AUTH_MAC,  "R_Save the authenticate MAC" },
     // do match scratchpad    
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_MATCH_SCRATCH_AUTH,"C_Perform the match scratchpad on auth mac" },  
     { ST_FINISH,         "Finished"}}; 

// do a authenticated read of the roving monitary page
Script S_WriteMoney[] = 
    {// create packet for signing
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_SET_BUFF_ACCNT_FXT, "C_Set (fxt) account in temporary buffer" },
     // write to money page of co-processor
     { ST_TARGET_C_MMS,   "C_Target co-processor money master secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" }, 
     // read the roving money page data and counters
     { ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_TARGET_R_UDS,   "R_Target roving unique device secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     // create buffer for validate 
     { ST_SET_RAND_NULL,  "R_Set random value (challenge) to NULL" },
     { ST_SET_BUFF_VAL,   "R_Create the validate data in temporary buffer" },
     // write to scratch with validate info 
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_TARGET_C_MMS,   "C_Target co-processor money master secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     // perform a sign data page 
     { ST_SHA_SIGN_PAGE,  "C_Set SHA to sign data page" },
     { ST_COMPUTE_SHA,    "C_Compute SHA" },
     // read the MAC
     { ST_READ_SCRATCH,   "C_Read the scratchpad" },  
     { ST_SAVE_MONEY_MAC, "C_Save the money MAC" },
     // create the final account in the temp buffer
     { ST_SET_BUFF_ACCNT, "C_Set account in temporary buffer" },
     // write back to roving account page
     { ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_TARGET_R_UDS,   "R_Target roving unique device secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_ERASE_SCRATCH,  "R_Erase the scratchpad" }, 
     { ST_WRITE_SCRATCH,  "R_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "R_Copy the scratchpad" }, 
     { ST_FINISH,         "Finished"}}; 

// do a authenticated the money read (assume already in eCertificate)
Script S_VerifyMoney[] = 
    {// create packet verify
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     // write the fixed info, balance, data page
     { ST_TARGET_C_MMS,   "C_Target co-processor money master secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_SET_BUFF_ACCNT_FXT, "C_Set (fxt) account in temporary buffer" },
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" }, 
     // create the validation packet
     { ST_SET_RAND_NULL,  "C_Set random value (challenge) to NULL" },
     { ST_SET_BUFF_VAL,   "C_Create the validate data in temporary buffer" },
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     // validate page
     { ST_SHA_VALIDATE_PAGE, "C_Set SHA to validate page" },
     { ST_COMPUTE_SHA,    "C_Compute SHA" },
     // do match with money MAC
     { ST_MATCH_SCRATCH_MONEY,"C_Perform the match scratchpad on money mac" },  
     { ST_FINISH,         "Finished"}}; 

// do a authenticated read of the roving monitary page (skip creating unique secret)
Script S_ReadBackAuth[] = 
    {// write random challenge to scratch of target
     { ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_ERASE_SCRATCH,  "R_Erase the scratchpad" }, 
     { ST_TARGET_R_UDS,   "R_Target roving unique device secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_GET_NEW_RAND,   "R_Get new random value" },
     { ST_SET_BUFF_RAND,  "R_Move random to temporary buffer" },
     { ST_WRITE_SCRATCH,  "R_Write the scratchpad" },  
     // read data page to check       
     { ST_TARGET_R_UDS,   "R_Target roving unique device secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_READ_AUTH_PAGE, "R_Read Authenticate page" },
     { ST_MAKE_ECERT,     "R_Make eCert from read auth data" },
     // write data page to (unique secret) page of coprocessor
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_ERASE_SCRATCH,  "C_Erase the scratchpad" }, 
     { ST_TARGET_C_UDS,   "C_Target co-processor unique device secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     { ST_COPY_SCRATCH,   "C_Copy the scratchpad" }, 
     // build the temp buffer with counter / rom / challenge
     { ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_TARGET_R_UDS,   "R_Target roving unique device secret" },
     { ST_TARGET_PAGE,    "R_Target page of secret" }, 
     { ST_SET_BUFF_VAL,   "R_Create the validate data in temporary buffer" },
     // write validate to scratch of coprocessor
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_TARGET_C_UDS,   "C_Target co-processor unique device secret" },
     { ST_TARGET_PAGE,    "C_Target page of secret" }, 
     { ST_WRITE_SCRATCH,  "C_Write the scratchpad" },  
     { ST_SHA_VALIDATE_PAGE, "C_Set SHA to validate page" },
     { ST_COMPUTE_SHA,    "C_Compute SHA" },
     // read scratch from token
     { ST_SELECT_ROVING,  "R_Select ROVING port" },
     { ST_READ_SCRATCH,   "R_Read the scratchpad" },  
     { ST_SAVE_AUTH_MAC,  "R_Save the authenticate MAC" },
     // do match scratchpad    
     { ST_SELECT_COPR,    "C_Select CO-PROCESSOR port" },
     { ST_MATCH_SCRATCH_AUTH,"C_Perform the match scratchpad on auth mac" },  
     { ST_FINISH,         "Finished"}}; 

// external function prototypes
extern void output_status(int, char *);
extern int EraseScratchSHA(int);
extern int ReadScratchSHA(int,ushort *,uchar *,uchar *);
extern int WriteScratchSHA(int,ushort,uchar *,int);
extern int CopyScratchSHA(int,ushort,int);
extern int ReadAuthPageSHA(int,ushort,uchar *);
extern int ReadPageUDP(int,ushort,uchar *);
extern int ComputeSHA(int,ushort,uchar);
extern int MatchScratchSHA(int,uchar *);
extern int FormatUDP(int,ushort,uchar *,int *);
extern int ReadUDP_SHA(int,ushort,uchar *,int *);
extern int ReadFileSHA(int,uchar *,int *,uchar *,int *,int);
extern void setcrc16(int,ushort);
extern ushort docrc16(int,ushort);
extern int  owNext(int,int,int);
extern void owSerialNum(int,uchar *,int);
extern void owFamilySearchSetup(int,int);
extern int  owAccess(int);
extern int  owVerify(int,int);
extern int owSpeed(int,int);
extern int key_abort(void);
extern int TestModeMatch(int); //\\//\\//\\//

// Local Function Prototypes 
static int RunScriptSHA(Script script[], void *, int, TransState *);
static int ScriptStepSHA(Script *, int *, int *, int *, char *, void *, int, TransState *);
static int Make_eCertificate(ushort, uchar *, eCertificate *, TransState *);

// global to keep track of overdrive state
extern int in_overdrive[MAX_PORTNUM];

//--------------------------------------------------------------------------
// Write a page into the UDP format.  
//
// 'pg_num'    - page number to write
// 'data'      - data to write
// 'data_len'  - length of data to write
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int WritePage(int pg_num, uchar *data, int data_len, TransState *ts)
{
   uchar pg[33];
   int i;

   // create the buffer 
   // page number in the first byte
   pg[0] = (uchar)(pg_num);

   // page data in the rest
   for (i = 0; i < data_len; i++)
      pg[i + 1] = data[i];

   // adjust the length with the page byte 
   return RunScriptSHA(S_WritePage, pg, data_len + 1, ts);
}

//--------------------------------------------------------------------------
// Create the authenticate secret on the current device
//
// 'auth_secret' - 32 byte secret
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int CreateAuthSecret(uchar *auth_secret, TransState *ts)
{
   return RunScriptSHA(S_CreateAuthSecret, auth_secret, 32, ts);
}

//--------------------------------------------------------------------------
// Create the money secret on the current device
//
// 'money_secret' - 32 byte secret
// 'ts'           - pointer to a structure type that holds the transaction
//                  state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int CreateMoneySecret(uchar *money_secret, TransState *ts)
{
   return RunScriptSHA(S_CreateMoneySecret, money_secret, 32, ts);
}


//--------------------------------------------------------------------------
// Select the Co-processor port  
//
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int SelectCoProcessor(TransState *ts)
{
   return RunScriptSHA(S_SelectCoProcessor, NULL, 0, ts);
}

//--------------------------------------------------------------------------
// Find the first SHA iButton on the current port and return in rom_buff 
//
// 'rom_buf'     - rom buffer of device found
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int FindSHA(uchar *rom_buf, TransState *ts)
{
   return RunScriptSHA(S_FindSHA, rom_buf, 8, ts);
}

//--------------------------------------------------------------------------
// Search to find a valid co-processor SHA iButton on the current port.
//
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed,  Abort due to repeated errors
//                       or user keypress, or coprocessor not present
//
int FindCoprocessor(TransState *ts)
{
   uchar trom[8];

   // clear to use as a flag for a found coprocessor
   ts->copr_rom[0] = 0;

   if (RunScriptSHA(S_FindCoprocessor, trom, 8, ts))
   {
      // check if provider information found
      if (ts->copr_rom[0])
         return TRUE;
   }

   return FALSE;
}

//--------------------------------------------------------------------------
// Select the Roving port  
//
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int SelectRoving(TransState *ts)
{
   return RunScriptSHA(S_SelectRoving, NULL, 0, ts);
}

//--------------------------------------------------------------------------
// Create the unique device secret on the coprocessor using info from the
// roving device and the provided authenticate secret
//
// 'auth_secret' - 32 byte secret
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int CreateUniqueDeviceSecret(uchar *auth_secret, TransState *ts)
{
   return RunScriptSHA(S_CreateUniqueDeviceSecret, auth_secret, 32, ts);
}


//--------------------------------------------------------------------------
// Read authenticate the money page on the roving device using the 
// coprocessor.  Put the result certificate in 'ec'.
//
// 'ec'          - e Certificate that contains the money credential
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int ReadAuth(eCertificate *ec, TransState *ts)
{
   return RunScriptSHA(S_ReadAuth, ec, sizeof(*ec), ts);
}

//--------------------------------------------------------------------------
// Write a valid money page on a roving device using the coprocessor.
// The money certificate information is in the 'ec'.  
//
// 'ec'          - e Certificate that contains the money credential
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int WriteMoney(eCertificate *ec, TransState *ts)
{
   return RunScriptSHA(S_WriteMoney, ec, sizeof(*ec), ts);
}

//--------------------------------------------------------------------------
// Verify the money in the eCertificate is valid money using the 
// coprocessor.  
//
// 'ec'          - e Certificate that contains the money credential
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int VerifyMoney(eCertificate *ec, TransState *ts)
{
   return RunScriptSHA(S_VerifyMoney, ec, sizeof(*ec), ts);
}

//--------------------------------------------------------------------------
// Find a nonroving
//
// 'portnum'     - number 0 to MAX_PORTNUM-1.  This number is provided to
//                 indicate the symbolic port number.
// 'SerialNum'   - Device serial number to download
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int FindRoving(TransState *ts)
{
   uchar trom[8];

   // clear to use as a flag for a found coprocessor
   ts->rov_rom[0] = 0;

   if (RunScriptSHA(S_FindRoving, trom, 8, ts))
   {
      // check if provider information found
      if (ts->rov_rom[0])
         return TRUE;
   }

   return FALSE;
}

//--------------------------------------------------------------------------
// Read back authenticate the money page on the roving device using the 
// coprocessor.  Put the result certificate in 'ec'.  Same as ReadAuth
// except skips the step of creating the unique device secret.
//
// 'ec'          - e Certificate that contains the money credential
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int ReadBackAuth(eCertificate *ec, TransState *ts)
{
   return RunScriptSHA(S_ReadBackAuth, ec, sizeof(*ec), ts);
}

//--------------------------------------------------------------------------
// Verify that the roving device is present  
//
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Returns:   TRUE (1) : success, device present
//            FALSE (0): not completed.  Abort due to repeated errors
//                       or user keypress.
//
int RovingIsPresent(TransState *ts)
{
   return RunScriptSHA(S_RovingIsPresent, NULL, 0, ts);
}

//--------------------------------------------------------------------------
// Script functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Run the specified script. 
// Status is printed using 'output_status' function.  The script result is
// returned in the buffer 'prslt' of max length 'rslt_max'.  Result data 
// format is script dependent.
//
// 'script'      - script of steps
// 'prslt'       - pointer to result buffer
// 'rslt_max'    - max number of bytes size of result buffer
// 'ts'          - pointer to a structure type that holds the transaction
//                 state information.
//
// Return: TRUE - all steps completed
//         FALSE - all steps NOT completed due to error or escape 
//
int RunScriptSHA(Script script[], void *prslt, int rslt_max, TransState *ts)
{
   char statusmsg[256],LastDescription[256],LastMsg[256];
   char tmsg[256];
   int StepCount,SubStep,ErrorCount,Status;
   uchar *puchar;   

   // reset the step to the begining
   StepCount = 0;
   SubStep = 0;
   ErrorCount = 0;
   Status = STATUS_INPROGRESS;
   statusmsg[0] = 0;
   puchar = prslt;
 
   // start of script marker
   output_status(LV_VERBOSE,"-------------------------------------\n");

   // loop to perform all of the steps to download the Thermochron
   do
   {   
      // switch on the status of the last step done
      switch(Status)
      {
         // step complete so go to the next
         case STATUS_STEP_COMPLETE:
            StepCount++;
            SubStep = 0;
            ErrorCount = 0;
            Status = STATUS_INPROGRESS;
            LastDescription[0] = 0;
            LastMsg[0] = 0;
            break;
         // in progress so call again
         case STATUS_INPROGRESS:
            // print step description if different 
            if (strcmp(LastDescription,
                script[StepCount].StepDescription) != 0)
            {
               sprintf(tmsg,"(%d)%s -->",StepCount,script[StepCount].StepDescription);
               output_status(LV_VERBOSE,tmsg);
               sprintf(LastDescription,"%s",script[StepCount].StepDescription);
            }

            // perform a step in the job
            ScriptStepSHA(&script[StepCount],&SubStep, 
                      &Status, &ErrorCount, statusmsg, 
                      prslt, rslt_max, ts);

            // print results if different
            if (strcmp(LastMsg,statusmsg) != 0)
            {
               sprintf(tmsg,"%s\n",statusmsg); 
               output_status(LV_VERBOSE,tmsg);
               sprintf(LastMsg,"%s",statusmsg);
            }
            else
               output_status(LV_VERBOSE,".");
            break;
         // encountered a transient error
         case STATUS_ERROR_TRANSIENT:
            // if MAX_TRANS_ERROR tansient errors in a row then abort
            if (ErrorCount > MAX_TRANS_ERROR)
               Status = STATUS_ERROR_HALT;
            else
               Status = STATUS_INPROGRESS;
            break;
         // all steps complete
         case STATUS_COMPLETE:
            output_status(LV_VERBOSE,"End script normally\n");
            return TRUE;
         // non-recoverable error
         case STATUS_ERROR_HALT:
            // try to do emergency clean up
            output_status(LV_VERBOSE,"Aborting script due to non-recoverable error\n");
            return FALSE;
      }
   }
   while (!key_abort());

   // key abort
   output_status(LV_ALWAYS,"Aborting script due to key press\n");
   return FALSE;
}

//----------------------------------------------------------------------
//  Use the script to perform a step and return.
//
//  Return:
//          STATUS_STEP_COMPLETE - step complete, go to next step
//          STATUS_INPROGRESS - step is not complete, call again
//          STATUS_ERROR_HALT -  error, halt script
//          STATUS_ERROR_TRANSIENT - transient error, call again
//
int ScriptStepSHA(Script *StateScript, int *SubStep,
              int *Status, int *ErrorCount, char *statusmsg,
              void *prslt, int rslt_max, TransState *ts)
{
   int rslt,statuslen,j,i,cnt;
   uchar ROM[8];
   ulong tcnt;
   eCertificate *ec;
   uchar *puchar;
   uchar es;
   ushort lastcrc16;

   // do the current step and return
   switch (StateScript->Step)
   {
      // the operation is complete      
      case ST_FINISH:
         sprintf(statusmsg,"Operation complete");
         *Status = STATUS_COMPLETE;
         break; 

      // set the search to target the SHA iButton
      case ST_SEARCH_TARGET:
         owFamilySearchSetup(ts->portnum[ts->dev],SHA_FAM);
         sprintf(statusmsg,"Search setup to look for SHA iButton");
         *Status = STATUS_STEP_COMPLETE;
         break; 

      // search to find a SHA, verify is a SHA and not EQUAL to copr_rom[]
      case ST_FIND_SHA:
         // force back to standard speed
         owSpeed(ts->portnum[ts->dev],MODE_NORMAL);
         in_overdrive[ts->portnum[ts->dev]] = FALSE;

         for (;;)
         {
            // get first device in list with the specified family 
            if (owNext(ts->portnum[ts->dev],TRUE, FALSE))
            {
               // verify correct device type
               owSerialNum(ts->portnum[ts->dev],ROM, TRUE);   

               // compare to copr_rom
               for (i = 0; i < 8; i++)
               {
                  if (ts->copr_rom[i] != ROM[i])
                     break;
               }

               // check if correct type and not copr_rom 
               if ((SHA_FAM == (ROM[0] & 0x7F)) && (i != 8))
               {
                  // save the ROM to the return buffer
                  puchar = prslt;
                  owSerialNum(ts->portnum[ts->dev],puchar,TRUE);   

                  statuslen = sprintf(statusmsg,"SHA iButton found: ");
                  for (j = 7; j >= 0; j--)
                     statuslen += sprintf(&statusmsg[statuslen],"%02X",ROM[j]);
                  *Status = STATUS_STEP_COMPLETE;
                  break;
               }
            } 
            else
            {
               sprintf(statusmsg,"SHA iButton not on 1-Wire Net");
               *Status = STATUS_ERROR_HALT;
               break;
            }
         }
         break;

      // check if device found is a 'roving' type
      case ST_CHECK_ROVING:
         // read the roving service file
         rslt = ReadFileSHA(ts->portnum[ts->dev],ts->provider, &i, ts->temp_buffer, &ts->buf_len, 64);
         if (rslt)
         {
            // look at length to see if have user data
            if (ts->buf_len > 28)
            {
               // extract the user data in global
               for (j = 0; j < (ts->buf_len - 28); j++)
                  ts->user_data[j] = ts->temp_buffer[j];
               ts->user_data[j] = 0;

               // set the money page secret from the start page of file
               ts->r_udevice_scrt = i;
               // record the found rom as the 'roving' rom
               owSerialNum(ts->portnum[ts->dev],ts->rov_rom,TRUE);   
               // stop script short on success
               sprintf(statusmsg,"Roving device found (%d bytes user data)",ts->buf_len-28);
               *Status = STATUS_COMPLETE;  
               break;
            }
            else if (ts->buf_len == 28)
            {
               // no user data
               ts->user_data[0] = 0;
               // set the money page secret from the start page of file
               if (i > 7)
                  ts->r_udevice_scrt = i - 8;
               // record the found rom as the 'roving' rom
               owSerialNum(ts->portnum[ts->dev],ts->rov_rom,TRUE);   
               // stop script short on success
               sprintf(statusmsg,"Roving device found (no user data)");
               *Status = STATUS_COMPLETE;  
               break;
            }
            else
            {
               sprintf(statusmsg,"Roving money file incorrect size");
               *Status = STATUS_ERROR_HALT;
               break;
            }
         }
         sprintf(statusmsg,"Could not read directory");
         // check for error last time 
         *ErrorCount = *ErrorCount + 1;
         *Status = STATUS_ERROR_TRANSIENT;
         // check if close to erroring out of this step
         if (*ErrorCount >= MAX_TRANS_ERROR)
         {
            sprintf(statusmsg,"Giving up on this SHA iButton, no roving service file");
            *ErrorCount = 0;
            *Status = STATUS_STEP_COMPLETE;
         }
         break;
   
      // check if device found is a 'roving' type
      case ST_CHECK_COPROC:
         // read the co-processor information file
         rslt = ReadFileSHA(ts->portnum[ts->dev],ts->copr_file, &i, ts->temp_buffer, &ts->buf_len, 64);
         if (rslt)
         {
            // extract the information out of the copr_file
            if (ts->buf_len >= 8)
            {
               // get provider name
               for (i = 0; i < 5; i++)
                  ts->provider[i] = ts->temp_buffer[i];
               // get money master scrt number
               ts->c_mmaster_scrt = ts->temp_buffer[5];
               // auth master secret number
               ts->c_amaster_scrt = ts->temp_buffer[6];
               // unique device secret(calculated)
               ts->c_udevice_scrt = ts->temp_buffer[7];
               // record the found rom as the 'coprocessor' rom
               owSerialNum(ts->portnum[ts->dev],ts->copr_rom,TRUE);   
               // stop script short on success
               sprintf(statusmsg,"Co-processor found");
               *Status = STATUS_COMPLETE;  
               break;
            }
            else
            {
               sprintf(statusmsg,"Co-processor info file incorrect size");
               *Status = STATUS_ERROR_HALT;
               break;
            }
         }
         sprintf(statusmsg,"Could not read directory");
         // check for error last time 
         *ErrorCount = *ErrorCount + 1;
         *Status = STATUS_ERROR_TRANSIENT;
         // check if close to erroring out of this step
         if (*ErrorCount >= MAX_TRANS_ERROR)
         {
            sprintf(statusmsg,"Giving up on this SHA iButton, no co-processor info file");
            *ErrorCount = 0;
            *Status = STATUS_STEP_COMPLETE;
         }
         break;

      // erase the scratch of the token
      case ST_ERASE_SCRATCH:
         if (EraseScratchSHA(ts->portnum[ts->dev]))
         {
            sprintf(statusmsg,"Scratchpad erased");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            cnt = sprintf(statusmsg,"Could not erase scratchpad, device not present");
            //\\//\\//\\//\\// TRY TEST MODE FIX
            TestModeMatch(ts->portnum[ts->dev]);
            //\\//\\//\\//\\// END TEST MODE FIX

            // check for error last time 
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;

      // read the scratch of the token
      case ST_READ_SCRATCH:
         if (ReadScratchSHA(ts->portnum[ts->dev],&ts->address,&es,ts->temp_buffer))
         {
            sprintf(statusmsg,"Scratchpad read");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            sprintf(statusmsg,"Could not read scratchpad, device not present");
            // check for error last time 
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;
         
      // write the scratch of the token
      case ST_WRITE_SCRATCH:
         // adjust length if too long
         if (ts->buf_len > (0x20 - (ts->address & 0x1F)))
            i = (0x20 - (ts->address & 0x1F));
         else
            i = ts->buf_len;        
         if (WriteScratchSHA(ts->portnum[ts->dev],ts->address,ts->temp_buffer,i))
         {
            sprintf(statusmsg,"Scratchpad written");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            sprintf(statusmsg,"Could not write scratchpad, device not present");
            // check for error last time 
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;

      // copy the scratch to the token
      case ST_COPY_SCRATCH:
         if (CopyScratchSHA(ts->portnum[ts->dev],ts->address,ts->buf_len))
         {
            sprintf(statusmsg,"Scratchpad copy complete");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            sprintf(statusmsg,"Could not copy scratchpad, device not present");
            // check for error last time 
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;

      // read authenticate page
      case ST_READ_AUTH_PAGE:
         if (ReadAuthPageSHA(ts->portnum[ts->dev],ts->address,ts->temp_buffer))
         {
            sprintf(statusmsg,"Read Authenticate page complete");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            sprintf(statusmsg,"Could not read auth page, device not present");
            // check for error last time 
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;

      // try to make an eCertificate from the buffer contents
      case ST_MAKE_ECERT:
         // fill in the eCertificate for return
         if (Make_eCertificate(ts->address,ts->temp_buffer,(eCertificate*)prslt,ts))
         {
            sprintf(statusmsg,"eCertificate successfully made");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            // not valid money page 
            sprintf(statusmsg,"Not valid money page, abort");
            *Status = STATUS_ERROR_HALT;
         }
         break;

      // compute sha
      case ST_COMPUTE_SHA:
         if (ComputeSHA(ts->portnum[ts->dev],ts->address, ts->sha_cmd))
         {
            sprintf(statusmsg,"SHA computed");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            sprintf(statusmsg,"Could not compute SHA, device not present");
            // check for error last time 
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;

      // match scratch
      case ST_MATCH_SCRATCH_MONEY:
         // save mac 
         if (!prslt)
         {
            sprintf(statusmsg,"eCertificate object not present");
            *Status = STATUS_ERROR_HALT;
            break;
         }
         // get a pointer to the eCert
         ec = (eCertificate*)prslt;
         if (MatchScratchSHA(ts->portnum[ts->dev],ec->money_mac))
         {
            sprintf(statusmsg,"Scratchpad matched!");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            sprintf(statusmsg,"Scratchpad did not match");
            // check for error last time 
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;

      // match scratch
      case ST_MATCH_SCRATCH_AUTH:
         // save mac 
         if (!prslt)
         {
            sprintf(statusmsg,"eCertificate object not present");
            *Status = STATUS_ERROR_HALT;
            break;
         }
         // get a pointer to the eCert
         ec = (eCertificate*)prslt;
         if (MatchScratchSHA(ts->portnum[ts->dev],ec->auth_mac))
         {
            sprintf(statusmsg,"Scratchpad matched!");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            sprintf(statusmsg,"Scratchpad did not match");
            // check for error last time 
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;

      // check to see if the current device is present
      case ST_PRESENT:
         if (!owVerify(ts->portnum[ts->dev],FALSE))
         {
            // not present so try to force out of overdrive
            owSpeed(ts->portnum[ts->dev],MODE_NORMAL);
            in_overdrive[ts->portnum[ts->dev]] = FALSE;

            sprintf(statusmsg,"Device Not present");
            // check for error last time 
            *ErrorCount = *ErrorCount + 2;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         else
         {
            sprintf(statusmsg,"Device present");
            *Status = STATUS_STEP_COMPLETE;
         }
         break;

      // select the ROVING
      case ST_SELECT_ROVING:
         ts->dev = IB_ROVING;
         // if both device on same port, then set rom 
         if (ts->portnum[IB_ROVING] == ts->portnum[IB_COPR])
            owSerialNum(ts->portnum[IB_ROVING],ts->rov_rom, FALSE);   
         sprintf(statusmsg,"Token is now current device");
         *Status = STATUS_STEP_COMPLETE;
         break;

      // select the CO-PROCESSOR
      case ST_SELECT_COPR:
         ts->dev = IB_COPR;
         // if both device on same port, then set rom 
         if (ts->portnum[IB_ROVING] == ts->portnum[IB_COPR])
            owSerialNum(ts->portnum[IB_COPR],ts->copr_rom, FALSE);   
         sprintf(statusmsg,"CO-Processor is now current device");
         *Status = STATUS_STEP_COMPLETE;
         break;

      // selet the co-processor money master secret
      case ST_TARGET_C_MMS:
         // get the target secret
         ts->target_secret = ts->c_mmaster_scrt;
         sprintf(statusmsg,"Target secret %d",ts->target_secret);
         *Status = STATUS_STEP_COMPLETE;
         break;         

      // select the co-processor auth master secret
      case ST_TARGET_C_AMS:
         // get the target secret
         ts->target_secret = ts->c_amaster_scrt;
         sprintf(statusmsg,"Target secret %d",ts->target_secret);
         *Status = STATUS_STEP_COMPLETE;
         break;         

      // select the co-processor unique device secret(calculated)
      case ST_TARGET_C_UDS:
         // get the target secret
         ts->target_secret = ts->c_udevice_scrt;
         sprintf(statusmsg,"Target secret %d",ts->target_secret);
         *Status = STATUS_STEP_COMPLETE;
         break;         

      // select the roving unique device secret
      case ST_TARGET_R_UDS:
         // get the target secret
         ts->target_secret = ts->r_udevice_scrt;
         sprintf(statusmsg,"Target secret %d",ts->target_secret);
         *Status = STATUS_STEP_COMPLETE;
         break;

      // select the till unique device secret
      case ST_TARGET_T_UDS:
         // get the target secret
         ts->target_secret = ts->t_udevice_scrt;
         sprintf(statusmsg,"Target secret %d",ts->target_secret);
         *Status = STATUS_STEP_COMPLETE;
         break;         

      // select the page associated with the target secret (with counter)
      case ST_TARGET_PAGE:
         // calculate the desired page ts->address
         ts->address = 0x100 + (ushort)(ts->target_secret * 0x20);
         ts->buf_len = 32;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Target address set to %4X (page %d)",ts->address,ts->address / 32);
         break;         

      // select the directory page
      case ST_TARGET_DIRECTORY:
         ts->address = 0;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Target address set to %4X (page %d)",ts->address,ts->address / 32);
         break;         

      // select the page associated with the target secret (with counter)
      case ST_TARGET_SECRET:
         // calculate the desired page address
         ts->address = 0x200 + (ushort)(ts->target_secret * 0x08);
         ts->buf_len = 8;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Target address set to %04X (secret %d)",ts->address,ts->address / 8);
         break;         
   
      // set compute sha command
      case ST_SHA_COMPUTE_FIRST_SCRT:
         ts->sha_cmd = 0x0F;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"SHA command set (%02X)",ts->sha_cmd);
         break;         
      case ST_SHA_COMPUTE_NEXT_SCRT:
         ts->sha_cmd = 0xF0;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"SHA command set (%02X)",ts->sha_cmd);
         break;         
      case ST_SHA_VALIDATE_PAGE:
         ts->sha_cmd = 0x3C;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"SHA command set (%02X)",ts->sha_cmd);
         break;         
      case ST_SHA_SIGN_PAGE:
         ts->sha_cmd = 0xC3;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"SHA command set (%02X)",ts->sha_cmd);
         break;         
      case ST_SHA_COMPUTE_CHALLENGE:
         ts->sha_cmd = 0xCC;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"SHA command set (%02X)",ts->sha_cmd);
         break;         
      case ST_SHA_AUTHENTICATE_HOST:
         ts->sha_cmd = 0xAA;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"SHA command set (%02X)",ts->sha_cmd);
         break;         

      // set the temp buffer to 0xFF's
      case ST_SET_BUFF_FF:
         for (i = 0; i < 32; i++)
            ts->temp_buffer[i] = 0xFF;
         ts->buf_len = 32;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Temporary buffer set to FF's");
         break;         

      // set the temp buffer to 0x00's
      case ST_SET_BUFF_00:
         for (i = 0; i < 32; i++)
            ts->temp_buffer[i] = 0;
         ts->buf_len = 32;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Temporary buffer set to 00's");
         break;         

      // set the buffer to the ROM number
      case ST_SET_BUFF_ROM:
         owSerialNum(ts->portnum[ts->dev],ROM,TRUE);   
         for (i = 0; i < 32; i++)
            ts->temp_buffer[i] = 0xFF;
         for (i = 0; i < 7; i++)
            ts->temp_buffer[i+13] = (uchar)ROM[i];   
         ts->buf_len = 32;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Temporary buffer set to ROM at offset 13");
         break;         

      // get a new random challenge value
      case ST_GET_NEW_RAND:
         // seed random number generator with time
         srand((unsigned int)time(NULL)); 
         tcnt = rand(); 
         for (i = 0; i < 3; i++)
         {
            ts->randnum[i] = (uchar)(tcnt & 0xFF); 
            tcnt >>= 8;
         }
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"New random value retrieved");
         break;         

      // set random challenge value to NULL
      case ST_SET_RAND_NULL:
         for (i = 0; i < 3; i++)
            ts->randnum[i] = 0;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Random challenge is NULL");
         break;

      // set random value in 20 to 22 in temp buffer
      case ST_SET_BUFF_RAND:
         for (i = 0; i < 32; i++)
            ts->temp_buffer[i] = 0xFF;
         for (i = 0; i < 3; i++)
            ts->temp_buffer[i+20] = ts->randnum[i]; 
         ts->buf_len = 32;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Temporary buffer set to random value");
         break;         

      // save the MAC in the ts->temp_buffer as the money MAC in the eCert 
      case ST_SAVE_MONEY_MAC:
         // save mac 
         if (prslt)
         {
            ec = (eCertificate*)prslt;
            for (i = 0; i < 20; i++)
               ec->money_mac[i] = ts->temp_buffer[i + 8];
            *Status = STATUS_STEP_COMPLETE;
            sprintf(statusmsg,"Money MAC save in eCertificate");
         }
         else
         {
            sprintf(statusmsg,"eCertificate object not present");
            *Status = STATUS_ERROR_HALT;
         }
         break;

      // save the MAC in the temp_buffer as the auth MAC in the eCert 
      case ST_SAVE_AUTH_MAC:
         // save mac 
         if (prslt)
         {
            ec = (eCertificate*)prslt;
            for (i = 0; i < 20; i++)
               ec->auth_mac[i] = ts->temp_buffer[i + 8];
            *Status = STATUS_STEP_COMPLETE;
            sprintf(statusmsg,"Auth MAC save in eCertificate");
         }
         else
         {
            sprintf(statusmsg,"eCertificate object not present");
            *Status = STATUS_ERROR_HALT;
         }
         break;

      // set account info in temp buffer (money, transaction ID, MAC)
      case ST_SET_BUFF_ACCNT_FXT:
         // save mac 
         if (!prslt)
         {
            sprintf(statusmsg,"eCertificate object not present");
            *Status = STATUS_ERROR_HALT;
            break;
         }
         // get a pointer to the eCert
         ec = (eCertificate*)prslt;
         // length byte
         cnt = 0;
         ts->temp_buffer[cnt++] = 0x1D;
         // transaction ID
         ts->temp_buffer[cnt++] = ec->trans_id & 0xFF;
         ts->temp_buffer[cnt++] = (ec->trans_id >> 8) & 0xFF;
         // monetary unit code & Multiplier
         ts->temp_buffer[cnt++] = ec->code_mult & 0xFF;
         ts->temp_buffer[cnt++] = (ec->code_mult >> 8) & 0xFF;
         // money balance 
         for (i = 0; i < 3; i++)
            ts->temp_buffer[cnt++] = ec->balance[i];
         // MAC (set to all zeros)
         for (i = 0; i < 20; i++)
            ts->temp_buffer[cnt++] = 0x00;
         // certificate type & algorithm
         ts->temp_buffer[cnt++] = ec->cert_type;
         // continuation pointer
         ts->temp_buffer[cnt++] = ec->pkt_cont_ptr;
         // zero crc 
         ts->temp_buffer[cnt++] = 0x00;
         ts->temp_buffer[cnt++] = 0x00;
         ts->buf_len = 32;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Temporary buffer set to verify account file");
         break;
   
      // set account info in temp buffer (money, transaction ID, MAC)
      case ST_SET_BUFF_ACCNT:
         // save mac 
         if (!prslt)
         {
            sprintf(statusmsg,"eCertificate object not present");
            *Status = STATUS_ERROR_HALT;
            break;
         }
         // get a pointer to the eCert
         ec = (eCertificate*)prslt;
         // length byte
         cnt = 0;
         ts->temp_buffer[cnt++] = 0x1D;
         // transaction ID
         ts->temp_buffer[cnt++] = ec->trans_id & 0xFF;
         ts->temp_buffer[cnt++] = (ec->trans_id >> 8) & 0xFF;
         // monetary unit code & Multiplier
         ts->temp_buffer[cnt++] = ec->code_mult & 0xFF;
         ts->temp_buffer[cnt++] = (ec->code_mult >> 8) & 0xFF;
         // money balance 
         for (i = 0; i < 3; i++)
            ts->temp_buffer[cnt++] = ec->balance[i];
         // money MAC 
         for (i = 0; i < 20; i++)
            ts->temp_buffer[cnt++] = ec->money_mac[i];
         // certificate type & algorithm
         ts->temp_buffer[cnt++] = ec->cert_type;
         // continuation pointer
         ts->temp_buffer[cnt++] = ec->pkt_cont_ptr;
         // calculate crc if MAC is not all zeros
         setcrc16(ts->portnum[ts->dev], ec->money_page);
         // calculate crc16
         for (i = 0; i < cnt; i++)
            lastcrc16 = docrc16(ts->portnum[ts->dev],ts->temp_buffer[i]);
         // add the crc16
         ts->temp_buffer[cnt++] = (uchar)(~(lastcrc16 & 0xFF));
         ts->temp_buffer[cnt++] = (uchar)(~((lastcrc16 & 0xFF00) >> 8));
         ts->buf_len = 32;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Temporary buffer set to account file");
         break;

      // increment current page counter
      case ST_INC_COUNTER:
         if (prslt)
         {
            ec = (eCertificate*)prslt;
            ec->write_cycle_cnt++;
         }
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Page write counter incremented");
         break;

      // set counter/ page number/ rom / random value in temp buffer
      case ST_SET_BUFF_VAL:
         // save mac 
         if (!prslt)
         {
            sprintf(statusmsg,"eCertificate object not present");
            *Status = STATUS_ERROR_HALT;
            break;
         }
         // get a pointer to the eCert
         ec = (eCertificate*)prslt;
         // first set all to FF
         for (i = 0; i < 32; i++)
            ts->temp_buffer[i] = 0xFF;
         // write page counter
         tcnt = ec->write_cycle_cnt;
         for (i = 8; i <= 11; i++) 
         {
            ts->temp_buffer[i] = (uchar)(tcnt & 0xFF);
            tcnt >>= 8;
         }
         // page number 
         ts->temp_buffer[12] = ec->money_page;
         // rom number of token
         for (i = 0; i < 7; i++)
            ts->temp_buffer[i+13] = ec->serial_num[i];   
         // random challenge
         for (i = 0; i < 3; i++)
            ts->temp_buffer[i+20] = ts->randnum[i]; 
         ts->buf_len = 32;
         *Status = STATUS_STEP_COMPLETE;
         sprintf(statusmsg,"Temporary buffer set with validate data");
         break;         

      // set temp buffer from data in prslt
      case ST_BUF_INPUT:
         // check if have a pointer to something
         if (!prslt)
         {
            sprintf(statusmsg,"Input buffer object not present");
            *Status = STATUS_ERROR_HALT;
            break;
         }
         // get a pointer to the buffer
         puchar = (uchar *)prslt;
         for (i = 0; i < rslt_max; i++)
            ts->temp_buffer[i] = puchar[i];
         ts->buf_len = rslt_max;
         sprintf(statusmsg,"Input data copied to temporary buffer");
         *Status = STATUS_STEP_COMPLETE;
         break;

      // format temp buffer to UDP 
      case ST_FORMAT_UDP:
         // extract page number (address) and shift data
         ts->address = ts->temp_buffer[0] << 5;
         for (i = 1; i <= ts->buf_len; i++)
            ts->temp_buffer[i - 1] = ts->temp_buffer[i];
         ts->buf_len--;
         if (FormatUDP(ts->portnum[ts->dev],ts->address,ts->temp_buffer,&ts->buf_len))
         {
            // change the length to write the entire page (to use CRC16)
            ts->buf_len = 32;
            sprintf(statusmsg,"Temporary buffer no UDP format");
            *Status = STATUS_STEP_COMPLETE;
         }
         else   
         {
            sprintf(statusmsg,"Error converting to UDP (too long)");
            *Status = STATUS_ERROR_HALT;
         }
         break;

      default:
         *Status = STATUS_ERROR_HALT;
         sprintf(statusmsg,"1-Wire Step not known\n");
         break;
   }

   return *Status;
}

//------------------------------------------------------------------------
// Write the file 'filename' to the current device with the provided data.
// The pages used the file will be specified in the 'page_map' array.
// Each element in the array represents a page number (0 to 255).
// For example if the page data is 2 pages worth, page_am could contain
// an array (0x04,0x12) to specify page 4 and page 12.  
//
// *** Current version of this function will FORMAT the device before
//     writing the file!
//
int WriteFileSHA(uchar *filename, uchar *buf, int fllen, uchar *page_map, TransState *ts)
{
   uchar pgbuf[32];
   uchar newdir[] = { 0xAA, 0, 0x80, 0x01, 0, 0, 0, ' ', ' ', ' ', ' ', 0, 1, 1, 0 };
   int numpgs,flpg,i,cntleft,pos;

   // calculate the number of pages needed to write the file
   numpgs = (fllen / 28) + ((fllen % 28) ? 1 : 0);
   
   // put the file in the newdirectory
   for(i = 0; i < 5; i++)
      newdir[i+7] = filename[i];
   newdir[12] = page_map[0];
   newdir[13] = (uchar)numpgs;

   // set the bitmap in the directory page
   for (i = 0; i < numpgs; i++)
      bitacc(1,1,page_map[i],&newdir[3]);

   // loop to write the file 
   cntleft = fllen;  // count of bytes left to write  
   pos = 0; // current position in the buffer to write
   flpg = 0;
   while (cntleft > 0)
   {
      // get a page of data to write
      for (i = 0; i < ((cntleft > 28) ? 28 : cntleft); i++)
         pgbuf[i] = buf[pos++];
                                    
      // adjust the bytes left
      cntleft -= i;
                                    
      // set the next page pointer   
      pgbuf[i] = (cntleft == 0) ? 0 : page_map[flpg+1];
      
      // write the page and check to result
      if (!WritePage(page_map[flpg], &pgbuf[0], i + 1, ts))
         return FALSE;
      
      // set the next page
      flpg++;      
   } 

   // now write the directory page
   if (!WritePage(0, &newdir[0], 15, ts))
      return FALSE;

   return TRUE;
}

//--------------------------------------------------------------------------
//  Read file specified by 5 byte file name 'filename' and place in
//  'read_buf'.
//
int ReadFileSHA(int portnum, uchar *filename, int *start_pg, uchar *read_buf, 
                int *read_len, int max_read)
{
   uchar dirpg=0,pgbuf[32],filepg=0;
   int bufcnt=0,i,pglen;
               
   // set the intial read length to 0
   *read_len = 0;

   // loop read directory pages until the file entry is found
   do
   {
      // read a directory page
      if (!ReadUDP_SHA(portnum,(ushort)(dirpg << 5), pgbuf, &pglen))
         return FALSE;
         
      // if this is the first page make sure this is a directory
      // structure
      if (  ((dirpg == 0) && 
            ((pgbuf[0] != 0xAA) || (pgbuf[1] != 0) || (pglen < 7)))
            ||
            ((pglen-1) % 7) ) 
         return FALSE;
      
      // loop through each file entry in directory page (page 0 exception)
      for (i = (dirpg == 0) ? 7 : 0; i < 28; i += 7)
      {     
         // file entry found?
         if ((filename[0] == pgbuf[i]) &&
             (filename[1] == pgbuf[i+1]) &&
             (filename[2] == pgbuf[i+2]) &&
             (filename[3] == pgbuf[i+3]) &&
             (filename[4] == (pgbuf[i+4] & 0x7F)) )
         {
            // get the file starting page number
            filepg = pgbuf[i+5];
            *start_pg = (int)filepg;
            break;
         }
      }
      
      // get the next directory page (from page pointer) 
      dirpg = pgbuf[pglen-1];
   }
   while (dirpg && (filepg == 0));  
   
   // check if file found
   if (!filepg)
      return FALSE;
   
   // loop to read the file pages
   do
   {
      // read a file page
      if (!ReadUDP_SHA(portnum,(ushort)(filepg << 5), pgbuf, &pglen))
         return FALSE;
      
      // check for too big
      if ((pglen + bufcnt - 1) > max_read)
         return FALSE;

      // append the page data to the buffer
      for (i = 0; i < (pglen - 1); i++)
         read_buf[bufcnt++] = pgbuf[i];

      // get the next file page (from page pointer) 
      filepg = pgbuf[pglen-1];
   }
   while (filepg);

   // return the number of data bytes read
   *read_len = bufcnt;
   
   return TRUE;
}

//----------------------------------------------------------------------
// Verify that this is the correct format for a money page (UDP). Copy
// from RAW ecert to the ecert structure
//
// Return: TRUE - read sucessfull
//         FALSE - wrong format
//                                                           
int Make_eCertificate(ushort address, uchar *buf, eCertificate *ec, TransState *ts)
{
   int i,cnt;
   uchar ROM[8];
   ushort lastcrc16;

   // verify length
   if (buf[0] != 0x1D)
      return FALSE;

   // verify the crc16 
   setcrc16(ts->portnum[ts->dev],(ushort)(address >> 5));
   for (i = 0; i < buf[0] + 3; i++)
      lastcrc16 = docrc16(ts->portnum[ts->dev],buf[i]);
      
   // verify CRC16 is correct
   if (lastcrc16 != 0xB001)
      return FALSE;

   // copy the contents over into the eCertificate structure
   ec->obj_descriptor = 0x0501;
   ec->obj_len = sizeof(ec) - 3;
   for (i = 0; i < 20; i++)
      ec->constant_data[i] = 0;
   cnt = 0;
   ec->pkt_len = buf[cnt++];
   ec->trans_id = buf[cnt] | (buf[cnt+1] << 8);
   cnt += 2;
   ec->code_mult = buf[cnt] | (buf[cnt+1] << 8);
   cnt += 2;
   for (i = 0; i < 3; i++)
      ec->balance[i] = buf[cnt++];
   for (i = 0; i < 20; i++)
      ec->money_mac[i] = buf[cnt++];
   ec->cert_type = buf[cnt++];
   ec->pkt_cont_ptr = buf[cnt++];
   for (i = 0; i < 2; i++)
      ec->inv_crc[i] = buf[cnt++];
   ec->write_cycle_cnt = 0;
   for (i = 3; i >= 0; i--)
   {
      ec->write_cycle_cnt <<= 8;
      ec->write_cycle_cnt |= buf[cnt + i];
   }      
   cnt += 4;
   ec->money_page = address >> 5;

   owSerialNum(ts->portnum[ts->dev],ROM,TRUE);   
   for (i = 0; i < 7; i++)
      ec->serial_num[i] = (uchar)ROM[i];

   // get user data from global
   sprintf(ec->user_data,"%s",ts->user_data);

   return TRUE;
}

//--------------------------------------------------------------------------
//  Bit utility to read and write a bit in the buffer 'buf'.
//
int bitacc(int op, int state, int loc, uchar *buf)
{
     int nbyt,nbit;

     nbyt = (loc / 8);
     nbit = loc - (nbyt * 8);

     if (op == 1)
     {
          if (state)
             buf[nbyt] |= (0x01 << nbit);
          else
             buf[nbyt] &= ~(0x01 << nbit);

          return 1;
     }
     else
          return ((buf[nbyt] >> nbit) & 0x01);
}

