/*
Copyright 2019 The Knative Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package v1

import (
	"context"
	"crypto/x509"
	"encoding/pem"

	"knative.dev/pkg/apis"
)

// Destination represents a target of an invocation over HTTP.
type Destination struct {
	// Ref points to an Addressable.
	// +optional
	Ref *KReference `json:"ref,omitempty"`

	// URI can be an absolute URL(non-empty scheme and non-empty host) pointing to the target or a relative URI. Relative URIs will be resolved using the base URI retrieved from Ref.
	// +optional
	URI *apis.URL `json:"uri,omitempty"`

	// CACerts are Certification Authority (CA) certificates in PEM format
	// according to https://www.rfc-editor.org/rfc/rfc7468.
	// If set, these CAs are appended to the set of CAs provided
	// by the Addressable target, if any.
	// +optional
	CACerts *string `json:"CACerts,omitempty"`

	// Audience is the OIDC audience.
	// This need only be set, if the target is not an Addressable
	// and thus the Audience can't be received from the Addressable itself.
	// In case the Addressable specifies an Audience too, the Destinations
	// Audience takes preference.
	// +optional
	Audience *string `json:"audience,omitempty"`
}

// Validate the Destination has all the necessary fields and check the
// Namespace matches that of the parent object (using apis.ParentMeta).
func (d *Destination) Validate(ctx context.Context) *apis.FieldError {
	if d == nil {
		return nil
	}
	return ValidateDestination(ctx, *d).ViaField(apis.CurrentField)
}

// ValidateDestination validates Destination.
func ValidateDestination(ctx context.Context, dest Destination) *apis.FieldError {
	ref := dest.Ref
	uri := dest.URI
	caCerts := dest.CACerts
	if ref == nil && uri == nil {
		return apis.ErrGeneric("expected at least one, got none", "ref", "uri")
	}

	if ref != nil && uri != nil && uri.URL().IsAbs() {
		return apis.ErrGeneric("Absolute URI is not allowed when Ref or [apiVersion, kind, name] is present", "[apiVersion, kind, name]", "ref", "uri")
	}
	// IsAbs() check whether the URL has a non-empty scheme. Besides the non-empty scheme, we also require uri has a non-empty host
	if ref == nil && uri != nil && (!uri.URL().IsAbs() || uri.Host == "") {
		return apis.ErrInvalidValue("Relative URI is not allowed when Ref and [apiVersion, kind, name] is absent", "uri")
	}
	if ref != nil && uri == nil {
		return ref.Validate(ctx).ViaField("ref")
	}
	if caCerts != nil {
		return validateCACerts(caCerts)
	}
	return nil
}

// GetRef gets the KReference from this Destination, if one is present. If no ref is present,
// then nil is returned.
func (d *Destination) GetRef() *KReference {
	if d == nil {
		return nil
	}
	return d.Ref
}

func (d *Destination) SetDefaults(ctx context.Context) {
	if d == nil {
		return
	}

	if d.Ref != nil && d.Ref.Namespace == "" {
		d.Ref.Namespace = apis.ParentMeta(ctx).Namespace
	}
}

func validateCACerts(CACert *string) *apis.FieldError {
	// Check the object.
	var errs *apis.FieldError

	block, err := pem.Decode([]byte(*CACert))
	if err != nil && block == nil {
		errs = errs.Also(apis.ErrInvalidValue("CA Cert provided is invalid", "caCert"))
		return errs
	}
	if block.Type != "CERTIFICATE" {
		errs = errs.Also(apis.ErrInvalidValue("CA Cert provided is not a certificate", "caCert"))
	} else if _, err := x509.ParseCertificate(block.Bytes); err != nil {
		errs = errs.Also(apis.ErrInvalidValue("CA Cert provided is invalid", "caCert"))
	}
	return errs
}
