// Package computervision implements the Azure ARM Computervision service API version 3.1.
//
// The Computer Vision API provides state-of-the-art algorithms to process images and return information. For example,
// it can be used to determine if an image contains mature content, or it can be used to find all the faces in an
// image.  It also has other features like estimating dominant and accent colors, categorizing the content of images,
// and describing an image with complete English sentences.  Additionally, it can also intelligently generate images
// thumbnails for displaying large images effectively.
package computervision

// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License. See License.txt in the project root for license information.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"context"
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	"github.com/Azure/go-autorest/tracing"
	"github.com/gofrs/uuid"
	"io"
	"net/http"
)

// BaseClient is the base client for Computervision.
type BaseClient struct {
	autorest.Client
	Endpoint string
}

// New creates an instance of the BaseClient client.
func New(endpoint string) BaseClient {
	return NewWithoutDefaults(endpoint)
}

// NewWithoutDefaults creates an instance of the BaseClient client.
func NewWithoutDefaults(endpoint string) BaseClient {
	return BaseClient{
		Client:   autorest.NewClientWithUserAgent(UserAgent()),
		Endpoint: endpoint,
	}
}

// AnalyzeImage this operation extracts a rich set of visual features based on the image content.
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL. Within your request, there
// is an optional parameter to allow you to choose which features to return. By default, image categories are returned
// in the response.
// A successful response will be returned in JSON. If the request failed, the response will contain an error code and a
// message to help understand what went wrong.
// Parameters:
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
// visualFeatures - a string indicating what visual feature types to return. Multiple values should be
// comma-separated. Valid visual feature types include: Categories - categorizes image content according to a
// taxonomy defined in documentation. Tags - tags the image with a detailed list of words related to the image
// content. Description - describes the image content with a complete English sentence. Faces - detects if
// faces are present. If present, generate coordinates, gender and age. ImageType - detects if image is clipart
// or a line drawing. Color - determines the accent color, dominant color, and whether an image is black&white.
// Adult - detects if the image is pornographic in nature (depicts nudity or a sex act), or is gory (depicts
// extreme violence or blood). Sexually suggestive content (aka racy content) is also detected. Objects -
// detects various objects within an image, including the approximate location. The Objects argument is only
// available in English. Brands - detects various brands within an image, including the approximate location.
// The Brands argument is only available in English.
// details - a string indicating which domain-specific details to return. Multiple values should be
// comma-separated. Valid visual feature types include: Celebrities - identifies celebrities if detected in the
// image, Landmarks - identifies notable landmarks in the image.
// language - the desired language for output generation. If this parameter is not specified, the default value
// is &quot;en&quot;.Supported languages:en - English, Default. es - Spanish, ja - Japanese, pt - Portuguese,
// zh - Simplified Chinese.
// descriptionExclude - turn off specified domain models when generating the description.
func (client BaseClient) AnalyzeImage(ctx context.Context, imageURL ImageURL, visualFeatures []VisualFeatureTypes, details []Details, language string, descriptionExclude []DescriptionExclude) (result ImageAnalysis, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.AnalyzeImage")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "AnalyzeImage", err.Error())
	}

	req, err := client.AnalyzeImagePreparer(ctx, imageURL, visualFeatures, details, language, descriptionExclude)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImage", nil, "Failure preparing request")
		return
	}

	resp, err := client.AnalyzeImageSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImage", resp, "Failure sending request")
		return
	}

	result, err = client.AnalyzeImageResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImage", resp, "Failure responding to request")
		return
	}

	return
}

// AnalyzeImagePreparer prepares the AnalyzeImage request.
func (client BaseClient) AnalyzeImagePreparer(ctx context.Context, imageURL ImageURL, visualFeatures []VisualFeatureTypes, details []Details, language string, descriptionExclude []DescriptionExclude) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{}
	if visualFeatures != nil && len(visualFeatures) > 0 {
		queryParameters["visualFeatures"] = autorest.Encode("query", visualFeatures, ",")
	}
	if details != nil && len(details) > 0 {
		queryParameters["details"] = autorest.Encode("query", details, ",")
	}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}
	if descriptionExclude != nil && len(descriptionExclude) > 0 {
		queryParameters["descriptionExclude"] = autorest.Encode("query", descriptionExclude, ",")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/analyze"),
		autorest.WithJSON(imageURL),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// AnalyzeImageSender sends the AnalyzeImage request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) AnalyzeImageSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// AnalyzeImageResponder handles the response to the AnalyzeImage request. The method always
// closes the http.Response Body.
func (client BaseClient) AnalyzeImageResponder(resp *http.Response) (result ImageAnalysis, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// AnalyzeImageByDomain this operation recognizes content within an image by applying a domain-specific model. The list
// of domain-specific models that are supported by the Computer Vision API can be retrieved using the /models GET
// request. Currently, the API provides following domain-specific models: celebrities, landmarks.
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL.
// A successful response will be returned in JSON.
// If the request failed, the response will contain an error code and a message to help understand what went wrong.
// Parameters:
// model - the domain-specific content to recognize.
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
// language - the desired language for output generation. If this parameter is not specified, the default value
// is &quot;en&quot;.Supported languages:en - English, Default. es - Spanish, ja - Japanese, pt - Portuguese,
// zh - Simplified Chinese.
func (client BaseClient) AnalyzeImageByDomain(ctx context.Context, model string, imageURL ImageURL, language string) (result DomainModelResults, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.AnalyzeImageByDomain")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "AnalyzeImageByDomain", err.Error())
	}

	req, err := client.AnalyzeImageByDomainPreparer(ctx, model, imageURL, language)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImageByDomain", nil, "Failure preparing request")
		return
	}

	resp, err := client.AnalyzeImageByDomainSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImageByDomain", resp, "Failure sending request")
		return
	}

	result, err = client.AnalyzeImageByDomainResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImageByDomain", resp, "Failure responding to request")
		return
	}

	return
}

// AnalyzeImageByDomainPreparer prepares the AnalyzeImageByDomain request.
func (client BaseClient) AnalyzeImageByDomainPreparer(ctx context.Context, model string, imageURL ImageURL, language string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"model": autorest.Encode("path", model),
	}

	queryParameters := map[string]interface{}{}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPathParameters("/models/{model}/analyze", pathParameters),
		autorest.WithJSON(imageURL),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// AnalyzeImageByDomainSender sends the AnalyzeImageByDomain request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) AnalyzeImageByDomainSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// AnalyzeImageByDomainResponder handles the response to the AnalyzeImageByDomain request. The method always
// closes the http.Response Body.
func (client BaseClient) AnalyzeImageByDomainResponder(resp *http.Response) (result DomainModelResults, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// AnalyzeImageByDomainInStream this operation recognizes content within an image by applying a domain-specific model.
// The list of domain-specific models that are supported by the Computer Vision API can be retrieved using the /models
// GET request. Currently, the API provides following domain-specific models: celebrities, landmarks.
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL.
// A successful response will be returned in JSON.
// If the request failed, the response will contain an error code and a message to help understand what went wrong.
// Parameters:
// model - the domain-specific content to recognize.
// imageParameter - an image stream.
// language - the desired language for output generation. If this parameter is not specified, the default value
// is &quot;en&quot;.Supported languages:en - English, Default. es - Spanish, ja - Japanese, pt - Portuguese,
// zh - Simplified Chinese.
func (client BaseClient) AnalyzeImageByDomainInStream(ctx context.Context, model string, imageParameter io.ReadCloser, language string) (result DomainModelResults, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.AnalyzeImageByDomainInStream")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.AnalyzeImageByDomainInStreamPreparer(ctx, model, imageParameter, language)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImageByDomainInStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.AnalyzeImageByDomainInStreamSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImageByDomainInStream", resp, "Failure sending request")
		return
	}

	result, err = client.AnalyzeImageByDomainInStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImageByDomainInStream", resp, "Failure responding to request")
		return
	}

	return
}

// AnalyzeImageByDomainInStreamPreparer prepares the AnalyzeImageByDomainInStream request.
func (client BaseClient) AnalyzeImageByDomainInStreamPreparer(ctx context.Context, model string, imageParameter io.ReadCloser, language string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"model": autorest.Encode("path", model),
	}

	queryParameters := map[string]interface{}{}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPathParameters("/models/{model}/analyze", pathParameters),
		autorest.WithFile(imageParameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// AnalyzeImageByDomainInStreamSender sends the AnalyzeImageByDomainInStream request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) AnalyzeImageByDomainInStreamSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// AnalyzeImageByDomainInStreamResponder handles the response to the AnalyzeImageByDomainInStream request. The method always
// closes the http.Response Body.
func (client BaseClient) AnalyzeImageByDomainInStreamResponder(resp *http.Response) (result DomainModelResults, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// AnalyzeImageInStream this operation extracts a rich set of visual features based on the image content.
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL. Within your request, there
// is an optional parameter to allow you to choose which features to return. By default, image categories are returned
// in the response.
// A successful response will be returned in JSON. If the request failed, the response will contain an error code and a
// message to help understand what went wrong.
// Parameters:
// imageParameter - an image stream.
// visualFeatures - a string indicating what visual feature types to return. Multiple values should be
// comma-separated. Valid visual feature types include: Categories - categorizes image content according to a
// taxonomy defined in documentation. Tags - tags the image with a detailed list of words related to the image
// content. Description - describes the image content with a complete English sentence. Faces - detects if
// faces are present. If present, generate coordinates, gender and age. ImageType - detects if image is clipart
// or a line drawing. Color - determines the accent color, dominant color, and whether an image is black&white.
// Adult - detects if the image is pornographic in nature (depicts nudity or a sex act), or is gory (depicts
// extreme violence or blood). Sexually suggestive content (aka racy content) is also detected. Objects -
// detects various objects within an image, including the approximate location. The Objects argument is only
// available in English. Brands - detects various brands within an image, including the approximate location.
// The Brands argument is only available in English.
// details - a string indicating which domain-specific details to return. Multiple values should be
// comma-separated. Valid visual feature types include: Celebrities - identifies celebrities if detected in the
// image, Landmarks - identifies notable landmarks in the image.
// language - the desired language for output generation. If this parameter is not specified, the default value
// is &quot;en&quot;.Supported languages:en - English, Default. es - Spanish, ja - Japanese, pt - Portuguese,
// zh - Simplified Chinese.
// descriptionExclude - turn off specified domain models when generating the description.
func (client BaseClient) AnalyzeImageInStream(ctx context.Context, imageParameter io.ReadCloser, visualFeatures []VisualFeatureTypes, details []Details, language string, descriptionExclude []DescriptionExclude) (result ImageAnalysis, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.AnalyzeImageInStream")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.AnalyzeImageInStreamPreparer(ctx, imageParameter, visualFeatures, details, language, descriptionExclude)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImageInStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.AnalyzeImageInStreamSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImageInStream", resp, "Failure sending request")
		return
	}

	result, err = client.AnalyzeImageInStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "AnalyzeImageInStream", resp, "Failure responding to request")
		return
	}

	return
}

// AnalyzeImageInStreamPreparer prepares the AnalyzeImageInStream request.
func (client BaseClient) AnalyzeImageInStreamPreparer(ctx context.Context, imageParameter io.ReadCloser, visualFeatures []VisualFeatureTypes, details []Details, language string, descriptionExclude []DescriptionExclude) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{}
	if visualFeatures != nil && len(visualFeatures) > 0 {
		queryParameters["visualFeatures"] = autorest.Encode("query", visualFeatures, ",")
	}
	if details != nil && len(details) > 0 {
		queryParameters["details"] = autorest.Encode("query", details, ",")
	}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}
	if descriptionExclude != nil && len(descriptionExclude) > 0 {
		queryParameters["descriptionExclude"] = autorest.Encode("query", descriptionExclude, ",")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/analyze"),
		autorest.WithFile(imageParameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// AnalyzeImageInStreamSender sends the AnalyzeImageInStream request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) AnalyzeImageInStreamSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// AnalyzeImageInStreamResponder handles the response to the AnalyzeImageInStream request. The method always
// closes the http.Response Body.
func (client BaseClient) AnalyzeImageInStreamResponder(resp *http.Response) (result ImageAnalysis, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DescribeImage this operation generates a description of an image in human readable language with complete sentences.
// The description is based on a collection of content tags, which are also returned by the operation. More than one
// description can be generated for each image. Descriptions are ordered by their confidence score. Descriptions may
// include results from celebrity and landmark domain models, if applicable.
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL.
// A successful response will be returned in JSON. If the request failed, the response will contain an error code and a
// message to help understand what went wrong.
// Parameters:
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
// maxCandidates - maximum number of candidate descriptions to be returned.  The default is 1.
// language - the desired language for output generation. If this parameter is not specified, the default value
// is &quot;en&quot;.Supported languages:en - English, Default. es - Spanish, ja - Japanese, pt - Portuguese,
// zh - Simplified Chinese.
// descriptionExclude - turn off specified domain models when generating the description.
func (client BaseClient) DescribeImage(ctx context.Context, imageURL ImageURL, maxCandidates *int32, language string, descriptionExclude []DescriptionExclude) (result ImageDescription, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DescribeImage")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "DescribeImage", err.Error())
	}

	req, err := client.DescribeImagePreparer(ctx, imageURL, maxCandidates, language, descriptionExclude)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DescribeImage", nil, "Failure preparing request")
		return
	}

	resp, err := client.DescribeImageSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DescribeImage", resp, "Failure sending request")
		return
	}

	result, err = client.DescribeImageResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DescribeImage", resp, "Failure responding to request")
		return
	}

	return
}

// DescribeImagePreparer prepares the DescribeImage request.
func (client BaseClient) DescribeImagePreparer(ctx context.Context, imageURL ImageURL, maxCandidates *int32, language string, descriptionExclude []DescriptionExclude) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{}
	if maxCandidates != nil {
		queryParameters["maxCandidates"] = autorest.Encode("query", *maxCandidates)
	} else {
		queryParameters["maxCandidates"] = autorest.Encode("query", 1)
	}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}
	if descriptionExclude != nil && len(descriptionExclude) > 0 {
		queryParameters["descriptionExclude"] = autorest.Encode("query", descriptionExclude, ",")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/describe"),
		autorest.WithJSON(imageURL),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DescribeImageSender sends the DescribeImage request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DescribeImageSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DescribeImageResponder handles the response to the DescribeImage request. The method always
// closes the http.Response Body.
func (client BaseClient) DescribeImageResponder(resp *http.Response) (result ImageDescription, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DescribeImageInStream this operation generates a description of an image in human readable language with complete
// sentences. The description is based on a collection of content tags, which are also returned by the operation. More
// than one description can be generated for each image. Descriptions are ordered by their confidence score.
// Descriptions may include results from celebrity and landmark domain models, if applicable.
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL.
// A successful response will be returned in JSON. If the request failed, the response will contain an error code and a
// message to help understand what went wrong.
// Parameters:
// imageParameter - an image stream.
// maxCandidates - maximum number of candidate descriptions to be returned.  The default is 1.
// language - the desired language for output generation. If this parameter is not specified, the default value
// is &quot;en&quot;.Supported languages:en - English, Default. es - Spanish, ja - Japanese, pt - Portuguese,
// zh - Simplified Chinese.
// descriptionExclude - turn off specified domain models when generating the description.
func (client BaseClient) DescribeImageInStream(ctx context.Context, imageParameter io.ReadCloser, maxCandidates *int32, language string, descriptionExclude []DescriptionExclude) (result ImageDescription, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DescribeImageInStream")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DescribeImageInStreamPreparer(ctx, imageParameter, maxCandidates, language, descriptionExclude)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DescribeImageInStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.DescribeImageInStreamSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DescribeImageInStream", resp, "Failure sending request")
		return
	}

	result, err = client.DescribeImageInStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DescribeImageInStream", resp, "Failure responding to request")
		return
	}

	return
}

// DescribeImageInStreamPreparer prepares the DescribeImageInStream request.
func (client BaseClient) DescribeImageInStreamPreparer(ctx context.Context, imageParameter io.ReadCloser, maxCandidates *int32, language string, descriptionExclude []DescriptionExclude) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{}
	if maxCandidates != nil {
		queryParameters["maxCandidates"] = autorest.Encode("query", *maxCandidates)
	} else {
		queryParameters["maxCandidates"] = autorest.Encode("query", 1)
	}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}
	if descriptionExclude != nil && len(descriptionExclude) > 0 {
		queryParameters["descriptionExclude"] = autorest.Encode("query", descriptionExclude, ",")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/describe"),
		autorest.WithFile(imageParameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DescribeImageInStreamSender sends the DescribeImageInStream request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DescribeImageInStreamSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DescribeImageInStreamResponder handles the response to the DescribeImageInStream request. The method always
// closes the http.Response Body.
func (client BaseClient) DescribeImageInStreamResponder(resp *http.Response) (result ImageDescription, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DetectObjects performs object detection on the specified image.
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL.
// A successful response will be returned in JSON. If the request failed, the response will contain an error code and a
// message to help understand what went wrong.
// Parameters:
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
func (client BaseClient) DetectObjects(ctx context.Context, imageURL ImageURL) (result DetectResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DetectObjects")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "DetectObjects", err.Error())
	}

	req, err := client.DetectObjectsPreparer(ctx, imageURL)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DetectObjects", nil, "Failure preparing request")
		return
	}

	resp, err := client.DetectObjectsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DetectObjects", resp, "Failure sending request")
		return
	}

	result, err = client.DetectObjectsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DetectObjects", resp, "Failure responding to request")
		return
	}

	return
}

// DetectObjectsPreparer prepares the DetectObjects request.
func (client BaseClient) DetectObjectsPreparer(ctx context.Context, imageURL ImageURL) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/detect"),
		autorest.WithJSON(imageURL))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DetectObjectsSender sends the DetectObjects request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DetectObjectsSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DetectObjectsResponder handles the response to the DetectObjects request. The method always
// closes the http.Response Body.
func (client BaseClient) DetectObjectsResponder(resp *http.Response) (result DetectResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DetectObjectsInStream performs object detection on the specified image.
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL.
// A successful response will be returned in JSON. If the request failed, the response will contain an error code and a
// message to help understand what went wrong.
// Parameters:
// imageParameter - an image stream.
func (client BaseClient) DetectObjectsInStream(ctx context.Context, imageParameter io.ReadCloser) (result DetectResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DetectObjectsInStream")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DetectObjectsInStreamPreparer(ctx, imageParameter)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DetectObjectsInStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.DetectObjectsInStreamSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DetectObjectsInStream", resp, "Failure sending request")
		return
	}

	result, err = client.DetectObjectsInStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "DetectObjectsInStream", resp, "Failure responding to request")
		return
	}

	return
}

// DetectObjectsInStreamPreparer prepares the DetectObjectsInStream request.
func (client BaseClient) DetectObjectsInStreamPreparer(ctx context.Context, imageParameter io.ReadCloser) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/detect"),
		autorest.WithFile(imageParameter))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DetectObjectsInStreamSender sends the DetectObjectsInStream request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DetectObjectsInStreamSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DetectObjectsInStreamResponder handles the response to the DetectObjectsInStream request. The method always
// closes the http.Response Body.
func (client BaseClient) DetectObjectsInStreamResponder(resp *http.Response) (result DetectResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GenerateThumbnail this operation generates a thumbnail image with the user-specified width and height. By default,
// the service analyzes the image, identifies the region of interest (ROI), and generates smart cropping coordinates
// based on the ROI. Smart cropping helps when you specify an aspect ratio that differs from that of the input image.
// A successful response contains the thumbnail image binary. If the request failed, the response contains an error
// code and a message to help determine what went wrong.
// Upon failure, the error code and an error message are returned. The error code could be one of InvalidImageUrl,
// InvalidImageFormat, InvalidImageSize, InvalidThumbnailSize, NotSupportedImage, FailedToProcess, Timeout, or
// InternalServerError.
// Parameters:
// width - width of the thumbnail, in pixels. It must be between 1 and 1024. Recommended minimum of 50.
// height - height of the thumbnail, in pixels. It must be between 1 and 1024. Recommended minimum of 50.
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
// smartCropping - boolean flag for enabling smart cropping.
func (client BaseClient) GenerateThumbnail(ctx context.Context, width int32, height int32, imageURL ImageURL, smartCropping *bool) (result ReadCloser, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GenerateThumbnail")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: width,
			Constraints: []validation.Constraint{{Target: "width", Name: validation.InclusiveMaximum, Rule: int64(1024), Chain: nil},
				{Target: "width", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil}}},
		{TargetValue: height,
			Constraints: []validation.Constraint{{Target: "height", Name: validation.InclusiveMaximum, Rule: int64(1024), Chain: nil},
				{Target: "height", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil}}},
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "GenerateThumbnail", err.Error())
	}

	req, err := client.GenerateThumbnailPreparer(ctx, width, height, imageURL, smartCropping)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GenerateThumbnail", nil, "Failure preparing request")
		return
	}

	resp, err := client.GenerateThumbnailSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GenerateThumbnail", resp, "Failure sending request")
		return
	}

	result, err = client.GenerateThumbnailResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GenerateThumbnail", resp, "Failure responding to request")
		return
	}

	return
}

// GenerateThumbnailPreparer prepares the GenerateThumbnail request.
func (client BaseClient) GenerateThumbnailPreparer(ctx context.Context, width int32, height int32, imageURL ImageURL, smartCropping *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{
		"height": autorest.Encode("query", height),
		"width":  autorest.Encode("query", width),
	}
	if smartCropping != nil {
		queryParameters["smartCropping"] = autorest.Encode("query", *smartCropping)
	} else {
		queryParameters["smartCropping"] = autorest.Encode("query", false)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/generateThumbnail"),
		autorest.WithJSON(imageURL),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GenerateThumbnailSender sends the GenerateThumbnail request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GenerateThumbnailSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GenerateThumbnailResponder handles the response to the GenerateThumbnail request. The method always
// closes the http.Response Body.
func (client BaseClient) GenerateThumbnailResponder(resp *http.Response) (result ReadCloser, err error) {
	result.Value = &resp.Body
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK))
	result.Response = autorest.Response{Response: resp}
	return
}

// GenerateThumbnailInStream this operation generates a thumbnail image with the user-specified width and height. By
// default, the service analyzes the image, identifies the region of interest (ROI), and generates smart cropping
// coordinates based on the ROI. Smart cropping helps when you specify an aspect ratio that differs from that of the
// input image.
// A successful response contains the thumbnail image binary. If the request failed, the response contains an error
// code and a message to help determine what went wrong.
// Upon failure, the error code and an error message are returned. The error code could be one of InvalidImageUrl,
// InvalidImageFormat, InvalidImageSize, InvalidThumbnailSize, NotSupportedImage, FailedToProcess, Timeout, or
// InternalServerError.
// Parameters:
// width - width of the thumbnail, in pixels. It must be between 1 and 1024. Recommended minimum of 50.
// height - height of the thumbnail, in pixels. It must be between 1 and 1024. Recommended minimum of 50.
// imageParameter - an image stream.
// smartCropping - boolean flag for enabling smart cropping.
func (client BaseClient) GenerateThumbnailInStream(ctx context.Context, width int32, height int32, imageParameter io.ReadCloser, smartCropping *bool) (result ReadCloser, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GenerateThumbnailInStream")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: width,
			Constraints: []validation.Constraint{{Target: "width", Name: validation.InclusiveMaximum, Rule: int64(1024), Chain: nil},
				{Target: "width", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil}}},
		{TargetValue: height,
			Constraints: []validation.Constraint{{Target: "height", Name: validation.InclusiveMaximum, Rule: int64(1024), Chain: nil},
				{Target: "height", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "GenerateThumbnailInStream", err.Error())
	}

	req, err := client.GenerateThumbnailInStreamPreparer(ctx, width, height, imageParameter, smartCropping)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GenerateThumbnailInStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.GenerateThumbnailInStreamSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GenerateThumbnailInStream", resp, "Failure sending request")
		return
	}

	result, err = client.GenerateThumbnailInStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GenerateThumbnailInStream", resp, "Failure responding to request")
		return
	}

	return
}

// GenerateThumbnailInStreamPreparer prepares the GenerateThumbnailInStream request.
func (client BaseClient) GenerateThumbnailInStreamPreparer(ctx context.Context, width int32, height int32, imageParameter io.ReadCloser, smartCropping *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{
		"height": autorest.Encode("query", height),
		"width":  autorest.Encode("query", width),
	}
	if smartCropping != nil {
		queryParameters["smartCropping"] = autorest.Encode("query", *smartCropping)
	} else {
		queryParameters["smartCropping"] = autorest.Encode("query", false)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/generateThumbnail"),
		autorest.WithFile(imageParameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GenerateThumbnailInStreamSender sends the GenerateThumbnailInStream request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GenerateThumbnailInStreamSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GenerateThumbnailInStreamResponder handles the response to the GenerateThumbnailInStream request. The method always
// closes the http.Response Body.
func (client BaseClient) GenerateThumbnailInStreamResponder(resp *http.Response) (result ReadCloser, err error) {
	result.Value = &resp.Body
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK))
	result.Response = autorest.Response{Response: resp}
	return
}

// GetAreaOfInterest this operation returns a bounding box around the most important area of the image.
// A successful response will be returned in JSON. If the request failed, the response contains an error code and a
// message to help determine what went wrong.
// Upon failure, the error code and an error message are returned. The error code could be one of InvalidImageUrl,
// InvalidImageFormat, InvalidImageSize, NotSupportedImage, FailedToProcess, Timeout, or InternalServerError.
// Parameters:
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
func (client BaseClient) GetAreaOfInterest(ctx context.Context, imageURL ImageURL) (result AreaOfInterestResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetAreaOfInterest")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "GetAreaOfInterest", err.Error())
	}

	req, err := client.GetAreaOfInterestPreparer(ctx, imageURL)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GetAreaOfInterest", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetAreaOfInterestSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GetAreaOfInterest", resp, "Failure sending request")
		return
	}

	result, err = client.GetAreaOfInterestResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GetAreaOfInterest", resp, "Failure responding to request")
		return
	}

	return
}

// GetAreaOfInterestPreparer prepares the GetAreaOfInterest request.
func (client BaseClient) GetAreaOfInterestPreparer(ctx context.Context, imageURL ImageURL) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/areaOfInterest"),
		autorest.WithJSON(imageURL))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetAreaOfInterestSender sends the GetAreaOfInterest request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetAreaOfInterestSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetAreaOfInterestResponder handles the response to the GetAreaOfInterest request. The method always
// closes the http.Response Body.
func (client BaseClient) GetAreaOfInterestResponder(resp *http.Response) (result AreaOfInterestResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetAreaOfInterestInStream this operation returns a bounding box around the most important area of the image.
// A successful response will be returned in JSON. If the request failed, the response contains an error code and a
// message to help determine what went wrong.
// Upon failure, the error code and an error message are returned. The error code could be one of InvalidImageUrl,
// InvalidImageFormat, InvalidImageSize, NotSupportedImage, FailedToProcess, Timeout, or InternalServerError.
// Parameters:
// imageParameter - an image stream.
func (client BaseClient) GetAreaOfInterestInStream(ctx context.Context, imageParameter io.ReadCloser) (result AreaOfInterestResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetAreaOfInterestInStream")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetAreaOfInterestInStreamPreparer(ctx, imageParameter)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GetAreaOfInterestInStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetAreaOfInterestInStreamSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GetAreaOfInterestInStream", resp, "Failure sending request")
		return
	}

	result, err = client.GetAreaOfInterestInStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GetAreaOfInterestInStream", resp, "Failure responding to request")
		return
	}

	return
}

// GetAreaOfInterestInStreamPreparer prepares the GetAreaOfInterestInStream request.
func (client BaseClient) GetAreaOfInterestInStreamPreparer(ctx context.Context, imageParameter io.ReadCloser) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/areaOfInterest"),
		autorest.WithFile(imageParameter))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetAreaOfInterestInStreamSender sends the GetAreaOfInterestInStream request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetAreaOfInterestInStreamSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetAreaOfInterestInStreamResponder handles the response to the GetAreaOfInterestInStream request. The method always
// closes the http.Response Body.
func (client BaseClient) GetAreaOfInterestInStreamResponder(resp *http.Response) (result AreaOfInterestResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetReadResult this interface is used for getting OCR results of Read operation. The URL to this interface should be
// retrieved from 'Operation-Location' field returned from Read interface.
// Parameters:
// operationID - id of read operation returned in the response of the 'Read' interface.
func (client BaseClient) GetReadResult(ctx context.Context, operationID uuid.UUID) (result ReadOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetReadResult")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetReadResultPreparer(ctx, operationID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GetReadResult", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetReadResultSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GetReadResult", resp, "Failure sending request")
		return
	}

	result, err = client.GetReadResultResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "GetReadResult", resp, "Failure responding to request")
		return
	}

	return
}

// GetReadResultPreparer prepares the GetReadResult request.
func (client BaseClient) GetReadResultPreparer(ctx context.Context, operationID uuid.UUID) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"operationId": autorest.Encode("path", operationID),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPathParameters("/read/analyzeResults/{operationId}", pathParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetReadResultSender sends the GetReadResult request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetReadResultSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetReadResultResponder handles the response to the GetReadResult request. The method always
// closes the http.Response Body.
func (client BaseClient) GetReadResultResponder(resp *http.Response) (result ReadOperationResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ListModels this operation returns the list of domain-specific models that are supported by the Computer Vision API.
// Currently, the API supports following domain-specific models: celebrity recognizer, landmark recognizer.
// A successful response will be returned in JSON. If the request failed, the response will contain an error code and a
// message to help understand what went wrong.
func (client BaseClient) ListModels(ctx context.Context) (result ListModelsResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.ListModels")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.ListModelsPreparer(ctx)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "ListModels", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListModelsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "ListModels", resp, "Failure sending request")
		return
	}

	result, err = client.ListModelsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "ListModels", resp, "Failure responding to request")
		return
	}

	return
}

// ListModelsPreparer prepares the ListModels request.
func (client BaseClient) ListModelsPreparer(ctx context.Context) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/models"))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListModelsSender sends the ListModels request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ListModelsSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListModelsResponder handles the response to the ListModels request. The method always
// closes the http.Response Body.
func (client BaseClient) ListModelsResponder(resp *http.Response) (result ListModelsResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Read use this interface to get the result of a Read operation, employing the state-of-the-art Optical Character
// Recognition (OCR) algorithms optimized for text-heavy documents. When you use the Read interface, the response
// contains a field called 'Operation-Location'. The 'Operation-Location' field contains the URL that you must use for
// your 'GetReadResult' operation to access OCR results.​
// Parameters:
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
// language - the BCP-47 language code of the text in the document. Currently, only English ('en'), Dutch
// (‘nl’), French (‘fr’), German (‘de’), Italian (‘it’), Portuguese (‘pt), and Spanish ('es') are supported.
// Read supports auto language identification and multi-language documents, so only provide a language code if
// you would like to force the documented to be processed as that specific language.
func (client BaseClient) Read(ctx context.Context, imageURL ImageURL, language OcrDetectionLanguage) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.Read")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "Read", err.Error())
	}

	req, err := client.ReadPreparer(ctx, imageURL, language)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "Read", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReadSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "Read", resp, "Failure sending request")
		return
	}

	result, err = client.ReadResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "Read", resp, "Failure responding to request")
		return
	}

	return
}

// ReadPreparer prepares the Read request.
func (client BaseClient) ReadPreparer(ctx context.Context, imageURL ImageURL, language OcrDetectionLanguage) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/read/analyze"),
		autorest.WithJSON(imageURL),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReadSender sends the Read request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReadSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReadResponder handles the response to the Read request. The method always
// closes the http.Response Body.
func (client BaseClient) ReadResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ReadInStream use this interface to get the result of a Read operation, employing the state-of-the-art Optical
// Character Recognition (OCR) algorithms optimized for text-heavy documents. When you use the Read interface, the
// response contains a field called 'Operation-Location'. The 'Operation-Location' field contains the URL that you must
// use for your 'GetReadResult' operation to access OCR results.​
// Parameters:
// imageParameter - an image stream.
// language - the BCP-47 language code of the text in the document. Currently, only English ('en'), Dutch
// (‘nl’), French (‘fr’), German (‘de’), Italian (‘it’), Portuguese (‘pt), and Spanish ('es') are supported.
// Read supports auto language identification and multi-language documents, so only provide a language code if
// you would like to force the documented to be processed as that specific language.
func (client BaseClient) ReadInStream(ctx context.Context, imageParameter io.ReadCloser, language OcrDetectionLanguage) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.ReadInStream")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.ReadInStreamPreparer(ctx, imageParameter, language)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "ReadInStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.ReadInStreamSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "ReadInStream", resp, "Failure sending request")
		return
	}

	result, err = client.ReadInStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "ReadInStream", resp, "Failure responding to request")
		return
	}

	return
}

// ReadInStreamPreparer prepares the ReadInStream request.
func (client BaseClient) ReadInStreamPreparer(ctx context.Context, imageParameter io.ReadCloser, language OcrDetectionLanguage) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/read/analyze"),
		autorest.WithFile(imageParameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ReadInStreamSender sends the ReadInStream request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ReadInStreamSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ReadInStreamResponder handles the response to the ReadInStream request. The method always
// closes the http.Response Body.
func (client BaseClient) ReadInStreamResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RecognizePrintedText optical Character Recognition (OCR) detects text in an image and extracts the recognized
// characters into a machine-usable character stream.
// Upon success, the OCR results will be returned.
// Upon failure, the error code together with an error message will be returned. The error code can be one of
// InvalidImageUrl, InvalidImageFormat, InvalidImageSize, NotSupportedImage, NotSupportedLanguage, or
// InternalServerError.
// Parameters:
// detectOrientation - whether detect the text orientation in the image. With detectOrientation=true the OCR
// service tries to detect the image orientation and correct it before further processing (e.g. if it's
// upside-down).
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
// language - the BCP-47 language code of the text to be detected in the image. The default value is 'unk'.
func (client BaseClient) RecognizePrintedText(ctx context.Context, detectOrientation bool, imageURL ImageURL, language OcrLanguages) (result OcrResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RecognizePrintedText")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "RecognizePrintedText", err.Error())
	}

	req, err := client.RecognizePrintedTextPreparer(ctx, detectOrientation, imageURL, language)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "RecognizePrintedText", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecognizePrintedTextSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "RecognizePrintedText", resp, "Failure sending request")
		return
	}

	result, err = client.RecognizePrintedTextResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "RecognizePrintedText", resp, "Failure responding to request")
		return
	}

	return
}

// RecognizePrintedTextPreparer prepares the RecognizePrintedText request.
func (client BaseClient) RecognizePrintedTextPreparer(ctx context.Context, detectOrientation bool, imageURL ImageURL, language OcrLanguages) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{
		"detectOrientation": autorest.Encode("query", detectOrientation),
	}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "unk")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/ocr"),
		autorest.WithJSON(imageURL),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecognizePrintedTextSender sends the RecognizePrintedText request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecognizePrintedTextSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RecognizePrintedTextResponder handles the response to the RecognizePrintedText request. The method always
// closes the http.Response Body.
func (client BaseClient) RecognizePrintedTextResponder(resp *http.Response) (result OcrResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RecognizePrintedTextInStream optical Character Recognition (OCR) detects text in an image and extracts the
// recognized characters into a machine-usable character stream.
// Upon success, the OCR results will be returned.
// Upon failure, the error code together with an error message will be returned. The error code can be one of
// InvalidImageUrl, InvalidImageFormat, InvalidImageSize, NotSupportedImage, NotSupportedLanguage, or
// InternalServerError.
// Parameters:
// detectOrientation - whether detect the text orientation in the image. With detectOrientation=true the OCR
// service tries to detect the image orientation and correct it before further processing (e.g. if it's
// upside-down).
// imageParameter - an image stream.
// language - the BCP-47 language code of the text to be detected in the image. The default value is 'unk'.
func (client BaseClient) RecognizePrintedTextInStream(ctx context.Context, detectOrientation bool, imageParameter io.ReadCloser, language OcrLanguages) (result OcrResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RecognizePrintedTextInStream")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.RecognizePrintedTextInStreamPreparer(ctx, detectOrientation, imageParameter, language)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "RecognizePrintedTextInStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecognizePrintedTextInStreamSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "RecognizePrintedTextInStream", resp, "Failure sending request")
		return
	}

	result, err = client.RecognizePrintedTextInStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "RecognizePrintedTextInStream", resp, "Failure responding to request")
		return
	}

	return
}

// RecognizePrintedTextInStreamPreparer prepares the RecognizePrintedTextInStream request.
func (client BaseClient) RecognizePrintedTextInStreamPreparer(ctx context.Context, detectOrientation bool, imageParameter io.ReadCloser, language OcrLanguages) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{
		"detectOrientation": autorest.Encode("query", detectOrientation),
	}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "unk")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/ocr"),
		autorest.WithFile(imageParameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecognizePrintedTextInStreamSender sends the RecognizePrintedTextInStream request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecognizePrintedTextInStreamSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RecognizePrintedTextInStreamResponder handles the response to the RecognizePrintedTextInStream request. The method always
// closes the http.Response Body.
func (client BaseClient) RecognizePrintedTextInStreamResponder(resp *http.Response) (result OcrResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// TagImage this operation generates a list of words, or tags, that are relevant to the content of the supplied image.
// The Computer Vision API can return tags based on objects, living beings, scenery or actions found in images. Unlike
// categories, tags are not organized according to a hierarchical classification system, but correspond to image
// content. Tags may contain hints to avoid ambiguity or provide context, for example the tag "ascomycete" may be
// accompanied by the hint "fungus".
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL.
// A successful response will be returned in JSON. If the request failed, the response will contain an error code and a
// message to help understand what went wrong.
// Parameters:
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
// language - the desired language for output generation. If this parameter is not specified, the default value
// is &quot;en&quot;.Supported languages:en - English, Default. es - Spanish, ja - Japanese, pt - Portuguese,
// zh - Simplified Chinese.
func (client BaseClient) TagImage(ctx context.Context, imageURL ImageURL, language string) (result TagResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.TagImage")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("computervision.BaseClient", "TagImage", err.Error())
	}

	req, err := client.TagImagePreparer(ctx, imageURL, language)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "TagImage", nil, "Failure preparing request")
		return
	}

	resp, err := client.TagImageSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "TagImage", resp, "Failure sending request")
		return
	}

	result, err = client.TagImageResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "TagImage", resp, "Failure responding to request")
		return
	}

	return
}

// TagImagePreparer prepares the TagImage request.
func (client BaseClient) TagImagePreparer(ctx context.Context, imageURL ImageURL, language string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/tag"),
		autorest.WithJSON(imageURL),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// TagImageSender sends the TagImage request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) TagImageSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// TagImageResponder handles the response to the TagImage request. The method always
// closes the http.Response Body.
func (client BaseClient) TagImageResponder(resp *http.Response) (result TagResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// TagImageInStream this operation generates a list of words, or tags, that are relevant to the content of the supplied
// image. The Computer Vision API can return tags based on objects, living beings, scenery or actions found in images.
// Unlike categories, tags are not organized according to a hierarchical classification system, but correspond to image
// content. Tags may contain hints to avoid ambiguity or provide context, for example the tag "ascomycete" may be
// accompanied by the hint "fungus".
// Two input methods are supported -- (1) Uploading an image or (2) specifying an image URL.
// A successful response will be returned in JSON. If the request failed, the response will contain an error code and a
// message to help understand what went wrong.
// Parameters:
// imageParameter - an image stream.
// language - the desired language for output generation. If this parameter is not specified, the default value
// is &quot;en&quot;.Supported languages:en - English, Default. es - Spanish, ja - Japanese, pt - Portuguese,
// zh - Simplified Chinese.
func (client BaseClient) TagImageInStream(ctx context.Context, imageParameter io.ReadCloser, language string) (result TagResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.TagImageInStream")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.TagImageInStreamPreparer(ctx, imageParameter, language)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "TagImageInStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.TagImageInStreamSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "TagImageInStream", resp, "Failure sending request")
		return
	}

	result, err = client.TagImageInStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "computervision.BaseClient", "TagImageInStream", resp, "Failure responding to request")
		return
	}

	return
}

// TagImageInStreamPreparer prepares the TagImageInStream request.
func (client BaseClient) TagImageInStreamPreparer(ctx context.Context, imageParameter io.ReadCloser, language string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{}
	if len(string(language)) > 0 {
		queryParameters["language"] = autorest.Encode("query", language)
	} else {
		queryParameters["language"] = autorest.Encode("query", "en")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/vision/v3.1", urlParameters),
		autorest.WithPath("/tag"),
		autorest.WithFile(imageParameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// TagImageInStreamSender sends the TagImageInStream request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) TagImageInStreamSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// TagImageInStreamResponder handles the response to the TagImageInStream request. The method always
// closes the http.Response Body.
func (client BaseClient) TagImageInStreamResponder(resp *http.Response) (result TagResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}
