//
//  unicodelib.h
//
//  Copyright (c) 2020 Yuji Hirose. All rights reserved.
//  MIT License
//

#pragma once

#include <cstdlib>
#include <cstring>
#include <string>
#include <unordered_map>
#include <vector>

#if !defined(__cplusplus) || __cplusplus < 201703L
#error "Requires complete C++17 support"
#endif

namespace unicode {

//-----------------------------------------------------------------------------
// General Category
//-----------------------------------------------------------------------------

enum class GeneralCategory {
  Lu,
  Uppercase_Letter = Lu,
  Ll,
  Lowercase_Letter = Ll,
  Lt,
  Titlecase_Letter = Lt,
  Lm,
  Modifier_Letter = Lm,
  Lo,
  Other_Letter = Lo,
  Mn,
  Nonspacing_Mark = Mn,
  Mc,
  Spacing_Mark = Mc,
  Me,
  Enclosing_Mark = Me,
  Nd,
  Decimal_Number = Nd,
  Nl,
  Letter_Number = Nl,
  No,
  Other_Number = No,
  Pc,
  Connector_Punctuation = Pc,
  Pd,
  Dash_Punctuation = Pd,
  Ps,
  Open_Punctuation = Ps,
  Pe,
  Close_Punctuation = Pe,
  Pi,
  Initial_Punctuation = Pi,
  Pf,
  Final_Punctuation = Pf,
  Po,
  Other_Punctuation = Po,
  Sm,
  Math_Symbol = Sm,
  Sc,
  Currency_Symbol = Sc,
  Sk,
  Modifier_Symbol = Sk,
  So,
  Other_Symbol = So,
  Zs,
  Space_Separator = Zs,
  Zl,
  Line_Separator = Zl,
  Zp,
  Paragraph_Separator = Zp,
  Cc,
  Control = Cc,
  Cf,
  Format = Cf,
  Cs,
  Surrogate = Cs,
  Co,
  Private_Use = Co,
  Cn,
  Unassigned = Cn,
};

GeneralCategory general_category(char32_t cp);

bool is_cased_letter_category(GeneralCategory gc);
bool is_letter_category(GeneralCategory gc);
bool is_mark_category(GeneralCategory gc);
bool is_number_category(GeneralCategory gc);
bool is_punctuation_category(GeneralCategory gc);
bool is_symbol_category(GeneralCategory gc);
bool is_separator_category(GeneralCategory gc);
bool is_other_category(GeneralCategory gc);

bool is_cased_letter(char32_t cp);
bool is_letter(char32_t cp);
bool is_mark(char32_t cp);
bool is_number(char32_t cp);
bool is_punctuation(char32_t cp);
bool is_symbol(char32_t cp);
bool is_separator(char32_t cp);
bool is_other(char32_t cp);

//-----------------------------------------------------------------------------
// Property
//-----------------------------------------------------------------------------

bool is_white_space(char32_t cp);
bool is_bidi_control(char32_t cp);
bool is_join_control(char32_t cp);
bool is_dash(char32_t cp);
bool is_hyphen(char32_t cp);
bool is_quotation_mark(char32_t cp);
bool is_terminal_punctuation(char32_t cp);
bool is_other_math(char32_t cp);
bool is_hex_digit(char32_t cp);
bool is_ascii_hex_digit(char32_t cp);
bool is_other_alphabetic(char32_t cp);
bool is_ideographic(char32_t cp);
bool is_diacritic(char32_t cp);
bool is_extender(char32_t cp);
bool is_other_lowercase(char32_t cp);
bool is_other_uppercase(char32_t cp);
bool is_noncharacter_code_point(char32_t cp);
bool is_other_grapheme_extend(char32_t cp);
bool is_ids_binary_operator(char32_t cp);
bool is_ids_trinary_operator(char32_t cp);
bool is_radical(char32_t cp);
bool is_unified_ideograph(char32_t cp);
bool is_other_default_ignorable_code_point(char32_t cp);
bool is_deprecated(char32_t cp);
bool is_soft_dotted(char32_t cp);
bool is_logical_order_exception(char32_t cp);
bool is_other_id_start(char32_t cp);
bool is_other_id_continue(char32_t cp);
bool is_sentence_terminal(char32_t cp);
bool is_variation_selector(char32_t cp);
bool is_pattern_white_space(char32_t cp);
bool is_pattern_syntax(char32_t cp);
bool is_prepended_concatenation_mark(char32_t cp);

//-----------------------------------------------------------------------------
// Derived Property
//-----------------------------------------------------------------------------

bool is_math(char32_t cp);
bool is_alphabetic(char32_t cp);
bool is_lowercase(char32_t cp);
bool is_uppercase(char32_t cp);
bool is_cased(char32_t cp);
bool is_case_ignorable(char32_t cp);
bool is_changes_when_lowercased(char32_t cp);
bool is_changes_when_uppercased(char32_t cp);
bool is_changes_when_titlecased(char32_t cp);
bool is_changes_when_casefolded(char32_t cp);
bool is_changes_when_casemapped(char32_t cp);
bool is_id_start(char32_t cp);
bool is_id_continue(char32_t cp);
bool is_xid_start(char32_t cp);
bool is_xid_continue(char32_t cp);
bool is_default_ignorable_code_point(char32_t cp);
bool is_grapheme_extend(char32_t cp);
bool is_grapheme_base(char32_t cp);
bool is_grapheme_link(char32_t cp);

//-----------------------------------------------------------------------------
// Case
//-----------------------------------------------------------------------------

char32_t simple_uppercase_mapping(char32_t cp);
char32_t simple_lowercase_mapping(char32_t cp);
char32_t simple_titlecase_mapping(char32_t cp);
char32_t simple_case_folding(char32_t cp);

std::u32string to_uppercase(const char32_t *s32, size_t l,
                            const char *lang = nullptr);
std::u32string to_lowercase(const char32_t *s32, size_t l,
                            const char *lang = nullptr);
std::u32string to_titlecase(const char32_t *s32, size_t l,
                            const char *lang = nullptr);
std::u32string to_case_fold(
    const char32_t *s32, size_t l,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false);

bool is_uppercase(const char32_t *s32, size_t l);
bool is_lowercase(const char32_t *s32, size_t l);
bool is_titlecase(const char32_t *s32, size_t l);
bool is_case_fold(const char32_t *s32, size_t l);

bool caseless_match(
    const char32_t *s1, size_t l1, const char32_t *s2, size_t l2,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false);

bool canonical_caseless_match(
    const char32_t *s1, size_t l1, const char32_t *s2, size_t l2,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false);

bool compatibility_caseless_match(
    const char32_t *s1, size_t l1, const char32_t *s2, size_t l2,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false);

//-----------------------------------------------------------------------------
// Text Segmentation
//-----------------------------------------------------------------------------

bool is_graphic_character(char32_t cp);
bool is_base_character(char32_t cp);
bool is_combining_character(char32_t cp);

size_t combining_character_sequence_length(const char32_t *s32, size_t l);
size_t combining_character_sequence_count(const char32_t *s32, size_t l);

size_t extended_combining_character_sequence_length(const char32_t *s32,
                                                    size_t l);
size_t extended_combining_character_sequence_count(const char32_t *s32,
                                                   size_t l);

// The following functions support 'extended' grapheme cluster, not 'legacy'
// graphme cluster.
bool is_grapheme_boundary(const char32_t *s32, size_t l, size_t i);
size_t grapheme_length(const char32_t *s32, size_t l);
size_t grapheme_count(const char32_t *s32, size_t l);

bool is_word_boundary(const char32_t *s32, size_t l, size_t i);

bool is_sentence_boundary(const char32_t *s32, size_t l, size_t i);

//-----------------------------------------------------------------------------
// Block
//-----------------------------------------------------------------------------

// This is generated from 'Blocks.txt' in Unicode database.
// `python scripts/gen_property_values.py < UCD/Blocks.txt`
enum class Block {
  Unassigned,
  BasicLatin,
  Latin1Supplement,
  LatinExtendedA,
  LatinExtendedB,
  IPAExtensions,
  SpacingModifierLetters,
  CombiningDiacriticalMarks,
  GreekAndCoptic,
  Cyrillic,
  CyrillicSupplement,
  Armenian,
  Hebrew,
  Arabic,
  Syriac,
  ArabicSupplement,
  Thaana,
  NKo,
  Samaritan,
  Mandaic,
  SyriacSupplement,
  ArabicExtendedA,
  Devanagari,
  Bengali,
  Gurmukhi,
  Gujarati,
  Oriya,
  Tamil,
  Telugu,
  Kannada,
  Malayalam,
  Sinhala,
  Thai,
  Lao,
  Tibetan,
  Myanmar,
  Georgian,
  HangulJamo,
  Ethiopic,
  EthiopicSupplement,
  Cherokee,
  UnifiedCanadianAboriginalSyllabics,
  Ogham,
  Runic,
  Tagalog,
  Hanunoo,
  Buhid,
  Tagbanwa,
  Khmer,
  Mongolian,
  UnifiedCanadianAboriginalSyllabicsExtended,
  Limbu,
  TaiLe,
  NewTaiLue,
  KhmerSymbols,
  Buginese,
  TaiTham,
  CombiningDiacriticalMarksExtended,
  Balinese,
  Sundanese,
  Batak,
  Lepcha,
  OlChiki,
  CyrillicExtendedC,
  GeorgianExtended,
  SundaneseSupplement,
  VedicExtensions,
  PhoneticExtensions,
  PhoneticExtensionsSupplement,
  CombiningDiacriticalMarksSupplement,
  LatinExtendedAdditional,
  GreekExtended,
  GeneralPunctuation,
  SuperscriptsAndSubscripts,
  CurrencySymbols,
  CombiningDiacriticalMarksForSymbols,
  LetterlikeSymbols,
  NumberForms,
  Arrows,
  MathematicalOperators,
  MiscellaneousTechnical,
  ControlPictures,
  OpticalCharacterRecognition,
  EnclosedAlphanumerics,
  BoxDrawing,
  BlockElements,
  GeometricShapes,
  MiscellaneousSymbols,
  Dingbats,
  MiscellaneousMathematicalSymbolsA,
  SupplementalArrowsA,
  BraillePatterns,
  SupplementalArrowsB,
  MiscellaneousMathematicalSymbolsB,
  SupplementalMathematicalOperators,
  MiscellaneousSymbolsAndArrows,
  Glagolitic,
  LatinExtendedC,
  Coptic,
  GeorgianSupplement,
  Tifinagh,
  EthiopicExtended,
  CyrillicExtendedA,
  SupplementalPunctuation,
  CJKRadicalsSupplement,
  KangxiRadicals,
  IdeographicDescriptionCharacters,
  CJKSymbolsAndPunctuation,
  Hiragana,
  Katakana,
  Bopomofo,
  HangulCompatibilityJamo,
  Kanbun,
  BopomofoExtended,
  CJKStrokes,
  KatakanaPhoneticExtensions,
  EnclosedCJKLettersAndMonths,
  CJKCompatibility,
  CJKUnifiedIdeographsExtensionA,
  YijingHexagramSymbols,
  CJKUnifiedIdeographs,
  YiSyllables,
  YiRadicals,
  Lisu,
  Vai,
  CyrillicExtendedB,
  Bamum,
  ModifierToneLetters,
  LatinExtendedD,
  SylotiNagri,
  CommonIndicNumberForms,
  PhagsPa,
  Saurashtra,
  DevanagariExtended,
  KayahLi,
  Rejang,
  HangulJamoExtendedA,
  Javanese,
  MyanmarExtendedB,
  Cham,
  MyanmarExtendedA,
  TaiViet,
  MeeteiMayekExtensions,
  EthiopicExtendedA,
  LatinExtendedE,
  CherokeeSupplement,
  MeeteiMayek,
  HangulSyllables,
  HangulJamoExtendedB,
  HighSurrogates,
  HighPrivateUseSurrogates,
  LowSurrogates,
  PrivateUseArea,
  CJKCompatibilityIdeographs,
  AlphabeticPresentationForms,
  ArabicPresentationFormsA,
  VariationSelectors,
  VerticalForms,
  CombiningHalfMarks,
  CJKCompatibilityForms,
  SmallFormVariants,
  ArabicPresentationFormsB,
  HalfwidthAndFullwidthForms,
  Specials,
  LinearBSyllabary,
  LinearBIdeograms,
  AegeanNumbers,
  AncientGreekNumbers,
  AncientSymbols,
  PhaistosDisc,
  Lycian,
  Carian,
  CopticEpactNumbers,
  OldItalic,
  Gothic,
  OldPermic,
  Ugaritic,
  OldPersian,
  Deseret,
  Shavian,
  Osmanya,
  Osage,
  Elbasan,
  CaucasianAlbanian,
  LinearA,
  CypriotSyllabary,
  ImperialAramaic,
  Palmyrene,
  Nabataean,
  Hatran,
  Phoenician,
  Lydian,
  MeroiticHieroglyphs,
  MeroiticCursive,
  Kharoshthi,
  OldSouthArabian,
  OldNorthArabian,
  Manichaean,
  Avestan,
  InscriptionalParthian,
  InscriptionalPahlavi,
  PsalterPahlavi,
  OldTurkic,
  OldHungarian,
  HanifiRohingya,
  RumiNumeralSymbols,
  Yezidi,
  OldSogdian,
  Sogdian,
  Chorasmian,
  Elymaic,
  Brahmi,
  Kaithi,
  SoraSompeng,
  Chakma,
  Mahajani,
  Sharada,
  SinhalaArchaicNumbers,
  Khojki,
  Multani,
  Khudawadi,
  Grantha,
  Newa,
  Tirhuta,
  Siddham,
  Modi,
  MongolianSupplement,
  Takri,
  Ahom,
  Dogra,
  WarangCiti,
  DivesAkuru,
  Nandinagari,
  ZanabazarSquare,
  Soyombo,
  PauCinHau,
  Bhaiksuki,
  Marchen,
  MasaramGondi,
  GunjalaGondi,
  Makasar,
  LisuSupplement,
  TamilSupplement,
  Cuneiform,
  CuneiformNumbersAndPunctuation,
  EarlyDynasticCuneiform,
  EgyptianHieroglyphs,
  EgyptianHieroglyphFormatControls,
  AnatolianHieroglyphs,
  BamumSupplement,
  Mro,
  BassaVah,
  PahawhHmong,
  Medefaidrin,
  Miao,
  IdeographicSymbolsAndPunctuation,
  Tangut,
  TangutComponents,
  KhitanSmallScript,
  TangutSupplement,
  KanaSupplement,
  KanaExtendedA,
  SmallKanaExtension,
  Nushu,
  Duployan,
  ShorthandFormatControls,
  ByzantineMusicalSymbols,
  MusicalSymbols,
  AncientGreekMusicalNotation,
  MayanNumerals,
  TaiXuanJingSymbols,
  CountingRodNumerals,
  MathematicalAlphanumericSymbols,
  SuttonSignWriting,
  GlagoliticSupplement,
  NyiakengPuachueHmong,
  Wancho,
  MendeKikakui,
  Adlam,
  IndicSiyaqNumbers,
  OttomanSiyaqNumbers,
  ArabicMathematicalAlphabeticSymbols,
  MahjongTiles,
  DominoTiles,
  PlayingCards,
  EnclosedAlphanumericSupplement,
  EnclosedIdeographicSupplement,
  MiscellaneousSymbolsAndPictographs,
  Emoticons,
  OrnamentalDingbats,
  TransportAndMapSymbols,
  AlchemicalSymbols,
  GeometricShapesExtended,
  SupplementalArrowsC,
  SupplementalSymbolsAndPictographs,
  ChessSymbols,
  SymbolsAndPictographsExtendedA,
  SymbolsForLegacyComputing,
  CJKUnifiedIdeographsExtensionB,
  CJKUnifiedIdeographsExtensionC,
  CJKUnifiedIdeographsExtensionD,
  CJKUnifiedIdeographsExtensionE,
  CJKUnifiedIdeographsExtensionF,
  CJKCompatibilityIdeographsSupplement,
  CJKUnifiedIdeographsExtensionG,
  Tags,
  VariationSelectorsSupplement,
  SupplementaryPrivateUseAreaA,
  SupplementaryPrivateUseAreaB,
};

Block block(char32_t cp);

//-----------------------------------------------------------------------------
// Script
//-----------------------------------------------------------------------------

// This is generated from 'Scripts.txt' in Unicode database.
// `python scripts/gen_property_values.py < UCD/Scripts.txt`
enum class Script {
  Unassigned,
  Common,
  Latin,
  Greek,
  Cyrillic,
  Armenian,
  Hebrew,
  Arabic,
  Syriac,
  Thaana,
  Devanagari,
  Bengali,
  Gurmukhi,
  Gujarati,
  Oriya,
  Tamil,
  Telugu,
  Kannada,
  Malayalam,
  Sinhala,
  Thai,
  Lao,
  Tibetan,
  Myanmar,
  Georgian,
  Hangul,
  Ethiopic,
  Cherokee,
  Canadian_Aboriginal,
  Ogham,
  Runic,
  Khmer,
  Mongolian,
  Hiragana,
  Katakana,
  Bopomofo,
  Han,
  Yi,
  Old_Italic,
  Gothic,
  Deseret,
  Inherited,
  Tagalog,
  Hanunoo,
  Buhid,
  Tagbanwa,
  Limbu,
  Tai_Le,
  Linear_B,
  Ugaritic,
  Shavian,
  Osmanya,
  Cypriot,
  Braille,
  Buginese,
  Coptic,
  New_Tai_Lue,
  Glagolitic,
  Tifinagh,
  Syloti_Nagri,
  Old_Persian,
  Kharoshthi,
  Balinese,
  Cuneiform,
  Phoenician,
  Phags_Pa,
  Nko,
  Sundanese,
  Lepcha,
  Ol_Chiki,
  Vai,
  Saurashtra,
  Kayah_Li,
  Rejang,
  Lycian,
  Carian,
  Lydian,
  Cham,
  Tai_Tham,
  Tai_Viet,
  Avestan,
  Egyptian_Hieroglyphs,
  Samaritan,
  Lisu,
  Bamum,
  Javanese,
  Meetei_Mayek,
  Imperial_Aramaic,
  Old_South_Arabian,
  Inscriptional_Parthian,
  Inscriptional_Pahlavi,
  Old_Turkic,
  Kaithi,
  Batak,
  Brahmi,
  Mandaic,
  Chakma,
  Meroitic_Cursive,
  Meroitic_Hieroglyphs,
  Miao,
  Sharada,
  Sora_Sompeng,
  Takri,
  Caucasian_Albanian,
  Bassa_Vah,
  Duployan,
  Elbasan,
  Grantha,
  Pahawh_Hmong,
  Khojki,
  Linear_A,
  Mahajani,
  Manichaean,
  Mende_Kikakui,
  Modi,
  Mro,
  Old_North_Arabian,
  Nabataean,
  Palmyrene,
  Pau_Cin_Hau,
  Old_Permic,
  Psalter_Pahlavi,
  Siddham,
  Khudawadi,
  Tirhuta,
  Warang_Citi,
  Ahom,
  Anatolian_Hieroglyphs,
  Hatran,
  Multani,
  Old_Hungarian,
  SignWriting,
  Adlam,
  Bhaiksuki,
  Marchen,
  Newa,
  Osage,
  Tangut,
  Masaram_Gondi,
  Nushu,
  Soyombo,
  Zanabazar_Square,
  Dogra,
  Gunjala_Gondi,
  Makasar,
  Medefaidrin,
  Hanifi_Rohingya,
  Sogdian,
  Old_Sogdian,
  Elymaic,
  Nandinagari,
  Nyiakeng_Puachue_Hmong,
  Wancho,
  Chorasmian,
  Dives_Akuru,
  Khitan_Small_Script,
  Yezidi,
};

Script script(char32_t cp);
bool is_script(Script sc, char32_t cp);

//-----------------------------------------------------------------------------
// Normalization
//-----------------------------------------------------------------------------

std::u32string to_nfc(const char32_t *s32, size_t l);
std::u32string to_nfd(const char32_t *s32, size_t l);
std::u32string to_nfkc(const char32_t *s32, size_t l);
std::u32string to_nfkd(const char32_t *s32, size_t l);

//-----------------------------------------------------------------------------
// Inline Wrapper functions
//-----------------------------------------------------------------------------

inline std::u32string to_uppercase(const std::u32string_view s32,
                                   const char *lang = nullptr) {
  return to_uppercase(s32.data(), s32.length(), lang);
}

inline std::u32string to_uppercase(const char32_t *s32,
                                   const char *lang = nullptr) {
  return to_uppercase(s32, std::char_traits<char32_t>::length(s32), lang);
}

inline std::u32string to_lowercase(const std::u32string_view s32,
                                   const char *lang = nullptr) {
  return to_lowercase(s32.data(), s32.length(), lang);
}

inline std::u32string to_lowercase(const char32_t *s32,
                                   const char *lang = nullptr) {
  return to_lowercase(s32, std::char_traits<char32_t>::length(s32), lang);
}

inline std::u32string to_titlecase(const std::u32string_view s32,
                                   const char *lang = nullptr) {
  return to_titlecase(s32.data(), s32.length(), lang);
}

inline std::u32string to_titlecase(const char32_t *s32,
                                   const char *lang = nullptr) {
  return to_titlecase(s32, std::char_traits<char32_t>::length(s32), lang);
}

inline std::u32string to_case_fold(
    const std::u32string_view s32,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false) {
  return to_case_fold(s32.data(), s32.length(),
                      special_case_for_uppercase_I_and_dotted_uppercase_I);
}

inline std::u32string to_case_fold(
    const char32_t *s32,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false) {
  return to_case_fold(s32, std::char_traits<char32_t>::length(s32),
                      special_case_for_uppercase_I_and_dotted_uppercase_I);
}

inline bool is_uppercase(const std::u32string_view s32) {
  return is_uppercase(s32.data(), s32.length());
}

inline bool is_uppercase(const char32_t *s32) {
  return is_uppercase(s32, std::char_traits<char32_t>::length(s32));
}

inline bool is_lowercase(const std::u32string_view s32) {
  return is_lowercase(s32.data(), s32.length());
}

inline bool is_lowercase(const char32_t *s32) {
  return is_lowercase(s32, std::char_traits<char32_t>::length(s32));
}

inline bool is_titlecase(const std::u32string_view s32) {
  return is_titlecase(s32.data(), s32.length());
}

inline bool is_titlecase(const char32_t *s32) {
  return is_titlecase(s32, std::char_traits<char32_t>::length(s32));
}

inline bool is_case_fold(const std::u32string_view s32) {
  return is_case_fold(s32.data(), s32.length());
}

inline bool is_case_fold(const char32_t *s32) {
  return is_case_fold(s32, std::char_traits<char32_t>::length(s32));
}

inline bool caseless_match(
    const std::u32string_view s1, const std::u32string_view s2,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false) {
  return caseless_match(s1.data(), s1.length(), s2.data(), s2.length(),
                        special_case_for_uppercase_I_and_dotted_uppercase_I);
}

inline bool caseless_match(
    const char32_t *s1, const char32_t *s2,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false) {
  return caseless_match(s1, std::char_traits<char32_t>::length(s1), s2,
                        std::char_traits<char32_t>::length(s2),
                        special_case_for_uppercase_I_and_dotted_uppercase_I);
}

inline bool canonical_caseless_match(
    const std::u32string_view s1, const std::u32string_view s2,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false) {
  return canonical_caseless_match(
      s1.data(), s1.length(), s2.data(), s2.length(),
      special_case_for_uppercase_I_and_dotted_uppercase_I);
}

inline bool canonical_caseless_match(
    const char32_t *s1, const char32_t *s2,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false) {
  return canonical_caseless_match(
      s1, std::char_traits<char32_t>::length(s1), s2,
      std::char_traits<char32_t>::length(s2),
      special_case_for_uppercase_I_and_dotted_uppercase_I);
}

inline bool compatibility_caseless_match(
    const std::u32string_view s1, const std::u32string_view s2,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false) {
  return compatibility_caseless_match(
      s1.data(), s1.length(), s2.data(), s2.length(),
      special_case_for_uppercase_I_and_dotted_uppercase_I);
}

inline bool compatibility_caseless_match(
    const char32_t *s1, const char32_t *s2,
    bool special_case_for_uppercase_I_and_dotted_uppercase_I = false) {
  return compatibility_caseless_match(
      s1, std::char_traits<char32_t>::length(s1), s2,
      std::char_traits<char32_t>::length(s2),
      special_case_for_uppercase_I_and_dotted_uppercase_I);
}

inline std::u32string to_nfc(const std::u32string_view s32) {
  return to_nfc(s32.data(), s32.length());
}

inline std::u32string to_nfc(const char32_t *s32) {
  return to_nfc(s32, std::char_traits<char32_t>::length(s32));
}

inline std::u32string to_nfd(const std::u32string_view s32) {
  return to_nfd(s32.data(), s32.length());
}

inline std::u32string to_nfd(const char32_t *s32) {
  return to_nfd(s32, std::char_traits<char32_t>::length(s32));
}

inline std::u32string to_nfkc(const std::u32string_view s32) {
  return to_nfkc(s32.data(), s32.length());
}

inline std::u32string to_nfkc(const char32_t *s32) {
  return to_nfkc(s32, std::char_traits<char32_t>::length(s32));
}

inline std::u32string to_nfkd(const std::u32string_view s32) {
  return to_nfkd(s32.data(), s32.length());
}

inline std::u32string to_nfkd(const char32_t *s32) {
  return to_nfkd(s32, std::char_traits<char32_t>::length(s32));
}

inline size_t grapheme_count(const std::u32string_view s32) {
  return grapheme_count(s32.data(), s32.length());
}

inline size_t grapheme_count(const char32_t *s32) {
  return grapheme_count(s32, std::char_traits<char32_t>::length(s32));
}

inline size_t grapheme_length(const std::u32string_view s32) {
  return grapheme_length(s32.data(), s32.length());
}

inline size_t grapheme_length(const char32_t *s32) {
  return grapheme_length(s32, std::char_traits<char32_t>::length(s32));
}

// ----------------------------------------------------------------------------

/*
 * Implementation
 */

enum class CaseMappingType {
  Upper = 0,
  Lower,
  Title,
};

enum class SpecialCasingContext {
  Unassigned,
  Final_Sigma,
  Not_Final_Sigma,
  After_Soft_Dotted,
  More_Above,
  Before_Dot,
  Not_Before_Dot,
  After_I,
};

struct SpecialCasing {
  const char32_t *lower;
  const char32_t *title;
  const char32_t *upper;
  const char *language;
  SpecialCasingContext context;

  const char32_t *case_mapping_codes(CaseMappingType type) const {
    switch (type) {
      case CaseMappingType::Upper:
        return upper;
      case CaseMappingType::Lower:
        return lower;
      case CaseMappingType::Title:
        return title;
    }
    // NOTREACHED
    return nullptr;
  }
};

struct CaseFolding {
  const char32_t C;
  const char32_t S;
  const char32_t *F;
  const char32_t T;
};

struct NormalizationProperties {
  int combining_class;
  const char *compat_format;
  const char32_t *codes;
};

// This is generated from 'GraphemeBreakProperty.txt' in Unicode database.
// `python scripts/gen_property_values.py <
// UCD/auxiliary/GraphemeBreakProperty.txt`
enum class GraphemeBreak {
  Unassigned,
  Prepend,
  CR,
  LF,
  Control,
  Extend,
  Regional_Indicator,
  SpacingMark,
  L,
  V,
  T,
  LV,
  LVT,
  ZWJ,
};

// This is generated from 'WordBreakProperty.txt' in Unicode database.
// `python scripts/gen_property_values.py < UCD/auxiliary/WordBreakProperty.txt`
enum class WordBreak {
  Unassigned,
  Double_Quote,
  Single_Quote,
  Hebrew_Letter,
  CR,
  LF,
  Newline,
  Extend,
  Regional_Indicator,
  Format,
  Katakana,
  ALetter,
  MidLetter,
  MidNum,
  MidNumLet,
  Numeric,
  ExtendNumLet,
  ZWJ,
  WSegSpace,
};

// This is generated from 'SentenceBreakProperty.txt' in Unicode database.
// `python scripts/gen_property_values.py <
// UCD/auxiliary/SentenceBreakProperty.txt`
enum class SentenceBreak {
  Unassigned,
  CR,
  LF,
  Extend,
  Sep,
  Format,
  Sp,
  Lower,
  Upper,
  OLetter,
  Numeric,
  ATerm,
  STerm,
  Close,
  SContinue,
};

// This is generated from 'emoji-data.txt' in Unicode database.
// `python scripts/gen_property_values.py < UCD/emoji/emoji-data.txt`
enum class Emoji {
  Unassigned,
  Emoji,
  Emoji_Presentation,
  Emoji_Modifier,
  Emoji_Modifier_Base,
  Emoji_Component,
  Extended_Pictographic,
};

//-----------------------------------------------------------------------------
// Unicode Data
//-----------------------------------------------------------------------------

// This is generated from Unicode database.
// `python scripts/gen_tables.py UCD`
namespace _general_category_properties {
using T = GeneralCategory;
const auto D = GeneralCategory::Cn;
static const size_t _block_size = 128;
static const GeneralCategory _0[] = { T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Zs,T::Po,T::Po,T::Po,T::Sc,T::Po,T::Po,T::Po,T::Ps,T::Pe,T::Po,T::Sm,T::Po,T::Pd,T::Po,T::Po,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Po,T::Sm,T::Sm,T::Sm,T::Po,T::Po,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ps,T::Po,T::Pe,T::Sk,T::Pc,T::Sk,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ps,T::Sm,T::Pe,T::Sm,T::Cc, };
static const GeneralCategory _1[] = { T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Cc,T::Zs,T::Po,T::Sc,T::Sc,T::Sc,T::Sc,T::So,T::Po,T::Sk,T::So,T::Lo,T::Pi,T::Sm,T::Cf,T::So,T::Sk,T::So,T::Sm,T::No,T::No,T::Sk,T::Ll,T::Po,T::Po,T::Sk,T::No,T::Lo,T::Pf,T::No,T::No,T::No,T::Po,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Sm,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _2[] = { T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll, };
static const GeneralCategory _3[] = { T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Ll,T::Lo,T::Lu,T::Ll,T::Ll,T::Ll,T::Lo,T::Lo,T::Lo,T::Lo,T::Lu,T::Lt,T::Ll,T::Lu,T::Lt,T::Ll,T::Lu,T::Lt,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Lu,T::Lt,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll, };
static const GeneralCategory _4[] = { T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _5[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lo,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Sk,T::Sk,T::Sk,T::Sk,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Lm,T::Sk,T::Lm,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk, };
static const GeneralCategory _6[] = { T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lu,T::Ll,T::Lu,T::Ll,T::Lm,T::Sk,T::Lu,T::Ll,D,D,T::Lm,T::Ll,T::Ll,T::Ll,T::Po,T::Lu, };
static const GeneralCategory _7[] = { D,D,D,D,T::Sk,T::Sk,T::Lu,T::Po,T::Lu,T::Lu,T::Lu,D,T::Lu,D,T::Lu,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Ll,T::Sm,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu, };
static const GeneralCategory _8[] = { T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll, };
static const GeneralCategory _9[] = { T::Lu,T::Ll,T::So,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Me,T::Me,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll, };
static const GeneralCategory _10[] = { T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,D,T::Lm,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _11[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Po,T::Pd,D,D,T::So,T::So,T::Sc,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Pd,T::Mn,T::Po,T::Mn,T::Mn,T::Po,T::Mn,T::Mn,T::Po,T::Mn,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _12[] = { T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Sm,T::Sm,T::Sm,T::Po,T::Po,T::Sc,T::Po,T::Po,T::So,T::So,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Cf,D,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Po,T::Po,T::Po,T::Lo,T::Lo,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _13[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Cf,T::So,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lm,T::Lm,T::Mn,T::Mn,T::So,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Lo,T::Lo,T::So,T::So,T::Lo, };
static const GeneralCategory _14[] = { T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,T::Cf,T::Lo,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _15[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lm,T::Lm,T::So,T::Po,T::Po,T::Po,T::Lm,D,D,T::Mn,T::Sc,T::Sc, };
static const GeneralCategory _16[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Lm,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lm,T::Mn,T::Mn,T::Mn,T::Lm,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,D,D,T::Po,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _17[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Cf,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn, };
static const GeneralCategory _18[] = { T::Mn,T::Mn,T::Mn,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mc,T::Mn,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mc,T::Mc,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Po,T::Po,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Lm,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _19[] = { T::Lo,T::Mn,T::Mc,T::Mc,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,D,D,T::Mc,T::Mc,D,D,T::Mc,T::Mc,T::Mn,T::Lo,D,D,D,D,D,D,D,D,T::Mc,D,D,D,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Lo,T::Sc,T::Sc,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::Sc,T::Lo,T::Po,T::Mn,D, };
static const GeneralCategory _20[] = { D,T::Mn,T::Mn,T::Mc,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,D,D,T::Mn,D,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,D,D,D,D,T::Mn,T::Mn,D,D,T::Mn,T::Mn,T::Mn,D,D,D,T::Mn,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Mn,T::Po,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _21[] = { D,T::Mn,T::Mn,T::Mc,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mn,T::Mc,D,T::Mc,T::Mc,T::Mn,D,D,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Mn,T::Mn,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Sc,D,D,D,D,D,D,D,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn, };
static const GeneralCategory _22[] = { D,T::Mn,T::Mc,T::Mc,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Lo,T::Mc,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,D,D,T::Mc,T::Mc,D,D,T::Mc,T::Mc,T::Mn,D,D,D,D,D,D,D,T::Mn,T::Mn,T::Mc,D,D,D,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::So,T::Lo,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D, };
static const GeneralCategory _23[] = { D,D,T::Mn,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::Lo,T::Lo,D,T::Lo,D,T::Lo,T::Lo,D,D,D,T::Lo,T::Lo,D,D,D,T::Lo,T::Lo,T::Lo,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::Mc,T::Mc,T::Mn,T::Mc,T::Mc,D,D,D,T::Mc,T::Mc,T::Mc,D,T::Mc,T::Mc,T::Mc,T::Mn,D,D,T::Lo,D,D,D,D,D,D,T::Mc,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::Sc,T::So,D,D,D,D,D, };
static const GeneralCategory _24[] = { T::Mn,T::Mc,T::Mc,T::Mc,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::Lo,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,D,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D,D,D,T::Mn,T::Mn,D,T::Lo,T::Lo,T::Lo,D,D,D,D,D,T::Lo,T::Lo,T::Mn,T::Mn,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,D,T::Po,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So, };
static const GeneralCategory _25[] = { T::Lo,T::Mn,T::Mc,T::Mc,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Lo,T::Mc,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,D,T::Mn,T::Mc,T::Mc,D,T::Mc,T::Mc,T::Mn,T::Mn,D,D,D,D,D,D,D,T::Mc,T::Mc,D,D,D,D,D,D,D,T::Lo,D,T::Lo,T::Lo,T::Mn,T::Mn,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _26[] = { T::Mn,T::Mn,T::Mc,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mc,T::Mc,T::Mc,D,T::Mc,T::Mc,T::Mc,T::Mn,T::Lo,T::So,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Mc,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _27[] = { D,T::Mn,T::Mc,T::Mc,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::Mn,D,D,D,D,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,D,T::Mn,D,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,T::Mc,T::Mc,T::Po,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _28[] = { D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,T::Sc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _29[] = { D,T::Lo,T::Lo,D,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lm,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _30[] = { T::Lo,T::So,T::So,T::So,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::So,T::Po,T::So,T::So,T::So,T::Mn,T::Mn,T::So,T::So,T::So,T::So,T::So,T::So,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::Mn,T::So,T::Mn,T::So,T::Mn,T::Ps,T::Pe,T::Ps,T::Pe,T::Mc,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc, };
static const GeneralCategory _31[] = { T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Mn,T::So,T::So,T::So,T::So,T::So,T::So,D,T::So,T::So,T::Po,T::Po,T::Po,T::Po,T::Po,T::So,T::So,T::So,T::So,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _32[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Lo,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Lo,T::Mc,T::Mc,T::Mc,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _33[] = { T::Lo,T::Lo,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Lo,T::Mc,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Mc,T::Mc,T::Mc,T::Mn,T::So,T::So,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Lu,D,D,D,D,D,T::Lu,D,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Po,T::Lm,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _36[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _37[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _38[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D, };
static const GeneralCategory _39[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D, };
static const GeneralCategory _40[] = { T::Pd,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _44[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::So,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _45[] = { T::Zs,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Ps,T::Pe,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,T::Po,T::Po,T::Nl,T::Nl,T::Nl,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D, };
static const GeneralCategory _46[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Po,T::Po,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,D,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _47[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Lm,T::Po,T::Po,T::Po,T::Sc,T::Lo,T::Mn,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D, };
static const GeneralCategory _48[] = { T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Pd,T::Po,T::Po,T::Po,T::Po,T::Mn,T::Mn,T::Mn,T::Cf,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D, };
static const GeneralCategory _49[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Lo,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _50[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,D,D,D,D,T::Mc,T::Mc,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,D,D,D,D,T::So,D,D,D,T::Po,T::Po,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _51[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::No,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _52[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,D,D,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mc,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,T::Mn, };
static const GeneralCategory _53[] = { T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Lm,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Me,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _54[] = { T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mc,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D, };
static const GeneralCategory _55[] = { T::Mn,T::Mn,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mc,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,D,D,D,D,D,D,D,D,T::Po,T::Po,T::Po,T::Po, };
static const GeneralCategory _56[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,D,D,D,T::Po,T::Po,T::Po,T::Po,T::Po,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,T::Lo,T::Lo,T::Lo,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Po,T::Po, };
static const GeneralCategory _57[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D,D,D,D,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,D,T::Lu,T::Lu,T::Lu,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,T::Mn,T::Mn,T::Mn,T::Po,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Lo,T::Lo,T::Mc,T::Mn,T::Mn,T::Lo,D,D,D,D,D, };
static const GeneralCategory _58[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _59[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn, };
static const GeneralCategory _60[] = { T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll, };
static const GeneralCategory _61[] = { T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll, };
static const GeneralCategory _62[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,T::Lu,D,T::Lu,D,T::Lu,D,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D, };
static const GeneralCategory _63[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Lt,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lt,T::Sk,T::Ll,T::Sk,T::Sk,T::Sk,T::Ll,T::Ll,T::Ll,D,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lt,T::Sk,T::Sk,T::Sk,T::Ll,T::Ll,T::Ll,T::Ll,D,D,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Sk,T::Sk,T::Sk,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Sk,T::Sk,T::Sk,D,D,T::Ll,T::Ll,T::Ll,D,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lt,T::Sk,T::Sk,D, };
static const GeneralCategory _64[] = { T::Zs,T::Zs,T::Zs,T::Zs,T::Zs,T::Zs,T::Zs,T::Zs,T::Zs,T::Zs,T::Zs,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Pd,T::Pd,T::Pd,T::Pd,T::Pd,T::Pd,T::Po,T::Po,T::Pi,T::Pf,T::Ps,T::Pi,T::Pi,T::Pf,T::Ps,T::Pi,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Zl,T::Zp,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Zs,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Pi,T::Pf,T::Po,T::Po,T::Po,T::Po,T::Pc,T::Pc,T::Po,T::Po,T::Po,T::Sm,T::Ps,T::Pe,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Sm,T::Po,T::Pc,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Zs,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,D,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::No,T::Lm,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::Sm,T::Sm,T::Sm,T::Ps,T::Pe,T::Lm, };
static const GeneralCategory _65[] = { T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::Sm,T::Sm,T::Sm,T::Ps,T::Pe,D,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,D,D,D,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,T::Sc,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Me,T::Me,T::Me,T::Me,T::Mn,T::Me,T::Me,T::Me,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _66[] = { T::So,T::So,T::Lu,T::So,T::So,T::So,T::So,T::Lu,T::So,T::So,T::Ll,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Ll,T::So,T::Lu,T::So,T::So,T::Sm,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::So,T::So,T::So,T::So,T::So,T::So,T::Lu,T::So,T::Lu,T::So,T::Lu,T::So,T::Lu,T::Lu,T::Lu,T::Lu,T::So,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Lo,T::Lo,T::Lo,T::Lo,T::Ll,T::So,T::So,T::Ll,T::Ll,T::Lu,T::Lu,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::So,T::Sm,T::So,T::So,T::Ll,T::So,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl, };
static const GeneralCategory _67[] = { T::Nl,T::Nl,T::Nl,T::Lu,T::Ll,T::Nl,T::Nl,T::Nl,T::Nl,T::No,T::So,T::So,D,D,D,D,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::So,T::So,T::So,T::So,T::So,T::Sm,T::Sm,T::So,T::So,T::So,T::So,T::Sm,T::So,T::So,T::Sm,T::So,T::So,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::Sm,T::So,T::So,T::Sm,T::So,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm, };
static const GeneralCategory _70[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Ps,T::Pe,T::Ps,T::Pe,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Ps,T::Pe,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::So,T::So,T::So, };
static const GeneralCategory _71[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _72[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No, };
static const GeneralCategory _73[] = { T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No, };
static const GeneralCategory _75[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm, };
static const GeneralCategory _76[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _78[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No, };
static const GeneralCategory _79[] = { T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Ps,T::Pe,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm, };
static const GeneralCategory _83[] = { T::Sm,T::Sm,T::Sm,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Ps,T::Pe,T::Ps,T::Pe,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Ps,T::Pe,T::Sm,T::Sm, };
static const GeneralCategory _86[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::So,T::So,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::Sm,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _87[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _88[] = { T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lm,T::Lm,T::Lu,T::Lu, };
static const GeneralCategory _89[] = { T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::So,T::So,T::So,T::So,T::So,T::So,T::Lu,T::Ll,T::Lu,T::Ll,T::Mn,T::Mn,T::Mn,T::Lu,T::Ll,D,D,D,D,D,T::Po,T::Po,T::Po,T::Po,T::No,T::Po,T::Po, };
static const GeneralCategory _90[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,T::Ll,D,D,D,D,D,T::Ll,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,T::Lm,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Mn, };
static const GeneralCategory _91[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn, };
static const GeneralCategory _92[] = { T::Po,T::Po,T::Pi,T::Pf,T::Pi,T::Pf,T::Po,T::Po,T::Po,T::Pi,T::Pf,T::Po,T::Pi,T::Pf,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Pd,T::Po,T::Po,T::Pd,T::Po,T::Pi,T::Pf,T::Po,T::Po,T::Pi,T::Pf,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Po,T::Po,T::Po,T::Po,T::Po,T::Lm,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Pd,T::Pd,T::Po,T::Po,T::Po,T::Po,T::Pd,T::Po,T::Ps,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::So,T::So,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _93[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _95[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D, };
static const GeneralCategory _96[] = { T::Zs,T::Po,T::Po,T::Po,T::So,T::Lm,T::Lo,T::Nl,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::So,T::So,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Pd,T::Ps,T::Pe,T::Pe,T::So,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Pd,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::So,T::So,T::Nl,T::Nl,T::Nl,T::Lm,T::Lo,T::Po,T::So,T::So,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _97[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Mn,T::Sk,T::Sk,T::Lm,T::Lm,T::Lo,T::Pd,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,T::Lm,T::Lm,T::Lm,T::Lo, };
static const GeneralCategory _98[] = { D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _99[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::So,T::So,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _100[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _101[] = { T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _155[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _319[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D, };
static const GeneralCategory _320[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _329[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Po,T::Po, };
static const GeneralCategory _332[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Po,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lo,T::Mn,T::Me,T::Me,T::Me,T::Po,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Lm, };
static const GeneralCategory _333[] = { T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lm,T::Lm,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D, };
static const GeneralCategory _334[] = { T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Sk,T::Sk,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Ll, };
static const GeneralCategory _335[] = { T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lm,T::Sk,T::Sk,T::Lu,T::Ll,T::Lu,T::Ll,T::Lo,T::Lu,T::Ll,T::Lu,T::Ll,T::Ll,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,T::Lu,T::Ll,D,D,T::Lu,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Lu,T::Ll,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lu,T::Ll,T::Lo,T::Lm,T::Lm,T::Ll,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _336[] = { T::Lo,T::Lo,T::Mn,T::Lo,T::Lo,T::Lo,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mn,T::Mn,T::Mc,T::So,T::So,T::So,T::So,T::Mn,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::So,T::Sc,T::So,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D, };
static const GeneralCategory _337[] = { T::Mc,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,D,D,D,D,D,D,D,D,T::Po,T::Po,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,T::Po,T::Po,T::Lo,T::Po,T::Lo,T::Lo,T::Mn, };
static const GeneralCategory _338[] = { T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,D,D,D,D,D,D,D,D,D,D,D,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D, };
static const GeneralCategory _339[] = { T::Mn,T::Mn,T::Mn,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,T::Lm,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Lm,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D, };
static const GeneralCategory _340[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mc,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,T::Po,T::Po,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::So,T::So,T::So,T::Lo,T::Mc,T::Mn,T::Mc,T::Lo,T::Lo, };
static const GeneralCategory _341[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Lo,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Lo,T::Mn,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lm,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mn,T::Mn,T::Mc,T::Mc,T::Po,T::Po,T::Lo,T::Lm,T::Lm,T::Mc,T::Mn,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _342[] = { D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Sk,T::Lm,T::Lm,T::Lm,T::Lm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lm,T::Sk,T::Sk,D,D,D,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _343[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mn,T::Mc,T::Mc,T::Mn,T::Mc,T::Mc,T::Po,T::Mc,T::Mn,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D, };
static const GeneralCategory _431[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D, };
static const GeneralCategory _500[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _501[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _502[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D,D,D,D,D,D,D,D,D,D,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D,D,D,D,T::Lo,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Sm,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _503[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _506[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Pe,T::Ps,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _507[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Sc,T::So,D,D, };
static const GeneralCategory _508[] = { T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Ps,T::Pe,T::Po,D,D,D,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Pd,T::Pd,T::Pc,T::Pc,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Po,T::Po,T::Ps,T::Pe,T::Po,T::Po,T::Po,T::Po,T::Pc,T::Pc,T::Pc,T::Po,T::Po,T::Po,D,T::Po,T::Po,T::Po,T::Po,T::Pd,T::Ps,T::Pe,T::Ps,T::Pe,T::Ps,T::Pe,T::Po,T::Po,T::Po,T::Sm,T::Pd,T::Sm,T::Sm,T::Sm,D,T::Po,T::Sc,T::Po,T::Po,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _509[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Cf, };
static const GeneralCategory _510[] = { D,T::Po,T::Po,T::Po,T::Sc,T::Po,T::Po,T::Po,T::Ps,T::Pe,T::Po,T::Sm,T::Po,T::Pd,T::Po,T::Po,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Po,T::Sm,T::Sm,T::Sm,T::Po,T::Po,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ps,T::Po,T::Pe,T::Sk,T::Pc,T::Sk,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ps,T::Sm,T::Pe,T::Sm,T::Ps,T::Pe,T::Po,T::Ps,T::Pe,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _511[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lm,T::Lm,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,D,D,D,T::Sc,T::Sc,T::Sm,T::Sk,T::So,T::Sc,T::Sc,D,T::So,T::Sm,T::Sm,T::Sm,T::Sm,T::So,T::So,D,D,D,D,D,D,D,D,D,D,T::Cf,T::Cf,T::Cf,T::So,T::So,D,D, };
static const GeneralCategory _512[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _513[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D, };
static const GeneralCategory _514[] = { T::Po,T::Po,T::Po,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _515[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::No,T::No,T::So,T::So,T::So,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Mn,D,D, };
static const GeneralCategory _517[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Mn,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D, };
static const GeneralCategory _518[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Nl,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Nl,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D, };
static const GeneralCategory _519[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _520[] = { T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _521[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,D,D,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D,D,D, };
static const GeneralCategory _522[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _526[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _528[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,D,D,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Po,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::So,T::So,T::No,T::No,T::No,T::No,T::No,T::No,T::No, };
static const GeneralCategory _529[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No, };
static const GeneralCategory _530[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _531[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::No,T::No,T::Lo,T::Lo,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No, };
static const GeneralCategory _532[] = { T::Lo,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mn,D,D,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Mn,T::Mn,D,D,D,D,T::Mn,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::No,T::No,T::Po, };
static const GeneralCategory _533[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::So,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _534[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No, };
static const GeneralCategory _535[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _536[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _537[] = { T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No, };
static const GeneralCategory _538[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _540[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D, };
static const GeneralCategory _541[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Mn,T::Mn,T::Pd,D,D,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _542[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::Lo,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::No,T::No,T::No,T::No,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _543[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _544[] = { T::Mc,T::Mn,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Mn, };
static const GeneralCategory _545[] = { T::Mn,T::Mn,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Po,T::Po,T::Cf,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,T::Cf,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D, };
static const GeneralCategory _546[] = { T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Po,T::Po,T::Po,T::Lo,T::Mc,T::Mc,T::Lo,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Po,T::Po,T::Lo,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _547[] = { T::Mn,T::Mn,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,T::Po,T::Po,T::Po,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Mc,T::Mn,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Lo,T::Po,T::Lo,T::Po,T::Po,T::Po,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _548[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mc,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _549[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D, };
static const GeneralCategory _550[] = { T::Mn,T::Mn,T::Mc,T::Mc,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Mn,T::Mn,T::Lo,T::Mc,T::Mc,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,D,D,T::Mc,T::Mc,D,D,T::Mc,T::Mc,T::Mc,D,D,T::Lo,D,D,D,D,D,D,T::Mc,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _552[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Po,T::Po,T::Po,T::Po,T::Po,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Po,T::Po,D,T::Po,T::Mn,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _553[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Lo,T::Lo,T::Po,T::Lo,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _555[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,D,D,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _556[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mc,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Lo,D,D,D,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _557[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mc,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Lo,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _558[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::No,T::No,T::Po,T::Po,T::Po,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _560[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _561[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo, };
static const GeneralCategory _562[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,D,T::Mc,T::Mc,D,D,T::Mn,T::Mn,T::Mc,T::Mn,T::Lo,T::Mc,T::Lo,T::Mc,T::Mn,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _563[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,D,D,T::Mn,T::Mn,T::Mc,T::Mc,T::Mc,T::Mc,T::Mn,T::Lo,T::Po,T::Lo,T::Mc,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _564[] = { T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Po,T::Mn,D,D,D,D,D,D,D,D,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _565[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Lo,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D, };
static const GeneralCategory _568[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Lo,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,T::Po,T::Po,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _569[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mc,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,T::Mc,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _570[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,T::Mn,D,T::Mn,T::Mn,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lo,T::Mn,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _571[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,D,T::Mn,T::Mn,D,T::Mc,T::Mc,T::Mn,T::Mc,T::Mn,T::Lo,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _573[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mc,T::Mc,T::Po,T::Po,D,D,D,D,D,D,D, };
static const GeneralCategory _575[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sc,T::Sc,T::Sc,T::Sc,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Po, };
static const GeneralCategory _583[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _584[] = { T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,T::Nl,D,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _586[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _616[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _652[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _724[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _725[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _726[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Po,T::Po,T::Po,T::Po,T::Po,T::So,T::So,T::So,T::So,T::Lm,T::Lm,T::Lm,T::Lm,T::Po,T::So,D,D,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _727[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _732[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _733[] = { T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _734[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,T::Mn,T::Lo,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc, };
static const GeneralCategory _735[] = { T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,D,D,D,D,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lm,T::Lm,T::Po,T::Lm,T::Mn,D,D,D,D,D,D,D,D,D,D,D,T::Mc,T::Mc,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _783[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D, };
static const GeneralCategory _793[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _794[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _866[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _869[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D, };
static const GeneralCategory _888[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D, };
static const GeneralCategory _889[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::So,T::Mn,T::Mn,T::Po,T::Cf,T::Cf,T::Cf,T::Cf,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _929[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _930[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Mc,T::Mc,T::Mn,T::Mn,T::Mn,T::So,T::So,T::So,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Mc,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn, };
static const GeneralCategory _931[] = { T::Mn,T::Mn,T::Mn,T::So,T::So,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Mn,T::Mn,T::Mn,T::Mn,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _932[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Mn,T::Mn,T::Mn,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _933[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _934[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D, };
static const GeneralCategory _936[] = { T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu, };
static const GeneralCategory _937[] = { T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,D,T::Lu,T::Lu,D,D,T::Lu,D,D,T::Lu,T::Lu,D,D,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,D,T::Ll,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _938[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,D,T::Lu,T::Lu,T::Lu,T::Lu,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,D,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Lu,D,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,D,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu, };
static const GeneralCategory _939[] = { T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _940[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu, };
static const GeneralCategory _941[] = { T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,D,D,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _942[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll, };
static const GeneralCategory _943[] = { T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Sm,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Lu,T::Ll,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd, };
static const GeneralCategory _948[] = { T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::So,T::So,T::So,T::So,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Mn,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _949[] = { T::So,T::So,T::So,T::So,T::Mn,T::So,T::So,T::Po,T::Po,T::Po,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _960[] = { T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,T::Mn,T::Mn,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _962[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,T::Lm,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,T::Lo,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _965[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Mn,T::Mn,T::Mn,T::Mn,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,T::Sc, };
static const GeneralCategory _977[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _978[] = { T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Lu,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Ll,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Lm,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,T::Po,T::Po,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _984[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No, };
static const GeneralCategory _985[] = { T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::No,T::No,T::No,T::Sc,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _986[] = { D,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _988[] = { T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,D,D,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D,T::Lo,D,D,D,D,D,D,T::Lo,D,D,D,D,T::Lo,D,T::Lo,D,T::Lo,D,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,D,T::Lo,D,D,T::Lo,D,T::Lo,D,T::Lo,D,T::Lo,D,T::Lo,D,T::Lo,T::Lo,D,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,D, };
static const GeneralCategory _989[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Sm,T::Sm,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _992[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _993[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _994[] = { T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::No,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _995[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _996[] = { T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _999[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::Sk,T::Sk,T::Sk,T::Sk,T::Sk, };
static const GeneralCategory _1005[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D, };
static const GeneralCategory _1006[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _1007[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _1008[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _1009[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _1010[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _1011[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So, };
static const GeneralCategory _1012[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,T::So,T::So,T::So,T::So,T::So,D,D,D,T::So,T::So,T::So,D,D,D,D,D, };
static const GeneralCategory _1013[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _1015[] = { T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,T::So,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,T::Nd,D,D,D,D,D,D, };
static const GeneralCategory _1357[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _1390[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _1392[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _1437[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo, };
static const GeneralCategory _1495[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _1524[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _1574[] = { T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _7168[] = { D,T::Cf,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf,T::Cf, };
static const GeneralCategory _7171[] = { T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,T::Mn,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const GeneralCategory _8191[] = { T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,D,D, };
static const GeneralCategory _8703[] = { T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,D,D, };
static const GeneralCategory *_blocks[] = {
 _0,_1,_2,_3,_4,_5,_6,_7,
 _8,_9,_10,_11,_12,_13,_14,_15,
 _16,_17,_18,_19,_20,_21,_22,_23,
 _24,_25,_26,_27,_28,_29,_30,_31,
 _32,_33,0,0,_36,_37,_38,_39,
 _40,0,0,0,_44,_45,_46,_47,
 _48,_49,_50,_51,_52,_53,_54,_55,
 _56,_57,_58,_59,_60,_61,_62,_63,
 _64,_65,_66,_67,0,0,_70,_71,
 _72,_73,0,_75,_76,0,_78,_79,
 0,0,0,_83,0,0,_86,_87,
 _88,_89,_90,_91,_92,_93,0,_95,
 _96,_97,_98,_99,_100,_101,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,_155,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_319,
 _320,0,0,0,0,0,0,0,
 0,_329,0,0,_332,_333,_334,_335,
 _336,_337,_338,_339,_340,_341,_342,_343,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_431,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_500,_501,_502,_503,
 0,0,_506,_507,_508,_509,_510,_511,
 _512,_513,_514,_515,0,_517,_518,_519,
 _520,_521,_522,0,0,0,_526,0,
 _528,_529,_530,_531,_532,_533,_534,_535,
 _536,_537,_538,0,_540,_541,_542,_543,
 _544,_545,_546,_547,_548,_549,_550,0,
 _552,_553,0,_555,_556,_557,_558,0,
 _560,_561,_562,_563,_564,_565,0,0,
 _568,_569,_570,_571,0,_573,0,_575,
 0,0,0,0,0,0,0,_583,
 _584,0,_586,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 _616,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_652,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_724,_725,_726,_727,
 0,0,0,0,_732,_733,_734,_735,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_783,
 0,0,0,0,0,0,0,0,
 0,_793,_794,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,_866,0,0,_869,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 _888,_889,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,_929,_930,_931,_932,_933,_934,0,
 _936,_937,_938,_939,_940,_941,_942,_943,
 0,0,0,0,_948,_949,0,0,
 0,0,0,0,0,0,0,0,
 _960,0,_962,0,0,_965,0,0,
 0,0,0,0,0,0,0,0,
 0,_977,_978,0,0,0,0,0,
 _984,_985,_986,0,_988,_989,0,0,
 _992,_993,_994,_995,_996,0,0,_999,
 0,0,0,0,0,_1005,_1006,_1007,
 _1008,_1009,_1010,_1011,_1012,_1013,0,_1015,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,_1357,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,_1390,0,
 _1392,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,_1437,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_1495,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_1524,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,_1574,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 _7168,0,0,_7171,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_8191,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_8703,
};
static const GeneralCategory _block_values[] = {
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,T::Lo,T::Lo,D,D,D,D,
 D,T::Lo,T::Lo,T::Lo,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,T::Sm,T::Sm,D,D,
 D,D,T::So,D,D,T::So,D,D,
 T::So,T::So,T::Sm,D,T::Sm,T::Sm,D,D,
 D,D,D,D,D,D,T::So,D,
 D,D,D,D,D,D,T::So,T::So,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,
 D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,D,T::Lo,T::Lo,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,
 T::Cs,T::Cs,T::Cs,T::Cs,T::Cs,T::Cs,T::Cs,T::Cs,
 T::Cs,T::Cs,T::Cs,T::Cs,T::Cs,T::Cs,T::Cs,T::Cs,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Lo,T::Lo,D,D,D,D,
 T::Lo,T::Lo,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,T::Lo,T::Lo,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,
 D,T::Lo,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,D,T::Lo,T::Lo,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::So,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::So,T::So,T::So,T::So,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,T::So,D,
 T::So,T::So,T::So,T::So,T::So,D,D,D,
 D,D,D,D,D,D,T::So,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,
 D,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,T::Lo,T::Lo,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,
 T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,T::Lo,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,T::Mn,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 D,D,D,D,D,D,D,D,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,D,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,
 T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,T::Co,D,
};
inline GeneralCategory get_value(char32_t cp) {
  auto i = cp / _block_size;
  auto bl = _blocks[i];
  if (bl) {
    auto off = cp % _block_size;
    return bl[off];
  }
  return _block_values[i];
}
}
namespace _properties {
static const size_t _block_size = 128;
static const uint64_t _0[] = { 0,0,0,0,0,0,0,0,0,1073741825,1073741825,1073741825,1073741825,1073741825,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1073741825,2415919168,2147483680,2147483648,2147483648,2147483648,2147483648,2147483680,2147483648,2147483648,2147483648,2147483648,2147483712,2147483672,2415919168,2147483648,768,768,768,768,768,768,768,768,768,768,2147483712,2147483712,2147483648,2147483648,2147483648,2415919168,2147483648,768,768,768,768,768,768,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2147483648,2147483648,2147483648,2147487872,0,2147487744,768,768,768,768,768,768,0,0,16777216,16777216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2147483648,2147483648,2147483648,2147483648,0, };
static const uint64_t _1[] = { 0,0,0,0,0,1073741825,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,4096,2147483648,16384,2147483680,2147483648,16,2147483648,4096,2147483648,2147483648,0,0,4096,0,2147483648,134230016,4096,0,16384,2147483680,0,0,0,2147483648,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2147483648,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2147483648,0,0,0,0,0,0,0,0, };
static const uint64_t _2[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16777216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8388608,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _4[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16777216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16777216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _5[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16777216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20480,20480,16797696,20480,20480,20480,20480,20480,20480,4096,4096,4096,4096,4096,4096,4096,20480,20480,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,12288,12288,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,20480,20480,20480,20480,20480,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096, };
static const uint64_t _6[] = { 4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,21504,4096,4096,4096,4096,4096,4096,4096,4096,4096,4194304,4096,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,4096,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,0,0,0,0,20480,0,0,0,64,0, };
static const uint64_t _7[] = { 0,0,0,0,4096,4096,0,134217792,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,128,128,0,0,128,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,128,0,16777216,128,128,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _8[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16777216,0,16777216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _9[] = { 0,0,0,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _10[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _11[] = { 0,0,0,0,0,0,0,0,0,268435520,24,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,8,5120,0,5120,5120,64,5120,1024,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _12[] = { 4294967296,4294967296,4294967296,4294967296,4294967296,4294967296,0,0,0,0,0,0,64,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,64,2,0,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8192,0,0,0,0,0,0,0,0,0,0,5120,5120,5120,5120,5120,5120,5120,5120,1024,1024,1024,1024,5120,4096,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,0,0,8388608,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _13[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,0,1024,1024,1024,1024,1024,1024,1024,4294967296,0,4096,4096,1024,1024,1024,1024,4096,4096,1024,1024,0,4096,4096,4096,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _14[] = { 268435520,268435520,268435520,64,64,64,64,64,64,64,64,0,64,0,0,4294967296,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _15[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,0,64,268435520,8192,0,0,0,0,0, };
static const uint64_t _16[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,4096,4096,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,1024,1024,1024,0,1024,1024,1024,1024,0,0,0,64,64,64,64,64,64,64,268435520,64,268435520,64,64,64,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _17[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,4294967296,5120,5120,5120,5120,5120,5120,5120,4096,4096,4096,4096,4096,4096,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,5120,1024, };
static const uint64_t _18[] = { 1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,4096,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,1024,1024,0,4096,4096,4096,4096,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,1024,1024,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _19[] = { 0,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,132096,1024,1024,1024,1024,1024,1024,0,0,1024,1024,0,0,1024,1024,4096,0,0,0,0,0,0,0,0,0,132096,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _20[] = { 0,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,1024,1024,1024,1024,1024,0,0,0,0,1024,1024,0,0,1024,1024,4096,0,0,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,1024,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _21[] = { 0,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,1024,1024,1024,1024,1024,1024,1024,1024,0,1024,1024,1024,0,1024,1024,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,4096,4096,4096, };
static const uint64_t _22[] = { 0,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,132096,1024,1024,1024,1024,1024,1024,0,0,1024,1024,0,0,1024,1024,4096,0,0,0,0,0,0,0,12288,1024,132096,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _23[] = { 0,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,132096,1024,1024,1024,1024,0,0,0,1024,1024,1024,0,1024,1024,1024,4096,0,0,0,0,0,0,0,0,0,132096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _24[] = { 1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,0,1024,1024,1024,0,1024,1024,1024,4096,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _25[] = { 0,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,1024,1024,1024,1024,132096,1024,1024,0,1024,1024,1024,0,1024,1024,1024,4096,0,0,0,0,0,0,0,132096,132096,0,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _26[] = { 1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,0,132096,1024,1024,1024,1024,1024,1024,0,1024,1024,1024,0,1024,1024,1024,4096,0,0,0,0,0,0,0,0,0,132096,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _27[] = { 0,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,0,0,0,132096,1024,1024,1024,1024,1024,0,1024,0,1024,1024,1024,1024,1024,1024,1024,132096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _28[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,0,0,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,0,33554432,33554432,33554432,33554432,33554432,0,8192,4096,4096,4096,4096,4096,4096,1024,4096,0,0,0,0,0,0,0,0,0,0,0,64,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _29[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,0,0,1024,1024,1024,1024,1024,1024,4096,1024,1024,0,0,0,33554432,33554432,33554432,33554432,33554432,0,8192,0,4096,4096,4096,4096,4096,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _30[] = { 0,0,0,0,0,0,0,0,64,0,0,0,0,64,64,64,64,64,64,0,0,0,0,0,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,4096,0,4096,0,0,0,0,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,8389632,1024,8389632,1024,1024,1024,1024,1024,1024, };
static const uint64_t _31[] = { 1024,1024,4096,4096,4096,0,4096,4096,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _32[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,1024,4096,4096,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,268435520,268435520,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,0,0,0,0,1024,1024,1024,0,1024,5120,5120,0,0,1024,1024,5120,5120,5120,5120,5120,0,0,0,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _33[] = { 0,0,1024,1024,1024,1024,1024,5120,5120,5120,5120,5120,5120,5120,0,5120,0,0,0,0,0,0,0,0,0,0,5120,5120,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _34[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4194304,4194304,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _38[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,0,64,268435520,64,64,64,64,268435520,268435520,134217728,134217728,134217728,134217728,134217728,134217728,134217728,134217728,134217728,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _40[] = { 8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _44[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _45[] = { 1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,64,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _46[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,0,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _47[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8388608,8388608,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4194304,4194304,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,64,64,64,0,0,0,64,0,0,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _48[] = { 0,0,64,268435520,64,64,24,0,64,268435520,8192,536870912,536870912,536870912,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8192,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _49[] = { 0,0,0,0,0,67109888,67109888,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _50[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,4096,0,0,0,0,0,0,0,0,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _51[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,33554432,33554432,33554432,0,0,33554432,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,134217728,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _52[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,4096,4096,4096,4096,4096,4096,0,0,4096, };
static const uint64_t _53[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8192,268435520,268435520,268435520,268435520,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _54[] = { 1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,132096,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,268435520,0,64,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _55[] = { 1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _56[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,13312,4096,0,0,0,268435520,268435520,64,64,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,12288,4096,4096,268435520,268435520, };
static const uint64_t _57[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,4096,0,0,0,0,0,0,4096,0,0,4096,4096,4096,0,0,0,0,0,0, };
static const uint64_t _58[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,20480,16797696,20480,20480,20480,20480,20480,20480,20480,20480,0,0,0,0,0,0,0,0,0,0,0,0,0,16384,0,0,0,0,0,0,0, };
static const uint64_t _59[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16777216,0,0,0,0,16384,16384,16384,16384,16384,16384,16384,16384,16384,16793600,16384,16384,16384,16793600,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,4096,4096,4096,0,0,0,4096,4096,4096, };
static const uint64_t _60[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16777216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _61[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16777216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _63[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,0, };
static const uint64_t _64[] = { 1,1,1,1,1,1,1,1,1,1,1,0,131076,4,1073741826,1073741826,2147483672,2147483672,2147483656,2147483656,2147483656,2147483656,2147483776,2147483648,2147483680,2147483680,2147483680,2147483680,2147483680,2147483680,2147483680,2147483680,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,1073741825,1073741825,2,2,2,2,2,1,2147483648,2147483648,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483680,2147483680,2147483648,2415919168,2415919168,2147483648,0,128,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2415919168,2415919168,2415919168,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483656,0,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,1,0,128,128,128,128,4194304,2,2,2,2,8388608,8388608,8388608,8388608,8388608,8388608,0,16793600,0,0,0,0,0,0,0,0,0,8,0,128,128,16384, };
static const uint64_t _65[] = { 0,0,0,0,0,0,0,0,0,0,0,8,0,128,128,0,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,128,128,128,128,128,128,128,128,128,128,128,128,0,0,0,0,128,0,0,0,128,128,0,0,0,0,128,128,128,128,128,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _66[] = { 0,0,128,0,0,0,0,128,0,0,128,128,128,128,128,128,128,128,128,128,0,128,0,0,67108864,128,128,128,128,128,0,0,0,0,0,0,128,0,0,0,128,128,0,0,128,128,67108864,128,128,128,0,128,128,128,128,128,128,0,0,0,128,128,128,128,0,0,0,0,0,128,128,128,16777344,16777344,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384,16384, };
static const uint64_t _67[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483648,2147483776,2147483776,2147483648,2147483776,2147483776,2147483648,2147483776,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483776,2147483776,2147483648,2147483776,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648, };
static const uint64_t _68[] = { 2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483656,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648, };
static const uint64_t _70[] = { 2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2155872256,2155872256,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648, };
static const uint64_t _71[] = { 2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483648,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648, };
static const uint64_t _72[] = { 2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _73[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,17408,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _75[] = { 2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483648,2147483776,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648, };
static const uint64_t _76[] = { 2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483648,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648, };
static const uint64_t _78[] = { 2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _79[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648, };
static const uint64_t _83[] = { 2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483776,2147483776,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483776,2147483776,2147483648,2147483648, };
static const uint64_t _88[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16793600,16384,0,0, };
static const uint64_t _89[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _91[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024, };
static const uint64_t _92[] = { 2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483672,2147483648,2147483648,2147483656,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2415919168,2147487744,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483656,2147483656,2415919168,2147483648,2147483648,2147483648,2147483656,2147483712,2147483680,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483712,2147483648,2147483712,2147483712,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648, };
static const uint64_t _93[] = { 1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,0,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _95[] = { 1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,1048576,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,262144,262144,524288,524288,262144,262144,262144,262144,262144,262144,262144,262144,0,0,0,0, };
static const uint64_t _96[] = { 1,2147483712,2415919168,2147483648,0,8192,2048,2048,2147483648,2147483648,2147483648,2147483648,2147483680,2147483680,2147483680,2147483680,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483648,2147483656,2147483680,2147483680,2147483680,2147483648,2048,2048,2048,2048,2048,2048,2048,2048,2048,4096,4096,4096,4096,135168,135168,2147483656,8192,8192,8192,8192,8192,0,0,2048,2048,2048,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _97[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,67112960,67112960,8192,8192,0,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,12288,8192,8192,0, };
static const uint64_t _98[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4194304,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _155[] = { 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _319[] = { 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,0,0, };
static const uint64_t _320[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8192,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _329[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,268435520, };
static const uint64_t _332[] = { 0,0,0,0,0,0,0,0,0,0,0,0,8192,64,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,0,4096, };
static const uint64_t _333[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20480,20480,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,0,268435520,64,64,64,268435520,0,0,0,0,0,0,0,0, };
static const uint64_t _334[] = { 4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16384,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _335[] = { 0,0,0,0,0,0,0,0,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20480,20480,0,0,0,0,0,0, };
static const uint64_t _336[] = { 0,0,1024,0,0,0,0,0,0,0,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,268435520,0,0,0,0,0,0,0,0, };
static const uint64_t _337[] = { 1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,1024,0,0,0,0,0,0,0,0,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,1024, };
static const uint64_t _338[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,4096,4096,4096,4096,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _339[] = { 1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,0,0,0,0,0,0,64,268435520,268435520,0,0,0,0,0,8192,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5120,8192,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _340[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,1024,0,0,0,0,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8192,0,0,0,0,0,0,0,0,0,0,5120,5120,5120,0,0, };
static const uint64_t _341[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,0,1024,1024,1024,33554432,33554432,1024,1024,33554432,0,33554432,33554432,0,1024,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8192,0,64,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,268435520,268435520,0,8192,8192,1024,4096,0,0,0,0,0,0,0,0,0, };
static const uint64_t _342[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,20480,20480,20480,20480,0,0,0,0,0,0,0,0,0,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _343[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,268435520,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _500[] = { 2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2099200,2099200,2048,2099200,2048,2099200,2099200,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2099200,2048,2099200,2048,2099200,2099200,2048,2048,2099200,2099200,2099200,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,0,0,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048, };
static const uint64_t _501[] = { 2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _502[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5120,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _506[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2147483648,2147483648,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _507[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,65536,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _508[] = { 536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,32,32,32,32,2147483648,2147483648,0,0,0,0,0,0,0,0,0,64,64,268435520,0,64,64,268435520,268435520,8,0,0,0,0,0,0,0,0,128,0,152,0,0,0,0,128,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _510[] = { 0,268435520,32,0,0,0,0,32,0,0,0,0,64,24,268435520,0,256,256,256,256,256,256,256,256,256,256,64,64,0,0,0,268435520,0,256,256,256,256,256,256,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,0,4224,0,4096,256,256,256,256,256,256,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,32,32,64,16,0,0,0,0,0,0,0,0,0,0,12288,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _511[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,135168,135168,4194304,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,0,0,0,0,0,0,0,0,0,0,0,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,0,0,0,0,0,65536,65536, };
static const uint64_t _517[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _518[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,0,0,0,0,0, };
static const uint64_t _519[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _528[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _530[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _532[] = { 0,1024,1024,1024,0,1024,1024,0,0,0,0,0,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _533[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,0,0,0,0,0,0,0,0,0,64,64,64,64,64,64,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _534[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,64,64,64,64,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _535[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,64,64,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _538[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,5120,5120,5120,5120,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _541[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _542[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,268435520,268435520,268435520,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _544[] = { 1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,268435520,268435520,64,64,64,64,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _545[] = { 0,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,0,0,4294967296,268435520,268435520,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,4294967296,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _546[] = { 1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,268435520,268435520,268435520,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _547[] = { 1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,0,0,0,0,268435520,268435520,0,0,0,4096,4096,4096,268435520,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _548[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,1024,268435520,268435520,64,268435520,268435520,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _549[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _550[] = { 1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,0,132096,1024,1024,1024,1024,1024,1024,0,0,1024,1024,0,0,1024,1024,4096,0,0,0,0,0,0,0,0,0,132096,0,0,0,0,0,8192,0,0,0,0,1024,1024,0,0,4096,4096,4096,4096,4096,4096,4096,0,0,0,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _552[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,1024,1024,1024,4096,0,0,0,0,268435520,268435520,64,0,0,0,0,0,0,0,0,0,0,0,0,64,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _553[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,132096,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,132096,1024,1024,1024,1024,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _555[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,132096,1024,1024,1024,1024,1024,1024,0,0,1024,1024,1024,1024,1024,1024,1024,4096,4096,0,268435520,268435520,64,64,8192,8192,8192,268435520,268435520,268435520,268435520,268435520,268435520,268435520,268435520,268435520,268435520,268435520,268435520,268435520,268435520,268435520,0,0,0,0,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _556[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,1024,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _557[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _558[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _560[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _562[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,132096,1024,1024,1024,1024,1024,0,1024,1024,0,0,1024,1024,4096,4096,0,1024,0,1024,4096,268435520,0,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _563[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,0,0,1024,1024,1024,1024,1024,1024,4096,0,0,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _564[] = { 0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,1024,1024,1024,1024,1024,0,1024,1024,1024,1024,0,0,0,268435520,268435520,0,0,0,4096,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _565[] = { 0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,8192,4096,0,268435520,268435520,0,0,0,0,64,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _568[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,0,1024,1024,1024,1024,1024,1024,1024,4096,0,268435520,268435520,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _569[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _570[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,1024,0,0,0,1024,0,1024,1024,0,1024,1024,1024,4096,1024,4096,4096,0,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _571[] = { 0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,1024,0,1024,1024,0,1024,1024,1024,1024,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _573[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,1024,1024,1024,268435520,268435520,0,0,0,0,0,0,0, };
static const uint64_t _584[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,64,64,64,64,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _724[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,268435520,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _725[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,268435520,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _726[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,268435520,268435520,64,0,0,0,0,0,0,0,0,8192,8192,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _733[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _734[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024, };
static const uint64_t _735[] = { 1024,1024,1024,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,5120,5120,5120,5120,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8192,8192,0,8192,2048,0,0,0,0,0,0,0,0,0,0,0,5120,5120,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _783[] = { 2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,0,0,0,0,0,0,0,0, };
static const uint64_t _793[] = { 2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _794[] = { 2048,2048,2048,2048,2048,2048,2048,2048,2048,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _866[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048, };
static const uint64_t _869[] = { 2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,0,0,0,0, };
static const uint64_t _889[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1024,268435520,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _930[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,0,4096,4096,4096,0,0,0,4096,135168,135168,135168,135168,135168,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096, };
static const uint64_t _931[] = { 4096,4096,4096,0,0,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _936[] = { 128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128, };
static const uint64_t _937[] = { 128,128,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,0,0,128,0,0,128,128,0,0,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,0,128,0,128,16777344,16777344,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128, };
static const uint64_t _938[] = { 128,128,128,128,128,128,0,128,128,128,128,0,0,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,0,128,128,128,128,128,0,128,0,0,0,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128, };
static const uint64_t _939[] = { 128,128,128,128,128,128,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128, };
static const uint64_t _940[] = { 128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128, };
static const uint64_t _941[] = { 128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,16777344,16777344,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128, };
static const uint64_t _942[] = { 128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128, };
static const uint64_t _943[] = { 128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,0,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128, };
static const uint64_t _949[] = { 0,0,0,0,0,0,0,64,268435520,64,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _960[] = { 1024,1024,1024,1024,1024,1024,1024,0,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,0,0,1024,1024,1024,1024,1024,1024,1024,0,1024,1024,0,1024,1024,1024,1024,1024,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _962[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,8192,8192,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _965[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _977[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4096,4096,4096,4096,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _978[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12288,12288,12288,1024,4096,4096,4096,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _988[] = { 128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,128,128,0,128,0,0,128,0,128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,0,128,0,128,0,0,0,0,0,0,128,0,0,0,0,128,0,128,0,128,0,128,128,128,0,128,128,0,128,0,0,128,0,128,0,128,0,128,0,128,0,128,128,0,128,0,0,128,128,128,128,0,128,128,128,128,128,128,128,0,128,128,128,128,0,128,128,128,128,0,128,0, };
static const uint64_t _989[] = { 128,128,128,128,128,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,0,0,0,0,128,128,128,0,128,128,128,128,128,0,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _994[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,0,0,0,0,0,0,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,0,0,0,0,0,0,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,33792, };
static const uint64_t _995[] = { 33792,33792,33792,33792,33792,33792,33792,33792,33792,33792,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592,8589934592, };
static const uint64_t _1023[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _1357[] = { 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _1390[] = { 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,0,0,0,0,0,0,0,0,0,0,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200, };
static const uint64_t _1392[] = { 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,0,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200, };
static const uint64_t _1437[] = { 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200, };
static const uint64_t _1495[] = { 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _1524[] = { 2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _1535[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _1574[] = { 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint64_t _2047[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _2559[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _3071[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _3583[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _4095[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _4607[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _5119[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _5631[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _6143[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _6655[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _7167[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _7168[] = { 4194304,8388608,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint64_t _7171[] = { 536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,536870912,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304, };
static const uint64_t _7679[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _8191[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t _8703[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65536,65536, };
static const uint64_t *_blocks[] = {
 _0,_1,_2,0,_4,_5,_6,_7,
 _8,_9,_10,_11,_12,_13,_14,_15,
 _16,_17,_18,_19,_20,_21,_22,_23,
 _24,_25,_26,_27,_28,_29,_30,_31,
 _32,_33,_34,0,0,0,_38,0,
 _40,0,0,0,_44,_45,_46,_47,
 _48,_49,_50,_51,_52,_53,_54,_55,
 _56,_57,_58,_59,_60,_61,0,_63,
 _64,_65,_66,_67,_68,0,_70,_71,
 _72,_73,0,_75,_76,0,_78,_79,
 0,0,0,_83,0,0,0,0,
 _88,_89,0,_91,_92,_93,0,_95,
 _96,_97,_98,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,_155,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_319,
 _320,0,0,0,0,0,0,0,
 0,_329,0,0,_332,_333,_334,_335,
 _336,_337,_338,_339,_340,_341,_342,_343,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_500,_501,_502,0,
 0,0,_506,_507,_508,0,_510,_511,
 0,0,0,0,0,_517,_518,_519,
 0,0,0,0,0,0,0,0,
 _528,0,_530,0,_532,_533,_534,_535,
 0,0,_538,0,0,_541,_542,0,
 _544,_545,_546,_547,_548,_549,_550,0,
 _552,_553,0,_555,_556,_557,_558,0,
 _560,0,_562,_563,_564,_565,0,0,
 _568,_569,_570,_571,0,_573,0,0,
 0,0,0,0,0,0,0,0,
 _584,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_724,_725,_726,0,
 0,0,0,0,0,_733,_734,_735,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_783,
 0,0,0,0,0,0,0,0,
 0,_793,_794,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,_866,0,0,_869,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,_889,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,_930,_931,0,0,0,0,
 _936,_937,_938,_939,_940,_941,_942,_943,
 0,0,0,0,0,_949,0,0,
 0,0,0,0,0,0,0,0,
 _960,0,_962,0,0,_965,0,0,
 0,0,0,0,0,0,0,0,
 0,_977,_978,0,0,0,0,0,
 0,0,0,0,_988,_989,0,0,
 0,0,_994,_995,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_1023,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,_1357,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,_1390,0,
 _1392,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,_1437,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_1495,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_1524,0,0,0,
 0,0,0,0,0,0,0,_1535,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,_1574,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_2047,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_2559,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_3071,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_3583,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_4095,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_4607,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_5119,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_5631,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_6143,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_6655,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_7167,
 _7168,0,0,_7171,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_7679,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_8191,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_8703,
};
static const uint64_t _block_values[] = {
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,2147483648,0,0,
 0,0,2147483648,0,0,2147483648,0,0,
 2147483648,2147483648,2147483648,0,2147483648,2147483648,2147483648,2147483648,
 0,0,0,0,0,0,1048576,0,
 0,0,0,0,0,0,0,0,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,0,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,2048,2048,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 2048,2048,2048,2048,2048,2048,2048,2048,
 2048,2048,2048,2048,2048,2048,2048,2048,
 2048,2048,2048,2048,2048,2048,2048,2048,
 2048,2048,2048,2048,2048,2048,2048,2048,
 2048,2048,2048,2048,2048,2048,2048,2048,
 2048,2048,2048,2048,2048,2048,2048,0,
 2048,2048,2048,2048,2048,2048,2048,2048,
 2048,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,2048,2048,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,0,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,0,2099200,
 0,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,0,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 2048,2048,2048,2048,0,0,0,0,
 0,0,0,0,0,0,0,0,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,2099200,2099200,
 2099200,2099200,2099200,2099200,2099200,2099200,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,4194304,536870912,0,4194304,4194304,4194304,4194304,
 4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,
 4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,
 4194304,4194304,4194304,4194304,4194304,4194304,4194304,4194304,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
};
inline uint64_t get_value(char32_t cp) {
  auto i = cp / _block_size;
  auto bl = _blocks[i];
  if (bl) {
    auto off = cp % _block_size;
    return bl[off];
  }
  return _block_values[i];
}
}
namespace _derived_core_properties {
static const size_t _block_size = 128;
static const uint32_t _0[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131104,131072,131072,131072,131073,131072,131072,131104,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131104,131072,131073,131073,131073,131072,131072,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,131072,131072,131072,131105,151552,131104,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,131072,131073,131072,131073,0, };
static const uint32_t _1[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131104,131072,161814,131072,131073,32800,131072,131104,131072,131073,131072,131072,131104,163734,131072,151584,131104,131072,161814,131072,131072,131072,131072,131072,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,131073,163418,163418,163418,163418,163418,163418,163418,163734,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,131073,163222,163222,163222,163222,163222,163222,163222,163222, };
static const uint32_t _2[] = { 163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,161814,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163734,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163418,163222,163418,163222,163418,163222,163734, };
static const uint32_t _3[] = { 163222,163418,163418,163222,163418,163222,163418,163418,163222,163418,163418,163418,163222,161814,163418,163418,163418,163418,163222,163418,163418,163222,163418,163418,163418,163222,163222,161814,163418,163418,163222,163418,163418,163222,163418,163222,163418,163222,163418,163418,163222,163418,161814,161814,163418,163222,163418,163418,163222,163418,163418,163418,163222,163418,163222,163418,163418,163222,161814,161794,163418,163222,161814,163222,161794,161794,161794,161794,163674,163538,163222,163674,163538,163222,163674,163538,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163222,163674,163538,163222,163418,163222,163418,163418,163418,163222,163418,163222,163418,163222,163418,163222, };
static const uint32_t _4[] = { 163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,161814,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,161814,161814,161814,161814,161814,161814,163418,163418,163222,163418,163418,163222,163222,163418,163222,163418,163418,163418,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163222,163222,163222,163222,163222,161814,163222,163222,161814,163222,161814,163222,163222,161814,161814,161814,163222,163222,161814,163222,161814,163222,163222,161814,163222,163222,163222,163222,163222,161814,161814,163222,161814,163222,163222,161814,161814,163222,161814,161814,161814,161814,161814,161814,161814,163222,161814,161814, };
static const uint32_t _5[] = { 163222,161814,163222,163222,161814,161814,161814,163222,163222,163222,163222,163222,163222,161814,161814,161814,161814,161814,163222,161814,161794,161814,161814,161814,161814,161814,161814,161814,161814,163222,163222,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161846,161846,161846,161846,161846,161846,161846,161846,161846,161826,161826,161826,161826,161826,161826,161826,161846,161846,131104,131104,131104,131104,161826,161826,161826,161826,161826,161826,161826,161826,161826,161826,161826,161826,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,161846,161846,161846,161846,161846,131104,131104,131104,131104,131104,131104,131104,161826,131104,161826,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104, };
static const uint32_t _6[] = { 86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,87990,86048,86048,86048,86048,86048,86048,86048,86048,86048,118816,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,163418,163222,163418,163222,161826,131104,163418,163222,0,0,137270,163222,163222,163222,131072,163418, };
static const uint32_t _7[] = { 0,0,0,0,131104,131104,163418,151584,163418,163418,163418,0,163418,0,163418,163418,163222,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,0,163418,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163734,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163418,163735,163735,161819,161818,161818,163735,163734,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163735,163735,163222,163222,163419,163735,131073,163418,163222,163418,163418,163222,161814,163418,163418,163418, };
static const uint32_t _8[] = { 163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222, };
static const uint32_t _9[] = { 163418,163222,131072,86048,86048,86048,86048,86048,65568,65568,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222, };
static const uint32_t _10[] = { 163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,0,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,0,0,161826,131072,131072,131072,131072,131072,131104,161814,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222, };
static const uint32_t _11[] = { 163222,163222,163222,163222,163222,163222,163222,163734,161814,131072,131072,0,0,131072,131072,131072,0,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,131072,86050,131072,86050,86050,131072,86050,86050,131072,86050,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,161794,161794,161794,161794,131072,131104,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _12[] = { 32,32,32,32,32,32,131073,131073,131073,131072,131072,131072,131072,131072,131072,131072,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,131072,32800,0,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161826,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86048,86050,86050,86050,86050,86050,86050,86050,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,161794,161794,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _13[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,161794,86050,86050,86050,86050,86050,86050,86050,32,131072,86048,86048,86050,86050,86050,86050,161826,161826,86050,86050,131072,86048,86048,86048,86050,161794,161794,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,161794,161794,131072,131072,161794, };
static const uint32_t _14[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,32,161794,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _15[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,86048,86048,86048,86048,86048,86048,86048,86048,161826,161826,131072,131072,131072,131072,161826,0,0,86048,131072,131072, };
static const uint32_t _16[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86048,86048,161826,86050,86050,86050,86050,86050,86050,86050,86050,86050,161826,86050,86050,86050,161826,86050,86050,86050,86050,86048,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,86048,86048,0,0,131072,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _17[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,86048,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86048,86048,32,86050,86050,86050,86050,86050,86050,86050,86048,86048,86048,86048,86048,86048,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050, };
static const uint32_t _18[] = { 86050,86050,86050,151554,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,151554,86048,161794,151554,151554,151554,86050,86050,86050,86050,86050,86050,86050,86050,151554,151554,151554,151554,348192,151554,151554,161794,86048,86048,86048,86048,86050,86050,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,131072,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,161826,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _19[] = { 161794,86050,151554,151554,0,161794,161794,161794,161794,161794,161794,161794,161794,0,0,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,0,0,0,161794,161794,161794,161794,0,0,86048,161794,86018,151554,151554,86050,86050,86050,86050,0,0,151554,151554,0,0,151554,151554,348192,161794,0,0,0,0,0,0,0,0,86018,0,0,0,0,161794,161794,0,161794,161794,161794,86050,86050,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,161794,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,161794,131072,86048,0, };
static const uint32_t _20[] = { 0,86050,86050,151554,0,161794,161794,161794,161794,161794,161794,0,0,0,0,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,0,161794,161794,0,161794,161794,0,0,86048,0,151554,151554,151554,86050,86050,0,0,0,0,86050,86050,0,0,86050,86050,348192,0,0,0,86050,0,0,0,0,0,0,0,161794,161794,161794,161794,0,161794,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,86050,86050,161794,161794,161794,86050,131072,0,0,0,0,0,0,0,0,0, };
static const uint32_t _21[] = { 0,86050,86050,151554,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,0,161794,161794,161794,161794,161794,0,0,86048,161794,151554,151554,151554,86050,86050,86050,86050,86050,0,86050,86050,151554,0,151554,151554,348192,0,0,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,86050,86050,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,0,0,0,0,0,0,0,161794,86050,86050,86050,86048,86048,86048, };
static const uint32_t _22[] = { 0,86050,151554,151554,0,161794,161794,161794,161794,161794,161794,161794,161794,0,0,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,0,161794,161794,161794,161794,161794,0,0,86048,161794,86018,86050,151554,86050,86050,86050,86050,0,0,151554,151554,0,0,151554,151554,348192,0,0,0,0,0,0,0,86048,86050,86018,0,0,0,0,161794,161794,0,161794,161794,161794,86050,86050,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,161794,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0, };
static const uint32_t _23[] = { 0,0,86050,161794,0,161794,161794,161794,161794,161794,161794,0,0,0,161794,161794,161794,0,161794,161794,161794,161794,0,0,0,161794,161794,0,161794,0,161794,161794,0,0,0,161794,161794,0,0,0,161794,161794,161794,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,86018,151554,86050,151554,151554,0,0,0,151554,151554,151554,0,151554,151554,151554,348192,0,0,161794,0,0,0,0,0,0,86018,0,0,0,0,0,0,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0, };
static const uint32_t _24[] = { 86050,151554,151554,151554,86048,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,161794,86050,86050,86050,151554,151554,151554,151554,0,86050,86050,86050,0,86050,86050,86050,348192,0,0,0,0,0,0,0,86050,86050,0,161794,161794,161794,0,0,0,0,0,161794,161794,86050,86050,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _25[] = { 161794,86050,151554,151554,131072,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,0,0,86048,161794,151554,86050,151554,151554,86018,151554,151554,0,86050,151554,151554,0,151554,151554,86050,348192,0,0,0,0,0,0,0,86018,86018,0,0,0,0,0,0,0,161794,0,161794,161794,86050,86050,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _26[] = { 86050,86050,151554,151554,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,348192,348192,161794,86018,151554,151554,86050,86050,86050,86050,0,151554,151554,151554,0,151554,151554,151554,348192,161794,131072,0,0,0,0,161794,161794,161794,86018,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,86050,86050,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794, };
static const uint32_t _27[] = { 0,86050,151554,151554,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,0,0,161794,161794,161794,161794,161794,161794,161794,0,0,0,348192,0,0,0,0,86018,151554,151554,86050,86050,86050,0,86050,0,151554,151554,151554,151554,151554,151554,151554,86018,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,151554,151554,131072,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _28[] = { 0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,161794,153602,86050,86050,86050,86050,86050,86050,348194,0,0,0,0,131072,161794,161794,161794,161794,161794,161794,161826,86048,86048,86048,86048,86048,86048,86050,86048,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _29[] = { 0,161794,161794,0,161794,0,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,161794,153602,86050,86050,86050,86050,86050,86050,348192,86050,86050,161794,0,0,161794,161794,161794,161794,161794,0,161826,0,86048,86048,86048,86048,86048,86050,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _30[] = { 161794,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,86048,86048,131072,131072,131072,131072,131072,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,86048,131072,86048,131072,86048,131072,131072,131072,131072,151552,151552,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,151554, };
static const uint32_t _31[] = { 86050,86050,86048,86048,348192,131072,86048,86048,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,0,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,0,131072,131072,131072,131072,131072,131072,131072,131072,86048,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _32[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,86050,86050,86050,86050,151554,86050,86050,86050,86050,86050,86048,151554,348192,348192,151554,151554,86050,86050,161794,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794,151554,151554,86050,86050,161794,161794,161794,161794,86050,86050,86050,161794,151554,151554,151554,161794,161794,151554,151554,151554,151554,151554,151554,151554,161794,161794,161794,86050,86050,86050,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _33[] = { 161794,161794,86050,151554,151554,86050,86050,151554,151554,151554,151554,151554,151554,86050,161794,151554,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,151554,151554,151554,86050,131072,131072,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,0,163418,0,0,0,0,0,163418,0,0,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,162966,131072,161826,162966,162966,162966, };
static const uint32_t _34[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,194562,194562,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _36[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,0,161794,0,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _37[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,0,161794,0,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _38[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,86048,86048,86048,131072,131072,131072,131072,131072,131072,131072,131072,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0, };
static const uint32_t _39[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,162906,0,0,163734,163734,163734,163734,163734,163734,0,0, };
static const uint32_t _40[] = { 131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _44[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _45[] = { 131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0, };
static const uint32_t _46[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,86050,86050,348192,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,348192,131072,131072,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,0,86050,86050,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _47[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,118816,118816,151554,86050,86050,86050,86050,86050,86050,86050,151554,151554,151554,151554,151554,151554,151554,151554,86050,151554,151554,86048,86048,86048,86048,86048,86048,86048,86048,86048,348192,86048,131072,131072,131072,161826,131072,131072,131072,131072,161794,86048,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0, };
static const uint32_t _48[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,118816,118816,118816,32800,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161826,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0, };
static const uint32_t _49[] = { 161794,161794,161794,161794,161794,96290,96290,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,161794,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _50[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,86050,86050,86050,151554,151554,151554,151554,86050,86050,151554,151554,151554,0,0,0,0,151554,151554,86050,151554,151554,151554,151554,151554,151554,86048,86048,86048,0,0,0,0,131072,0,0,0,131072,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _51[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _52[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,151554,151554,86050,0,0,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,86050,151554,86050,86050,86050,86050,86050,86050,86050,0,348192,151554,86050,151554,151554,86050,86050,86050,86050,86050,86050,86050,86050,151554,151554,151554,151554,151554,151554,86050,86050,86048,86048,86048,86048,86048,86048,86048,86048,0,0,86048, };
static const uint32_t _53[] = { 151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,161826,131072,131072,131072,131072,131072,131072,0,0,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,65568,86050,86050,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _54[] = { 86050,86050,86050,86050,151554,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,86018,86050,86050,86050,86050,86050,151554,86050,151554,151554,151554,151554,151554,86050,151554,413696,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,86048,86048,86048,86048,86048,86048,86048,86048,86048,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0, };
static const uint32_t _55[] = { 86050,86050,151554,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,86050,86050,86050,86050,151554,151554,86050,86050,413696,348192,86050,86050,161794,161794,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,151554,86050,86050,151554,151554,151554,86050,151554,86050,86050,86050,413696,413696,0,0,0,0,0,0,0,0,131072,131072,131072,131072, };
static const uint32_t _56[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,151554,151554,151554,151554,151554,86050,86050,86050,86050,86050,86050,86050,86050,151554,151554,86050,86048,0,0,0,131072,131072,131072,131072,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,161794,161794,161794,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161826,161826,161826,161826,161826,161826,131072,131072, };
static const uint32_t _57[] = { 163734,163734,163734,163734,163734,163734,163734,163734,163734,0,0,0,0,0,0,0,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,0,0,163418,163418,163418,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,86048,86048,86048,131072,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,151552,86048,86048,86048,86048,86048,86048,86048,161794,161794,161794,161794,86048,161794,161794,161794,161794,161794,161794,86048,161794,161794,151552,86048,86048,161794,0,0,0,0,0, };
static const uint32_t _58[] = { 161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161846,163222,161814,161814,161814,163222,161814,161814, };
static const uint32_t _59[] = { 161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,163222,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86048,86048,86048,86048,86048,0,86048,86048,86048,86048,86048, };
static const uint32_t _60[] = { 163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222, };
static const uint32_t _61[] = { 163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163222,163222,163222,163222,163734,163734,161814,161814,163418,161814,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222, };
static const uint32_t _62[] = { 163222,163222,163222,163222,163222,163222,163222,163222,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,0,0,163418,163418,163418,163418,163418,163418,0,0,163222,163222,163222,163222,163222,163222,163222,163222,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,163222,163222,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,0,0,163418,163418,163418,163418,163418,163418,0,0,163222,163222,163222,163222,163222,163222,163222,163222,0,163418,0,163418,0,163418,0,163418,163222,163222,163222,163222,163222,163222,163222,163222,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,0,0, };
static const uint32_t _63[] = { 163734,163734,163734,163734,163734,163734,163734,163734,163538,163538,163538,163538,163538,163538,163538,163538,163734,163734,163734,163734,163734,163734,163734,163734,163538,163538,163538,163538,163538,163538,163538,163538,163734,163734,163734,163734,163734,163734,163734,163734,163538,163538,163538,163538,163538,163538,163538,163538,163222,163222,163734,163734,163734,0,163222,163734,163418,163418,163418,163418,163538,131104,163222,131104,131104,131104,163734,163734,163734,0,163222,163734,163418,163418,163418,163418,163538,131104,131104,131104,163222,163222,163222,163222,0,0,163222,163222,163418,163418,163418,163418,0,131104,131104,131104,163222,163222,163222,163222,163222,163222,163222,163222,163418,163418,163418,163418,163418,131104,131104,131104,0,0,163734,163734,163734,0,163222,163734,163418,163418,163418,163418,163538,131104,131104,0, };
static const uint32_t _64[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,32800,98336,32800,32800,32800,131072,131072,131072,131072,131072,131072,131073,131072,131104,131104,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131104,131072,131072,131104,0,0,32800,32800,32800,32800,32800,131072,131072,131072,131073,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,151552,151553,131072,131072,131072,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131072,151552,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,32800,32801,32801,32801,32801,32768,32800,32800,32800,32800,32800,32800,32800,32800,32800,32800,131072,161846,0,0,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,161846, };
static const uint32_t _65[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,0,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,161846,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,86049,86049,86049,86049,86049,86049,86049,86049,86049,86049,86049,86049,86049,65568,65568,65568,65568,86049,65568,65568,65568,86049,86049,86048,86048,86048,86048,86049,86049,86049,86049,86049,86048,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _66[] = { 131072,131072,161819,131072,131072,131072,131072,161819,131072,131072,161815,161819,161819,161819,161815,161815,161819,161819,161819,161815,131072,161819,131072,131072,161793,161819,161819,161819,161819,161819,131072,131072,131072,131072,131072,131072,161819,131072,163418,131072,161819,131073,163418,163418,161819,161819,161792,161815,161819,161819,163418,161819,161815,161795,161795,161795,161795,161814,131072,131072,161815,161815,161819,161819,131073,131073,131073,131073,131073,161819,161815,161815,161815,161815,131072,131073,131072,131072,163222,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222, };
static const uint32_t _67[] = { 161794,161794,161794,163418,163222,161794,161794,161794,161794,131072,131072,131072,0,0,0,0,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131072,131073,131073,131073,131073,131073,131073,131072,131073,131073,131072,131072,131072,131072,131073,131073,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131072,131073,131072,131072,131072,131072,131072,131072,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073, };
static const uint32_t _70[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131072,131072,131072, };
static const uint32_t _71[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131072,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _72[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _73[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132698,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,132502,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _75[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,131072,131072,131072,131072,131073,131073,131072,131072,131073,131073,131072,131072,131072,131073,131073,131073,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131072,131073,131072,131072,131073,131073,131073,131073,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,131073,131073, };
static const uint32_t _76[] = { 131072,131072,131072,131072,131072,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131072,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _79[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073, };
static const uint32_t _86[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131073,131072,131072,131073,131073,131073,131073,131073,131073,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _87[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _88[] = { 163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,0,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,0,163418,163222,163418,163418,163418,163222,163222,163418,163222,163418,163222,163418,163222,163418,163418,163418,163418,161814,163418,163222,161814,163418,163222,161814,161814,161814,161814,161814,161846,161846,163418,163418, };
static const uint32_t _89[] = { 163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,161814,131072,131072,131072,131072,131072,131072,163418,163222,163418,163222,86048,86048,86048,163418,163222,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _90[] = { 163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,0,163222,0,0,0,0,0,163222,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,161826,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,348192, };
static const uint32_t _91[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050, };
static const uint32_t _92[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131106,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _93[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _95[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0, };
static const uint32_t _96[] = { 131072,131072,131072,131072,131072,161826,161794,161794,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,86048,86048,86048,86016,86016,131072,161826,161826,161826,161826,161826,131072,131072,161794,161794,161794,161826,161794,131072,131072,131072,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _97[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,86048,86048,137248,137248,161826,161826,161794,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,161826,161826,161826,161794, };
static const uint32_t _98[] = { 0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,194562,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _99[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _100[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _155[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _319[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0, };
static const uint32_t _320[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161826,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _329[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161826,161826,161826,161826,161826,161826,131072,131072, };
static const uint32_t _332[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161826,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,161794,86048,65568,65568,65568,131072,86050,86050,86050,86050,86050,86050,86050,86050,86048,86048,131072,161826, };
static const uint32_t _333[] = { 163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,161846,161846,86050,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,86048,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0, };
static const uint32_t _334[] = { 131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,161826,161826,161826,161826,161826,161826,161826,161826,161826,131104,131104,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,161814,161814,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,161846,161814,161814,161814,161814,161814,161814,161814,161814,163418,163222,163418,163222,163418,163418,163222, };
static const uint32_t _335[] = { 163418,163222,163418,163222,163418,163222,163418,163222,161826,131104,131104,163418,163222,163418,161814,161794,163418,163222,163418,163222,163222,161814,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163418,163418,163418,163418,161814,163418,163418,163418,163418,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,163418,163222,0,0,163418,163222,163418,163418,163418,163418,163222,163418,163222,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,163418,163222,161794,161846,161846,161814,161794,161794,161794,161794,161794, };
static const uint32_t _336[] = { 161794,161794,86050,161794,161794,161794,348192,161794,161794,161794,161794,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,86050,86050,151554,131072,131072,131072,131072,348192,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,0,0,0,0,0,0,0,0, };
static const uint32_t _337[] = { 151554,151554,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,348192,86050,0,0,0,0,0,0,0,0,131072,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,161794,161794,161794,161794,161794,161794,131072,131072,131072,161794,131072,161794,161794,86050, };
static const uint32_t _338[] = { 151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86048,86048,86048,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,151554,413696,0,0,0,0,0,0,0,0,0,0,0,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0, };
static const uint32_t _339[] = { 86050,86050,86050,151554,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,151554,151554,86050,86050,86050,86050,151554,151554,86050,86050,151554,151554,413696,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,161826,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,131072,131072,161794,161794,161794,161794,161794,86050,161826,161794,161794,161794,161794,161794,161794,161794,161794,161794,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,161794,161794,161794,161794,0, };
static const uint32_t _340[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86050,151554,151554,86050,86050,151554,151554,86050,86050,0,0,0,0,0,0,0,0,0,161794,161794,161794,86050,161794,161794,161794,161794,161794,161794,161794,161794,86050,151554,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161826,161794,161794,161794,161794,161794,161794,131072,131072,131072,161794,151554,86050,151554,161794,161794, };
static const uint32_t _341[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,161794,86050,86050,86050,161794,161794,86050,86050,161794,161794,161794,161794,161794,86050,86048,161794,86048,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161826,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,86050,86050,151554,151554,131072,131072,161794,161826,161826,151554,348192,0,0,0,0,0,0,0,0,0, };
static const uint32_t _342[] = { 0,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,161814,163222,161814,161814,161814,161814,161814,161814,161814,131104,161846,161846,161846,161846,161814,161814,161814,161814,161814,161814,161814,161814,161814,161826,131104,131104,0,0,0,0,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734, };
static const uint32_t _343[] = { 163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,163734,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,86050,151554,151554,86050,151554,151554,131072,151552,348192,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0, };
static const uint32_t _431[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0, };
static const uint32_t _500[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _501[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _502[] = { 163734,163734,163734,163734,163734,163734,163734,0,0,0,0,0,0,0,0,0,0,0,0,163734,163734,163734,163734,163734,0,0,0,0,0,161794,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131073,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,0,161794,0,161794,161794,0,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _503[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,131104,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _504[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,137218,137218,137218,137218,137218,137218,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _506[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _507[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,137218,137218,131072,131072,0,0, };
static const uint32_t _508[] = { 118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,131072,131072,131072,131104,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,131072,131072,131072,151552,151552,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,151552,151552,151552,131072,131072,131104,0,131072,131104,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131073,131073,131073,131073,131073,131073,0,131073,131072,131072,131072,0,0,0,0,137218,161794,137218,161794,137218,0,137218,161794,137218,161794,137218,161794,137218,161794,137218,161794, };
static const uint32_t _509[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,32800, };
static const uint32_t _510[] = { 0,131072,131072,131072,131072,131072,131072,131104,131072,131072,131072,131073,131072,131072,131104,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131104,131072,131073,131073,131073,131072,131072,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,131072,131073,131072,131105,151552,131104,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,131072,131073,131072,131073,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161826,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _511[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,88098,88098,194562,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,0,0,0,131072,131072,131073,131104,131072,131072,131072,0,131072,131073,131073,131073,131073,131072,131072,0,32768,32768,32768,32768,32768,32768,32768,32768,32768,32,32,32,131072,131072,0,0, };
static const uint32_t _512[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _513[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0, };
static const uint32_t _514[] = { 131072,131072,131072,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _515[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,86048,0,0, };
static const uint32_t _517[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,86048,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0, };
static const uint32_t _518[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,0,0,0,0,0, };
static const uint32_t _519[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,131072,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _520[] = { 163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _521[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,0,0,0,0,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,0,0,0,0, };
static const uint32_t _522[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _526[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _528[] = { 161794,161794,161794,161794,161794,161794,0,0,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,0,0,0,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _529[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,0,0,0,0,0,131072,131072,131072,131072,131072, };
static const uint32_t _530[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,131072,131072,0,0,0,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _531[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,131072,131072,161794,161794,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _532[] = { 161794,86050,86050,86050,0,86050,86050,0,0,0,0,0,86050,86050,86050,86050,161794,161794,161794,161794,0,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,86048,86048,86048,0,0,0,0,348192,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072, };
static const uint32_t _533[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,86048,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0, };
static const uint32_t _534[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,131072,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _535[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _536[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _537[] = { 163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,0,0,0,0,0,0,0,0,0,0,0,0,0,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072, };
static const uint32_t _538[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _540[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0, };
static const uint32_t _541[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,86050,86050,131072,0,0,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _542[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,161794,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _543[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0, };
static const uint32_t _544[] = { 151554,86050,151554,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,348192,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,348192, };
static const uint32_t _545[] = { 86048,86048,151554,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,86050,86050,86050,86050,151554,151554,348192,86048,131072,131072,32,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,32,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0, };
static const uint32_t _546[] = { 86050,86050,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,151554,86050,86050,86050,86050,86050,86050,348192,348192,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,161794,151554,151554,161794,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,131072,131072,161794,0,0,0,0,0,0,0,0,0, };
static const uint32_t _547[] = { 86050,86050,151554,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,86050,86050,86050,86050,86050,86050,86050,86050,86050,151554,413696,161794,161794,161794,161794,131072,131072,131072,131072,86048,86048,86048,86048,131072,151554,86050,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,161794,131072,161794,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _548[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,86050,86050,86050,151554,151554,86050,413696,86048,86050,131072,131072,131072,131072,131072,131072,86050,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _549[] = { 161794,161794,161794,161794,161794,161794,161794,0,161794,0,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,131072,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,151554,151554,151554,86050,86050,86050,86050,86050,86050,86048,348192,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0, };
static const uint32_t _550[] = { 86050,86050,151554,151554,0,161794,161794,161794,161794,161794,161794,161794,161794,0,0,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,0,161794,161794,161794,161794,161794,0,86048,86048,161794,86018,151554,86050,151554,151554,151554,151554,0,0,151554,151554,0,0,151554,151554,413696,0,0,161794,0,0,0,0,0,0,86018,0,0,0,0,0,161794,161794,161794,161794,161794,151554,151554,0,0,86048,86048,86048,86048,86048,86048,86048,0,0,0,86048,86048,86048,86048,86048,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _552[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,86050,86050,86050,86050,86050,86050,86050,86050,151554,151554,348192,86050,86050,151554,86048,161794,161794,161794,161794,131072,131072,131072,131072,131072,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,0,131072,86048,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _553[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86018,151554,151554,86050,86050,86050,86050,86050,86050,151554,86050,151554,151554,86018,151554,86050,86050,151554,348192,86048,161794,161794,131072,161794,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _555[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86018,151554,151554,86050,86050,86050,86050,0,0,151554,151554,151554,151554,86050,86050,151554,348192,86048,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,161794,161794,161794,161794,86050,86050,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _556[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,86050,86050,86050,86050,86050,86050,86050,86050,151554,151554,86050,151554,348192,86050,131072,131072,131072,161794,0,0,0,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _557[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,151554,86050,151554,151554,86050,86050,86050,86050,86050,86050,413696,86048,161794,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _558[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,86050,86050,86050,151554,151554,86050,86050,86050,86050,151554,86050,86050,86050,86050,348192,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _560[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,86050,86050,86050,86050,86050,86050,86050,86050,86050,151554,348192,86048,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _561[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,161794, };
static const uint32_t _562[] = { 161794,161794,161794,161794,161794,161794,161794,0,0,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86018,151554,151554,151554,151554,151554,0,151554,151554,0,0,86050,86050,413696,348192,161794,151554,161794,151554,86048,131072,131072,131072,0,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _563[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,86050,86050,86050,86050,0,0,86050,86050,151554,151554,151554,151554,348192,161794,131072,161794,151554,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _564[] = { 161794,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,348192,86050,86050,86050,86050,151554,161794,86050,86050,86050,86050,131072,131072,131072,131072,131072,131072,131072,131072,348192,0,0,0,0,0,0,0,0,161794,86050,86050,86050,86050,86050,86050,151554,151554,86050,86050,86050,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _565[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,151554,86048,348192,131072,131072,131072,161794,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0, };
static const uint32_t _568[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,86050,86050,86050,86050,86050,86050,86050,0,86050,86050,86050,86050,86050,86050,151554,348192,161794,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,131072,131072,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _569[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,0,151554,86050,86050,86050,86050,86050,86050,86050,151554,86050,86050,151554,86050,86050,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _570[] = { 161794,161794,161794,161794,161794,161794,161794,0,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,86050,86050,86050,86050,0,0,0,86050,0,86050,86050,0,86050,86050,86050,86048,86050,348192,348192,161794,86050,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,0,161794,161794,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _571[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,151554,151554,151554,151554,151554,0,86050,86050,0,151554,151554,86050,151554,348192,161794,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _573[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86050,86050,151554,151554,131072,131072,0,0,0,0,0,0,0, };
static const uint32_t _575[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,131072, };
static const uint32_t _583[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _584[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _586[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _616[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,32,32,32,32,32,32,32,32,32,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _652[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _724[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _725[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,86048,86048,86048,86048,86048,131072,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _726[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,86048,86048,86048,86048,86048,86048,131072,131072,131072,131072,131072,131072,131072,131072,131072,161826,161826,161826,161826,131072,131072,0,0,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,131072,131072,131072,131072,131072,131072,131072,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,161794,161794,161794, };
static const uint32_t _727[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _732[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222, };
static const uint32_t _733[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _734[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,86050,161794,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554,151554, };
static const uint32_t _735[] = { 151554,151554,151554,151554,151554,151554,151554,151554,0,0,0,0,0,0,0,86050,86050,86050,86050,161826,161826,161826,161826,161826,161826,161826,161826,161826,161826,161826,161826,161826,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161826,161826,131072,161826,86048,0,0,0,0,0,0,0,0,0,0,0,151554,151554,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _783[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0, };
static const uint32_t _793[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _794[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _866[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _869[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0, };
static const uint32_t _888[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0, };
static const uint32_t _889[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,131072,86048,86050,131072,32800,32800,32800,32800,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _929[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _930[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,86016,151552,86048,86048,86048,131072,131072,131072,151552,86016,86016,86016,86016,86016,32800,32800,32800,32800,32800,32800,32800,32800,86048,86048,86048,86048,86048, };
static const uint32_t _931[] = { 86048,86048,86048,131072,131072,86048,86048,86048,86048,86048,86048,86048,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,86048,86048,86048,86048,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _932[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,86048,86048,86048,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _933[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _934[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0, };
static const uint32_t _936[] = { 161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,161815,161815,161815,0,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819, };
static const uint32_t _937[] = { 161819,161819,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,0,161819,161819,0,0,161819,0,0,161819,161819,0,0,161819,161819,161819,161819,0,161819,161819,161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,0,161815,0,161815,161815,161815,161815,161815,161815,161815,0,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815, };
static const uint32_t _938[] = { 161815,161815,161815,161815,161819,161819,0,161819,161819,161819,161819,0,0,161819,161819,161819,161819,161819,161819,161819,161819,0,161819,161819,161819,161819,161819,161819,161819,0,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,0,161819,161819,161819,161819,0,161819,161819,161819,161819,161819,0,161819,0,0,0,161819,161819,161819,161819,161819,161819,161819,0,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819, };
static const uint32_t _939[] = { 161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815, };
static const uint32_t _940[] = { 161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819, };
static const uint32_t _941[] = { 161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,0,0,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,131073,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,131073,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,131073,161815,161815,161815,161815, };
static const uint32_t _942[] = { 161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,131073,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,131073,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,131073,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,131073,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815, };
static const uint32_t _943[] = { 161815,161815,161815,161815,161815,161815,161815,161815,161815,131073,161815,161815,161815,161815,161815,161815,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,161819,131073,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,161815,131073,161815,161815,161815,161815,161815,161815,161819,161815,0,0,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553,151553, };
static const uint32_t _948[] = { 86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,131072,131072,131072,131072,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,131072,131072,131072,131072,131072,131072,131072,131072,86048,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _949[] = { 131072,131072,131072,131072,86048,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,86048,86048,86048,86048,86048,0,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,86048,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _960[] = { 86050,86050,86050,86050,86050,86050,86050,0,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,86050,0,0,86050,86050,86050,86050,86050,86050,86050,0,86050,86050,0,86050,86050,86050,86050,86050,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _962[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,86048,86048,86048,86048,86048,86048,86048,161826,161826,161826,161826,161826,161826,161826,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,161794,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _965[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,86048,86048,86048,86048,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,131072, };
static const uint32_t _977[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,86048,86048,86048,86048,86048,86048,86048,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _978[] = { 163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163418,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,163222,86048,86048,86048,86050,86048,86048,86048,161826,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _984[] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _985[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _986[] = { 0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _988[] = { 161795,161795,161795,161795,0,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,0,161795,161795,0,161795,0,0,161795,0,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,0,161795,161795,161795,161795,0,161795,0,161795,0,0,0,0,0,0,161795,0,0,0,0,161795,0,161795,0,161795,0,161795,161795,161795,0,161795,161795,0,161795,0,0,161795,0,161795,0,161795,0,161795,0,161795,0,161795,161795,0,161795,0,0,161795,161795,161795,161795,0,161795,161795,161795,161795,161795,161795,161795,0,161795,161795,161795,161795,0,161795,161795,161795,161795,0,161795,0, };
static const uint32_t _989[] = { 161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,0,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,0,0,0,0,0,161795,161795,161795,0,161795,161795,161795,161795,161795,0,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,161795,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131073,131073,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _992[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _993[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _994[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131072,131072,131072,131072,131072,131072,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131072,131072,131072,131072,131072,131072,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131098, };
static const uint32_t _995[] = { 131098,131098,131098,131098,131098,131098,131098,131098,131098,131098,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _996[] = { 131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _999[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131104,131104,131104,131104,131104, };
static const uint32_t _1005[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0, };
static const uint32_t _1006[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _1007[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _1008[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _1009[] = { 131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _1010[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072, };
static const uint32_t _1011[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072, };
static const uint32_t _1012[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,131072,131072,131072,131072,131072,0,0,0,131072,131072,131072,0,0,0,0,0, };
static const uint32_t _1013[] = { 131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _1015[] = { 131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,131072,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,151552,151552,151552,151552,151552,151552,151552,151552,151552,151552,0,0,0,0,0,0, };
static const uint32_t _1357[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _1390[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _1392[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _1437[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794, };
static const uint32_t _1495[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _1524[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _1574[] = { 161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,161794,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, };
static const uint32_t _7168[] = { 32768,32800,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336,98336, };
static const uint32_t _7171[] = { 118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,118816,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768,32768, };
static const uint32_t *_blocks[] = {
 _0,_1,_2,_3,_4,_5,_6,_7,
 _8,_9,_10,_11,_12,_13,_14,_15,
 _16,_17,_18,_19,_20,_21,_22,_23,
 _24,_25,_26,_27,_28,_29,_30,_31,
 _32,_33,_34,0,_36,_37,_38,_39,
 _40,0,0,0,_44,_45,_46,_47,
 _48,_49,_50,_51,_52,_53,_54,_55,
 _56,_57,_58,_59,_60,_61,_62,_63,
 _64,_65,_66,_67,0,0,_70,_71,
 _72,_73,0,_75,_76,0,0,_79,
 0,0,0,0,0,0,_86,_87,
 _88,_89,_90,_91,_92,_93,0,_95,
 _96,_97,_98,_99,_100,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,_155,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_319,
 _320,0,0,0,0,0,0,0,
 0,_329,0,0,_332,_333,_334,_335,
 _336,_337,_338,_339,_340,_341,_342,_343,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_431,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_500,_501,_502,_503,
 _504,0,_506,_507,_508,_509,_510,_511,
 _512,_513,_514,_515,0,_517,_518,_519,
 _520,_521,_522,0,0,0,_526,0,
 _528,_529,_530,_531,_532,_533,_534,_535,
 _536,_537,_538,0,_540,_541,_542,_543,
 _544,_545,_546,_547,_548,_549,_550,0,
 _552,_553,0,_555,_556,_557,_558,0,
 _560,_561,_562,_563,_564,_565,0,0,
 _568,_569,_570,_571,0,_573,0,_575,
 0,0,0,0,0,0,0,_583,
 _584,0,_586,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 _616,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_652,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_724,_725,_726,_727,
 0,0,0,0,_732,_733,_734,_735,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_783,
 0,0,0,0,0,0,0,0,
 0,_793,_794,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,_866,0,0,_869,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 _888,_889,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,_929,_930,_931,_932,_933,_934,0,
 _936,_937,_938,_939,_940,_941,_942,_943,
 0,0,0,0,_948,_949,0,0,
 0,0,0,0,0,0,0,0,
 _960,0,_962,0,0,_965,0,0,
 0,0,0,0,0,0,0,0,
 0,_977,_978,0,0,0,0,0,
 _984,_985,_986,0,_988,_989,0,0,
 _992,_993,_994,_995,_996,0,0,_999,
 0,0,0,0,0,_1005,_1006,_1007,
 _1008,_1009,_1010,_1011,_1012,_1013,0,_1015,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,_1357,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,_1390,0,
 _1392,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,_1437,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,_1495,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,_1524,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,_1574,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 _7168,0,0,_7171,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
};
static const uint32_t _block_values[] = {
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,161794,0,0,0,0,
 0,161794,161794,161794,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,131073,131073,0,0,
 0,0,131072,0,0,131072,131072,0,
 131072,131072,131073,131073,131073,131073,0,0,
 0,0,0,0,0,0,131072,0,
 0,0,0,0,0,131072,131072,131072,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,0,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,0,
 0,161794,161794,161794,161794,161794,161794,161794,
 161794,0,161794,161794,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,161794,161794,0,0,0,0,
 0,161794,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,161794,161794,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,161794,161794,161794,161794,161794,161794,0,
 0,161794,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,161794,161794,161794,161794,161794,161794,161794,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,161794,161794,161794,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,161794,161794,161794,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,0,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,161794,0,161794,161794,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 131072,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 131072,131072,131072,131072,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,131072,0,
 131072,131072,131072,131072,131072,0,0,0,
 0,0,0,0,0,0,131072,0,
 0,0,0,0,0,0,0,0,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,0,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,0,161794,
 0,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,0,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,161794,161794,161794,0,0,0,0,
 0,0,0,0,0,0,0,0,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,161794,161794,
 161794,161794,161794,161794,161794,161794,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,32768,118816,0,32768,32768,32768,32768,
 32768,32768,32768,32768,32768,32768,32768,32768,
 32768,32768,32768,32768,32768,32768,32768,32768,
 32768,32768,32768,32768,32768,32768,32768,32768,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0,
};
inline uint32_t get_value(char32_t cp) {
  auto i = cp / _block_size;
  auto bl = _blocks[i];
  if (bl) {
    auto off = cp % _block_size;
    return bl[off];
  }
  return _block_values[i];
}
}
const std::unordered_map<char32_t, const char32_t*> _simple_case_mappings = {
{ 0x00000041, U"\U00000041\U00000061\U00000041" },
{ 0x00000042, U"\U00000042\U00000062\U00000042" },
{ 0x00000043, U"\U00000043\U00000063\U00000043" },
{ 0x00000044, U"\U00000044\U00000064\U00000044" },
{ 0x00000045, U"\U00000045\U00000065\U00000045" },
{ 0x00000046, U"\U00000046\U00000066\U00000046" },
{ 0x00000047, U"\U00000047\U00000067\U00000047" },
{ 0x00000048, U"\U00000048\U00000068\U00000048" },
{ 0x00000049, U"\U00000049\U00000069\U00000049" },
{ 0x0000004A, U"\U0000004A\U0000006A\U0000004A" },
{ 0x0000004B, U"\U0000004B\U0000006B\U0000004B" },
{ 0x0000004C, U"\U0000004C\U0000006C\U0000004C" },
{ 0x0000004D, U"\U0000004D\U0000006D\U0000004D" },
{ 0x0000004E, U"\U0000004E\U0000006E\U0000004E" },
{ 0x0000004F, U"\U0000004F\U0000006F\U0000004F" },
{ 0x00000050, U"\U00000050\U00000070\U00000050" },
{ 0x00000051, U"\U00000051\U00000071\U00000051" },
{ 0x00000052, U"\U00000052\U00000072\U00000052" },
{ 0x00000053, U"\U00000053\U00000073\U00000053" },
{ 0x00000054, U"\U00000054\U00000074\U00000054" },
{ 0x00000055, U"\U00000055\U00000075\U00000055" },
{ 0x00000056, U"\U00000056\U00000076\U00000056" },
{ 0x00000057, U"\U00000057\U00000077\U00000057" },
{ 0x00000058, U"\U00000058\U00000078\U00000058" },
{ 0x00000059, U"\U00000059\U00000079\U00000059" },
{ 0x0000005A, U"\U0000005A\U0000007A\U0000005A" },
{ 0x00000061, U"\U00000041\U00000061\U00000041" },
{ 0x00000062, U"\U00000042\U00000062\U00000042" },
{ 0x00000063, U"\U00000043\U00000063\U00000043" },
{ 0x00000064, U"\U00000044\U00000064\U00000044" },
{ 0x00000065, U"\U00000045\U00000065\U00000045" },
{ 0x00000066, U"\U00000046\U00000066\U00000046" },
{ 0x00000067, U"\U00000047\U00000067\U00000047" },
{ 0x00000068, U"\U00000048\U00000068\U00000048" },
{ 0x00000069, U"\U00000049\U00000069\U00000049" },
{ 0x0000006A, U"\U0000004A\U0000006A\U0000004A" },
{ 0x0000006B, U"\U0000004B\U0000006B\U0000004B" },
{ 0x0000006C, U"\U0000004C\U0000006C\U0000004C" },
{ 0x0000006D, U"\U0000004D\U0000006D\U0000004D" },
{ 0x0000006E, U"\U0000004E\U0000006E\U0000004E" },
{ 0x0000006F, U"\U0000004F\U0000006F\U0000004F" },
{ 0x00000070, U"\U00000050\U00000070\U00000050" },
{ 0x00000071, U"\U00000051\U00000071\U00000051" },
{ 0x00000072, U"\U00000052\U00000072\U00000052" },
{ 0x00000073, U"\U00000053\U00000073\U00000053" },
{ 0x00000074, U"\U00000054\U00000074\U00000054" },
{ 0x00000075, U"\U00000055\U00000075\U00000055" },
{ 0x00000076, U"\U00000056\U00000076\U00000056" },
{ 0x00000077, U"\U00000057\U00000077\U00000057" },
{ 0x00000078, U"\U00000058\U00000078\U00000058" },
{ 0x00000079, U"\U00000059\U00000079\U00000059" },
{ 0x0000007A, U"\U0000005A\U0000007A\U0000005A" },
{ 0x000000B5, U"\U0000039C\U000000B5\U0000039C" },
{ 0x000000C0, U"\U000000C0\U000000E0\U000000C0" },
{ 0x000000C1, U"\U000000C1\U000000E1\U000000C1" },
{ 0x000000C2, U"\U000000C2\U000000E2\U000000C2" },
{ 0x000000C3, U"\U000000C3\U000000E3\U000000C3" },
{ 0x000000C4, U"\U000000C4\U000000E4\U000000C4" },
{ 0x000000C5, U"\U000000C5\U000000E5\U000000C5" },
{ 0x000000C6, U"\U000000C6\U000000E6\U000000C6" },
{ 0x000000C7, U"\U000000C7\U000000E7\U000000C7" },
{ 0x000000C8, U"\U000000C8\U000000E8\U000000C8" },
{ 0x000000C9, U"\U000000C9\U000000E9\U000000C9" },
{ 0x000000CA, U"\U000000CA\U000000EA\U000000CA" },
{ 0x000000CB, U"\U000000CB\U000000EB\U000000CB" },
{ 0x000000CC, U"\U000000CC\U000000EC\U000000CC" },
{ 0x000000CD, U"\U000000CD\U000000ED\U000000CD" },
{ 0x000000CE, U"\U000000CE\U000000EE\U000000CE" },
{ 0x000000CF, U"\U000000CF\U000000EF\U000000CF" },
{ 0x000000D0, U"\U000000D0\U000000F0\U000000D0" },
{ 0x000000D1, U"\U000000D1\U000000F1\U000000D1" },
{ 0x000000D2, U"\U000000D2\U000000F2\U000000D2" },
{ 0x000000D3, U"\U000000D3\U000000F3\U000000D3" },
{ 0x000000D4, U"\U000000D4\U000000F4\U000000D4" },
{ 0x000000D5, U"\U000000D5\U000000F5\U000000D5" },
{ 0x000000D6, U"\U000000D6\U000000F6\U000000D6" },
{ 0x000000D8, U"\U000000D8\U000000F8\U000000D8" },
{ 0x000000D9, U"\U000000D9\U000000F9\U000000D9" },
{ 0x000000DA, U"\U000000DA\U000000FA\U000000DA" },
{ 0x000000DB, U"\U000000DB\U000000FB\U000000DB" },
{ 0x000000DC, U"\U000000DC\U000000FC\U000000DC" },
{ 0x000000DD, U"\U000000DD\U000000FD\U000000DD" },
{ 0x000000DE, U"\U000000DE\U000000FE\U000000DE" },
{ 0x000000E0, U"\U000000C0\U000000E0\U000000C0" },
{ 0x000000E1, U"\U000000C1\U000000E1\U000000C1" },
{ 0x000000E2, U"\U000000C2\U000000E2\U000000C2" },
{ 0x000000E3, U"\U000000C3\U000000E3\U000000C3" },
{ 0x000000E4, U"\U000000C4\U000000E4\U000000C4" },
{ 0x000000E5, U"\U000000C5\U000000E5\U000000C5" },
{ 0x000000E6, U"\U000000C6\U000000E6\U000000C6" },
{ 0x000000E7, U"\U000000C7\U000000E7\U000000C7" },
{ 0x000000E8, U"\U000000C8\U000000E8\U000000C8" },
{ 0x000000E9, U"\U000000C9\U000000E9\U000000C9" },
{ 0x000000EA, U"\U000000CA\U000000EA\U000000CA" },
{ 0x000000EB, U"\U000000CB\U000000EB\U000000CB" },
{ 0x000000EC, U"\U000000CC\U000000EC\U000000CC" },
{ 0x000000ED, U"\U000000CD\U000000ED\U000000CD" },
{ 0x000000EE, U"\U000000CE\U000000EE\U000000CE" },
{ 0x000000EF, U"\U000000CF\U000000EF\U000000CF" },
{ 0x000000F0, U"\U000000D0\U000000F0\U000000D0" },
{ 0x000000F1, U"\U000000D1\U000000F1\U000000D1" },
{ 0x000000F2, U"\U000000D2\U000000F2\U000000D2" },
{ 0x000000F3, U"\U000000D3\U000000F3\U000000D3" },
{ 0x000000F4, U"\U000000D4\U000000F4\U000000D4" },
{ 0x000000F5, U"\U000000D5\U000000F5\U000000D5" },
{ 0x000000F6, U"\U000000D6\U000000F6\U000000D6" },
{ 0x000000F8, U"\U000000D8\U000000F8\U000000D8" },
{ 0x000000F9, U"\U000000D9\U000000F9\U000000D9" },
{ 0x000000FA, U"\U000000DA\U000000FA\U000000DA" },
{ 0x000000FB, U"\U000000DB\U000000FB\U000000DB" },
{ 0x000000FC, U"\U000000DC\U000000FC\U000000DC" },
{ 0x000000FD, U"\U000000DD\U000000FD\U000000DD" },
{ 0x000000FE, U"\U000000DE\U000000FE\U000000DE" },
{ 0x000000FF, U"\U00000178\U000000FF\U00000178" },
{ 0x00000100, U"\U00000100\U00000101\U00000100" },
{ 0x00000101, U"\U00000100\U00000101\U00000100" },
{ 0x00000102, U"\U00000102\U00000103\U00000102" },
{ 0x00000103, U"\U00000102\U00000103\U00000102" },
{ 0x00000104, U"\U00000104\U00000105\U00000104" },
{ 0x00000105, U"\U00000104\U00000105\U00000104" },
{ 0x00000106, U"\U00000106\U00000107\U00000106" },
{ 0x00000107, U"\U00000106\U00000107\U00000106" },
{ 0x00000108, U"\U00000108\U00000109\U00000108" },
{ 0x00000109, U"\U00000108\U00000109\U00000108" },
{ 0x0000010A, U"\U0000010A\U0000010B\U0000010A" },
{ 0x0000010B, U"\U0000010A\U0000010B\U0000010A" },
{ 0x0000010C, U"\U0000010C\U0000010D\U0000010C" },
{ 0x0000010D, U"\U0000010C\U0000010D\U0000010C" },
{ 0x0000010E, U"\U0000010E\U0000010F\U0000010E" },
{ 0x0000010F, U"\U0000010E\U0000010F\U0000010E" },
{ 0x00000110, U"\U00000110\U00000111\U00000110" },
{ 0x00000111, U"\U00000110\U00000111\U00000110" },
{ 0x00000112, U"\U00000112\U00000113\U00000112" },
{ 0x00000113, U"\U00000112\U00000113\U00000112" },
{ 0x00000114, U"\U00000114\U00000115\U00000114" },
{ 0x00000115, U"\U00000114\U00000115\U00000114" },
{ 0x00000116, U"\U00000116\U00000117\U00000116" },
{ 0x00000117, U"\U00000116\U00000117\U00000116" },
{ 0x00000118, U"\U00000118\U00000119\U00000118" },
{ 0x00000119, U"\U00000118\U00000119\U00000118" },
{ 0x0000011A, U"\U0000011A\U0000011B\U0000011A" },
{ 0x0000011B, U"\U0000011A\U0000011B\U0000011A" },
{ 0x0000011C, U"\U0000011C\U0000011D\U0000011C" },
{ 0x0000011D, U"\U0000011C\U0000011D\U0000011C" },
{ 0x0000011E, U"\U0000011E\U0000011F\U0000011E" },
{ 0x0000011F, U"\U0000011E\U0000011F\U0000011E" },
{ 0x00000120, U"\U00000120\U00000121\U00000120" },
{ 0x00000121, U"\U00000120\U00000121\U00000120" },
{ 0x00000122, U"\U00000122\U00000123\U00000122" },
{ 0x00000123, U"\U00000122\U00000123\U00000122" },
{ 0x00000124, U"\U00000124\U00000125\U00000124" },
{ 0x00000125, U"\U00000124\U00000125\U00000124" },
{ 0x00000126, U"\U00000126\U00000127\U00000126" },
{ 0x00000127, U"\U00000126\U00000127\U00000126" },
{ 0x00000128, U"\U00000128\U00000129\U00000128" },
{ 0x00000129, U"\U00000128\U00000129\U00000128" },
{ 0x0000012A, U"\U0000012A\U0000012B\U0000012A" },
{ 0x0000012B, U"\U0000012A\U0000012B\U0000012A" },
{ 0x0000012C, U"\U0000012C\U0000012D\U0000012C" },
{ 0x0000012D, U"\U0000012C\U0000012D\U0000012C" },
{ 0x0000012E, U"\U0000012E\U0000012F\U0000012E" },
{ 0x0000012F, U"\U0000012E\U0000012F\U0000012E" },
{ 0x00000130, U"\U00000130\U00000069\U00000130" },
{ 0x00000131, U"\U00000049\U00000131\U00000049" },
{ 0x00000132, U"\U00000132\U00000133\U00000132" },
{ 0x00000133, U"\U00000132\U00000133\U00000132" },
{ 0x00000134, U"\U00000134\U00000135\U00000134" },
{ 0x00000135, U"\U00000134\U00000135\U00000134" },
{ 0x00000136, U"\U00000136\U00000137\U00000136" },
{ 0x00000137, U"\U00000136\U00000137\U00000136" },
{ 0x00000139, U"\U00000139\U0000013A\U00000139" },
{ 0x0000013A, U"\U00000139\U0000013A\U00000139" },
{ 0x0000013B, U"\U0000013B\U0000013C\U0000013B" },
{ 0x0000013C, U"\U0000013B\U0000013C\U0000013B" },
{ 0x0000013D, U"\U0000013D\U0000013E\U0000013D" },
{ 0x0000013E, U"\U0000013D\U0000013E\U0000013D" },
{ 0x0000013F, U"\U0000013F\U00000140\U0000013F" },
{ 0x00000140, U"\U0000013F\U00000140\U0000013F" },
{ 0x00000141, U"\U00000141\U00000142\U00000141" },
{ 0x00000142, U"\U00000141\U00000142\U00000141" },
{ 0x00000143, U"\U00000143\U00000144\U00000143" },
{ 0x00000144, U"\U00000143\U00000144\U00000143" },
{ 0x00000145, U"\U00000145\U00000146\U00000145" },
{ 0x00000146, U"\U00000145\U00000146\U00000145" },
{ 0x00000147, U"\U00000147\U00000148\U00000147" },
{ 0x00000148, U"\U00000147\U00000148\U00000147" },
{ 0x0000014A, U"\U0000014A\U0000014B\U0000014A" },
{ 0x0000014B, U"\U0000014A\U0000014B\U0000014A" },
{ 0x0000014C, U"\U0000014C\U0000014D\U0000014C" },
{ 0x0000014D, U"\U0000014C\U0000014D\U0000014C" },
{ 0x0000014E, U"\U0000014E\U0000014F\U0000014E" },
{ 0x0000014F, U"\U0000014E\U0000014F\U0000014E" },
{ 0x00000150, U"\U00000150\U00000151\U00000150" },
{ 0x00000151, U"\U00000150\U00000151\U00000150" },
{ 0x00000152, U"\U00000152\U00000153\U00000152" },
{ 0x00000153, U"\U00000152\U00000153\U00000152" },
{ 0x00000154, U"\U00000154\U00000155\U00000154" },
{ 0x00000155, U"\U00000154\U00000155\U00000154" },
{ 0x00000156, U"\U00000156\U00000157\U00000156" },
{ 0x00000157, U"\U00000156\U00000157\U00000156" },
{ 0x00000158, U"\U00000158\U00000159\U00000158" },
{ 0x00000159, U"\U00000158\U00000159\U00000158" },
{ 0x0000015A, U"\U0000015A\U0000015B\U0000015A" },
{ 0x0000015B, U"\U0000015A\U0000015B\U0000015A" },
{ 0x0000015C, U"\U0000015C\U0000015D\U0000015C" },
{ 0x0000015D, U"\U0000015C\U0000015D\U0000015C" },
{ 0x0000015E, U"\U0000015E\U0000015F\U0000015E" },
{ 0x0000015F, U"\U0000015E\U0000015F\U0000015E" },
{ 0x00000160, U"\U00000160\U00000161\U00000160" },
{ 0x00000161, U"\U00000160\U00000161\U00000160" },
{ 0x00000162, U"\U00000162\U00000163\U00000162" },
{ 0x00000163, U"\U00000162\U00000163\U00000162" },
{ 0x00000164, U"\U00000164\U00000165\U00000164" },
{ 0x00000165, U"\U00000164\U00000165\U00000164" },
{ 0x00000166, U"\U00000166\U00000167\U00000166" },
{ 0x00000167, U"\U00000166\U00000167\U00000166" },
{ 0x00000168, U"\U00000168\U00000169\U00000168" },
{ 0x00000169, U"\U00000168\U00000169\U00000168" },
{ 0x0000016A, U"\U0000016A\U0000016B\U0000016A" },
{ 0x0000016B, U"\U0000016A\U0000016B\U0000016A" },
{ 0x0000016C, U"\U0000016C\U0000016D\U0000016C" },
{ 0x0000016D, U"\U0000016C\U0000016D\U0000016C" },
{ 0x0000016E, U"\U0000016E\U0000016F\U0000016E" },
{ 0x0000016F, U"\U0000016E\U0000016F\U0000016E" },
{ 0x00000170, U"\U00000170\U00000171\U00000170" },
{ 0x00000171, U"\U00000170\U00000171\U00000170" },
{ 0x00000172, U"\U00000172\U00000173\U00000172" },
{ 0x00000173, U"\U00000172\U00000173\U00000172" },
{ 0x00000174, U"\U00000174\U00000175\U00000174" },
{ 0x00000175, U"\U00000174\U00000175\U00000174" },
{ 0x00000176, U"\U00000176\U00000177\U00000176" },
{ 0x00000177, U"\U00000176\U00000177\U00000176" },
{ 0x00000178, U"\U00000178\U000000FF\U00000178" },
{ 0x00000179, U"\U00000179\U0000017A\U00000179" },
{ 0x0000017A, U"\U00000179\U0000017A\U00000179" },
{ 0x0000017B, U"\U0000017B\U0000017C\U0000017B" },
{ 0x0000017C, U"\U0000017B\U0000017C\U0000017B" },
{ 0x0000017D, U"\U0000017D\U0000017E\U0000017D" },
{ 0x0000017E, U"\U0000017D\U0000017E\U0000017D" },
{ 0x0000017F, U"\U00000053\U0000017F\U00000053" },
{ 0x00000180, U"\U00000243\U00000180\U00000243" },
{ 0x00000181, U"\U00000181\U00000253\U00000181" },
{ 0x00000182, U"\U00000182\U00000183\U00000182" },
{ 0x00000183, U"\U00000182\U00000183\U00000182" },
{ 0x00000184, U"\U00000184\U00000185\U00000184" },
{ 0x00000185, U"\U00000184\U00000185\U00000184" },
{ 0x00000186, U"\U00000186\U00000254\U00000186" },
{ 0x00000187, U"\U00000187\U00000188\U00000187" },
{ 0x00000188, U"\U00000187\U00000188\U00000187" },
{ 0x00000189, U"\U00000189\U00000256\U00000189" },
{ 0x0000018A, U"\U0000018A\U00000257\U0000018A" },
{ 0x0000018B, U"\U0000018B\U0000018C\U0000018B" },
{ 0x0000018C, U"\U0000018B\U0000018C\U0000018B" },
{ 0x0000018E, U"\U0000018E\U000001DD\U0000018E" },
{ 0x0000018F, U"\U0000018F\U00000259\U0000018F" },
{ 0x00000190, U"\U00000190\U0000025B\U00000190" },
{ 0x00000191, U"\U00000191\U00000192\U00000191" },
{ 0x00000192, U"\U00000191\U00000192\U00000191" },
{ 0x00000193, U"\U00000193\U00000260\U00000193" },
{ 0x00000194, U"\U00000194\U00000263\U00000194" },
{ 0x00000195, U"\U000001F6\U00000195\U000001F6" },
{ 0x00000196, U"\U00000196\U00000269\U00000196" },
{ 0x00000197, U"\U00000197\U00000268\U00000197" },
{ 0x00000198, U"\U00000198\U00000199\U00000198" },
{ 0x00000199, U"\U00000198\U00000199\U00000198" },
{ 0x0000019A, U"\U0000023D\U0000019A\U0000023D" },
{ 0x0000019C, U"\U0000019C\U0000026F\U0000019C" },
{ 0x0000019D, U"\U0000019D\U00000272\U0000019D" },
{ 0x0000019E, U"\U00000220\U0000019E\U00000220" },
{ 0x0000019F, U"\U0000019F\U00000275\U0000019F" },
{ 0x000001A0, U"\U000001A0\U000001A1\U000001A0" },
{ 0x000001A1, U"\U000001A0\U000001A1\U000001A0" },
{ 0x000001A2, U"\U000001A2\U000001A3\U000001A2" },
{ 0x000001A3, U"\U000001A2\U000001A3\U000001A2" },
{ 0x000001A4, U"\U000001A4\U000001A5\U000001A4" },
{ 0x000001A5, U"\U000001A4\U000001A5\U000001A4" },
{ 0x000001A6, U"\U000001A6\U00000280\U000001A6" },
{ 0x000001A7, U"\U000001A7\U000001A8\U000001A7" },
{ 0x000001A8, U"\U000001A7\U000001A8\U000001A7" },
{ 0x000001A9, U"\U000001A9\U00000283\U000001A9" },
{ 0x000001AC, U"\U000001AC\U000001AD\U000001AC" },
{ 0x000001AD, U"\U000001AC\U000001AD\U000001AC" },
{ 0x000001AE, U"\U000001AE\U00000288\U000001AE" },
{ 0x000001AF, U"\U000001AF\U000001B0\U000001AF" },
{ 0x000001B0, U"\U000001AF\U000001B0\U000001AF" },
{ 0x000001B1, U"\U000001B1\U0000028A\U000001B1" },
{ 0x000001B2, U"\U000001B2\U0000028B\U000001B2" },
{ 0x000001B3, U"\U000001B3\U000001B4\U000001B3" },
{ 0x000001B4, U"\U000001B3\U000001B4\U000001B3" },
{ 0x000001B5, U"\U000001B5\U000001B6\U000001B5" },
{ 0x000001B6, U"\U000001B5\U000001B6\U000001B5" },
{ 0x000001B7, U"\U000001B7\U00000292\U000001B7" },
{ 0x000001B8, U"\U000001B8\U000001B9\U000001B8" },
{ 0x000001B9, U"\U000001B8\U000001B9\U000001B8" },
{ 0x000001BC, U"\U000001BC\U000001BD\U000001BC" },
{ 0x000001BD, U"\U000001BC\U000001BD\U000001BC" },
{ 0x000001BF, U"\U000001F7\U000001BF\U000001F7" },
{ 0x000001C4, U"\U000001C4\U000001C6\U000001C5" },
{ 0x000001C5, U"\U000001C4\U000001C6\U000001C5" },
{ 0x000001C6, U"\U000001C4\U000001C6\U000001C5" },
{ 0x000001C7, U"\U000001C7\U000001C9\U000001C8" },
{ 0x000001C8, U"\U000001C7\U000001C9\U000001C8" },
{ 0x000001C9, U"\U000001C7\U000001C9\U000001C8" },
{ 0x000001CA, U"\U000001CA\U000001CC\U000001CB" },
{ 0x000001CB, U"\U000001CA\U000001CC\U000001CB" },
{ 0x000001CC, U"\U000001CA\U000001CC\U000001CB" },
{ 0x000001CD, U"\U000001CD\U000001CE\U000001CD" },
{ 0x000001CE, U"\U000001CD\U000001CE\U000001CD" },
{ 0x000001CF, U"\U000001CF\U000001D0\U000001CF" },
{ 0x000001D0, U"\U000001CF\U000001D0\U000001CF" },
{ 0x000001D1, U"\U000001D1\U000001D2\U000001D1" },
{ 0x000001D2, U"\U000001D1\U000001D2\U000001D1" },
{ 0x000001D3, U"\U000001D3\U000001D4\U000001D3" },
{ 0x000001D4, U"\U000001D3\U000001D4\U000001D3" },
{ 0x000001D5, U"\U000001D5\U000001D6\U000001D5" },
{ 0x000001D6, U"\U000001D5\U000001D6\U000001D5" },
{ 0x000001D7, U"\U000001D7\U000001D8\U000001D7" },
{ 0x000001D8, U"\U000001D7\U000001D8\U000001D7" },
{ 0x000001D9, U"\U000001D9\U000001DA\U000001D9" },
{ 0x000001DA, U"\U000001D9\U000001DA\U000001D9" },
{ 0x000001DB, U"\U000001DB\U000001DC\U000001DB" },
{ 0x000001DC, U"\U000001DB\U000001DC\U000001DB" },
{ 0x000001DD, U"\U0000018E\U000001DD\U0000018E" },
{ 0x000001DE, U"\U000001DE\U000001DF\U000001DE" },
{ 0x000001DF, U"\U000001DE\U000001DF\U000001DE" },
{ 0x000001E0, U"\U000001E0\U000001E1\U000001E0" },
{ 0x000001E1, U"\U000001E0\U000001E1\U000001E0" },
{ 0x000001E2, U"\U000001E2\U000001E3\U000001E2" },
{ 0x000001E3, U"\U000001E2\U000001E3\U000001E2" },
{ 0x000001E4, U"\U000001E4\U000001E5\U000001E4" },
{ 0x000001E5, U"\U000001E4\U000001E5\U000001E4" },
{ 0x000001E6, U"\U000001E6\U000001E7\U000001E6" },
{ 0x000001E7, U"\U000001E6\U000001E7\U000001E6" },
{ 0x000001E8, U"\U000001E8\U000001E9\U000001E8" },
{ 0x000001E9, U"\U000001E8\U000001E9\U000001E8" },
{ 0x000001EA, U"\U000001EA\U000001EB\U000001EA" },
{ 0x000001EB, U"\U000001EA\U000001EB\U000001EA" },
{ 0x000001EC, U"\U000001EC\U000001ED\U000001EC" },
{ 0x000001ED, U"\U000001EC\U000001ED\U000001EC" },
{ 0x000001EE, U"\U000001EE\U000001EF\U000001EE" },
{ 0x000001EF, U"\U000001EE\U000001EF\U000001EE" },
{ 0x000001F1, U"\U000001F1\U000001F3\U000001F2" },
{ 0x000001F2, U"\U000001F1\U000001F3\U000001F2" },
{ 0x000001F3, U"\U000001F1\U000001F3\U000001F2" },
{ 0x000001F4, U"\U000001F4\U000001F5\U000001F4" },
{ 0x000001F5, U"\U000001F4\U000001F5\U000001F4" },
{ 0x000001F6, U"\U000001F6\U00000195\U000001F6" },
{ 0x000001F7, U"\U000001F7\U000001BF\U000001F7" },
{ 0x000001F8, U"\U000001F8\U000001F9\U000001F8" },
{ 0x000001F9, U"\U000001F8\U000001F9\U000001F8" },
{ 0x000001FA, U"\U000001FA\U000001FB\U000001FA" },
{ 0x000001FB, U"\U000001FA\U000001FB\U000001FA" },
{ 0x000001FC, U"\U000001FC\U000001FD\U000001FC" },
{ 0x000001FD, U"\U000001FC\U000001FD\U000001FC" },
{ 0x000001FE, U"\U000001FE\U000001FF\U000001FE" },
{ 0x000001FF, U"\U000001FE\U000001FF\U000001FE" },
{ 0x00000200, U"\U00000200\U00000201\U00000200" },
{ 0x00000201, U"\U00000200\U00000201\U00000200" },
{ 0x00000202, U"\U00000202\U00000203\U00000202" },
{ 0x00000203, U"\U00000202\U00000203\U00000202" },
{ 0x00000204, U"\U00000204\U00000205\U00000204" },
{ 0x00000205, U"\U00000204\U00000205\U00000204" },
{ 0x00000206, U"\U00000206\U00000207\U00000206" },
{ 0x00000207, U"\U00000206\U00000207\U00000206" },
{ 0x00000208, U"\U00000208\U00000209\U00000208" },
{ 0x00000209, U"\U00000208\U00000209\U00000208" },
{ 0x0000020A, U"\U0000020A\U0000020B\U0000020A" },
{ 0x0000020B, U"\U0000020A\U0000020B\U0000020A" },
{ 0x0000020C, U"\U0000020C\U0000020D\U0000020C" },
{ 0x0000020D, U"\U0000020C\U0000020D\U0000020C" },
{ 0x0000020E, U"\U0000020E\U0000020F\U0000020E" },
{ 0x0000020F, U"\U0000020E\U0000020F\U0000020E" },
{ 0x00000210, U"\U00000210\U00000211\U00000210" },
{ 0x00000211, U"\U00000210\U00000211\U00000210" },
{ 0x00000212, U"\U00000212\U00000213\U00000212" },
{ 0x00000213, U"\U00000212\U00000213\U00000212" },
{ 0x00000214, U"\U00000214\U00000215\U00000214" },
{ 0x00000215, U"\U00000214\U00000215\U00000214" },
{ 0x00000216, U"\U00000216\U00000217\U00000216" },
{ 0x00000217, U"\U00000216\U00000217\U00000216" },
{ 0x00000218, U"\U00000218\U00000219\U00000218" },
{ 0x00000219, U"\U00000218\U00000219\U00000218" },
{ 0x0000021A, U"\U0000021A\U0000021B\U0000021A" },
{ 0x0000021B, U"\U0000021A\U0000021B\U0000021A" },
{ 0x0000021C, U"\U0000021C\U0000021D\U0000021C" },
{ 0x0000021D, U"\U0000021C\U0000021D\U0000021C" },
{ 0x0000021E, U"\U0000021E\U0000021F\U0000021E" },
{ 0x0000021F, U"\U0000021E\U0000021F\U0000021E" },
{ 0x00000220, U"\U00000220\U0000019E\U00000220" },
{ 0x00000222, U"\U00000222\U00000223\U00000222" },
{ 0x00000223, U"\U00000222\U00000223\U00000222" },
{ 0x00000224, U"\U00000224\U00000225\U00000224" },
{ 0x00000225, U"\U00000224\U00000225\U00000224" },
{ 0x00000226, U"\U00000226\U00000227\U00000226" },
{ 0x00000227, U"\U00000226\U00000227\U00000226" },
{ 0x00000228, U"\U00000228\U00000229\U00000228" },
{ 0x00000229, U"\U00000228\U00000229\U00000228" },
{ 0x0000022A, U"\U0000022A\U0000022B\U0000022A" },
{ 0x0000022B, U"\U0000022A\U0000022B\U0000022A" },
{ 0x0000022C, U"\U0000022C\U0000022D\U0000022C" },
{ 0x0000022D, U"\U0000022C\U0000022D\U0000022C" },
{ 0x0000022E, U"\U0000022E\U0000022F\U0000022E" },
{ 0x0000022F, U"\U0000022E\U0000022F\U0000022E" },
{ 0x00000230, U"\U00000230\U00000231\U00000230" },
{ 0x00000231, U"\U00000230\U00000231\U00000230" },
{ 0x00000232, U"\U00000232\U00000233\U00000232" },
{ 0x00000233, U"\U00000232\U00000233\U00000232" },
{ 0x0000023A, U"\U0000023A\U00002C65\U0000023A" },
{ 0x0000023B, U"\U0000023B\U0000023C\U0000023B" },
{ 0x0000023C, U"\U0000023B\U0000023C\U0000023B" },
{ 0x0000023D, U"\U0000023D\U0000019A\U0000023D" },
{ 0x0000023E, U"\U0000023E\U00002C66\U0000023E" },
{ 0x0000023F, U"\U00002C7E\U0000023F\U00002C7E" },
{ 0x00000240, U"\U00002C7F\U00000240\U00002C7F" },
{ 0x00000241, U"\U00000241\U00000242\U00000241" },
{ 0x00000242, U"\U00000241\U00000242\U00000241" },
{ 0x00000243, U"\U00000243\U00000180\U00000243" },
{ 0x00000244, U"\U00000244\U00000289\U00000244" },
{ 0x00000245, U"\U00000245\U0000028C\U00000245" },
{ 0x00000246, U"\U00000246\U00000247\U00000246" },
{ 0x00000247, U"\U00000246\U00000247\U00000246" },
{ 0x00000248, U"\U00000248\U00000249\U00000248" },
{ 0x00000249, U"\U00000248\U00000249\U00000248" },
{ 0x0000024A, U"\U0000024A\U0000024B\U0000024A" },
{ 0x0000024B, U"\U0000024A\U0000024B\U0000024A" },
{ 0x0000024C, U"\U0000024C\U0000024D\U0000024C" },
{ 0x0000024D, U"\U0000024C\U0000024D\U0000024C" },
{ 0x0000024E, U"\U0000024E\U0000024F\U0000024E" },
{ 0x0000024F, U"\U0000024E\U0000024F\U0000024E" },
{ 0x00000250, U"\U00002C6F\U00000250\U00002C6F" },
{ 0x00000251, U"\U00002C6D\U00000251\U00002C6D" },
{ 0x00000252, U"\U00002C70\U00000252\U00002C70" },
{ 0x00000253, U"\U00000181\U00000253\U00000181" },
{ 0x00000254, U"\U00000186\U00000254\U00000186" },
{ 0x00000256, U"\U00000189\U00000256\U00000189" },
{ 0x00000257, U"\U0000018A\U00000257\U0000018A" },
{ 0x00000259, U"\U0000018F\U00000259\U0000018F" },
{ 0x0000025B, U"\U00000190\U0000025B\U00000190" },
{ 0x0000025C, U"\U0000A7AB\U0000025C\U0000A7AB" },
{ 0x00000260, U"\U00000193\U00000260\U00000193" },
{ 0x00000261, U"\U0000A7AC\U00000261\U0000A7AC" },
{ 0x00000263, U"\U00000194\U00000263\U00000194" },
{ 0x00000265, U"\U0000A78D\U00000265\U0000A78D" },
{ 0x00000266, U"\U0000A7AA\U00000266\U0000A7AA" },
{ 0x00000268, U"\U00000197\U00000268\U00000197" },
{ 0x00000269, U"\U00000196\U00000269\U00000196" },
{ 0x0000026A, U"\U0000A7AE\U0000026A\U0000A7AE" },
{ 0x0000026B, U"\U00002C62\U0000026B\U00002C62" },
{ 0x0000026C, U"\U0000A7AD\U0000026C\U0000A7AD" },
{ 0x0000026F, U"\U0000019C\U0000026F\U0000019C" },
{ 0x00000271, U"\U00002C6E\U00000271\U00002C6E" },
{ 0x00000272, U"\U0000019D\U00000272\U0000019D" },
{ 0x00000275, U"\U0000019F\U00000275\U0000019F" },
{ 0x0000027D, U"\U00002C64\U0000027D\U00002C64" },
{ 0x00000280, U"\U000001A6\U00000280\U000001A6" },
{ 0x00000282, U"\U0000A7C5\U00000282\U0000A7C5" },
{ 0x00000283, U"\U000001A9\U00000283\U000001A9" },
{ 0x00000287, U"\U0000A7B1\U00000287\U0000A7B1" },
{ 0x00000288, U"\U000001AE\U00000288\U000001AE" },
{ 0x00000289, U"\U00000244\U00000289\U00000244" },
{ 0x0000028A, U"\U000001B1\U0000028A\U000001B1" },
{ 0x0000028B, U"\U000001B2\U0000028B\U000001B2" },
{ 0x0000028C, U"\U00000245\U0000028C\U00000245" },
{ 0x00000292, U"\U000001B7\U00000292\U000001B7" },
{ 0x0000029D, U"\U0000A7B2\U0000029D\U0000A7B2" },
{ 0x0000029E, U"\U0000A7B0\U0000029E\U0000A7B0" },
{ 0x00000345, U"\U00000399\U00000345\U00000399" },
{ 0x00000370, U"\U00000370\U00000371\U00000370" },
{ 0x00000371, U"\U00000370\U00000371\U00000370" },
{ 0x00000372, U"\U00000372\U00000373\U00000372" },
{ 0x00000373, U"\U00000372\U00000373\U00000372" },
{ 0x00000376, U"\U00000376\U00000377\U00000376" },
{ 0x00000377, U"\U00000376\U00000377\U00000376" },
{ 0x0000037B, U"\U000003FD\U0000037B\U000003FD" },
{ 0x0000037C, U"\U000003FE\U0000037C\U000003FE" },
{ 0x0000037D, U"\U000003FF\U0000037D\U000003FF" },
{ 0x0000037F, U"\U0000037F\U000003F3\U0000037F" },
{ 0x00000386, U"\U00000386\U000003AC\U00000386" },
{ 0x00000388, U"\U00000388\U000003AD\U00000388" },
{ 0x00000389, U"\U00000389\U000003AE\U00000389" },
{ 0x0000038A, U"\U0000038A\U000003AF\U0000038A" },
{ 0x0000038C, U"\U0000038C\U000003CC\U0000038C" },
{ 0x0000038E, U"\U0000038E\U000003CD\U0000038E" },
{ 0x0000038F, U"\U0000038F\U000003CE\U0000038F" },
{ 0x00000391, U"\U00000391\U000003B1\U00000391" },
{ 0x00000392, U"\U00000392\U000003B2\U00000392" },
{ 0x00000393, U"\U00000393\U000003B3\U00000393" },
{ 0x00000394, U"\U00000394\U000003B4\U00000394" },
{ 0x00000395, U"\U00000395\U000003B5\U00000395" },
{ 0x00000396, U"\U00000396\U000003B6\U00000396" },
{ 0x00000397, U"\U00000397\U000003B7\U00000397" },
{ 0x00000398, U"\U00000398\U000003B8\U00000398" },
{ 0x00000399, U"\U00000399\U000003B9\U00000399" },
{ 0x0000039A, U"\U0000039A\U000003BA\U0000039A" },
{ 0x0000039B, U"\U0000039B\U000003BB\U0000039B" },
{ 0x0000039C, U"\U0000039C\U000003BC\U0000039C" },
{ 0x0000039D, U"\U0000039D\U000003BD\U0000039D" },
{ 0x0000039E, U"\U0000039E\U000003BE\U0000039E" },
{ 0x0000039F, U"\U0000039F\U000003BF\U0000039F" },
{ 0x000003A0, U"\U000003A0\U000003C0\U000003A0" },
{ 0x000003A1, U"\U000003A1\U000003C1\U000003A1" },
{ 0x000003A3, U"\U000003A3\U000003C3\U000003A3" },
{ 0x000003A4, U"\U000003A4\U000003C4\U000003A4" },
{ 0x000003A5, U"\U000003A5\U000003C5\U000003A5" },
{ 0x000003A6, U"\U000003A6\U000003C6\U000003A6" },
{ 0x000003A7, U"\U000003A7\U000003C7\U000003A7" },
{ 0x000003A8, U"\U000003A8\U000003C8\U000003A8" },
{ 0x000003A9, U"\U000003A9\U000003C9\U000003A9" },
{ 0x000003AA, U"\U000003AA\U000003CA\U000003AA" },
{ 0x000003AB, U"\U000003AB\U000003CB\U000003AB" },
{ 0x000003AC, U"\U00000386\U000003AC\U00000386" },
{ 0x000003AD, U"\U00000388\U000003AD\U00000388" },
{ 0x000003AE, U"\U00000389\U000003AE\U00000389" },
{ 0x000003AF, U"\U0000038A\U000003AF\U0000038A" },
{ 0x000003B1, U"\U00000391\U000003B1\U00000391" },
{ 0x000003B2, U"\U00000392\U000003B2\U00000392" },
{ 0x000003B3, U"\U00000393\U000003B3\U00000393" },
{ 0x000003B4, U"\U00000394\U000003B4\U00000394" },
{ 0x000003B5, U"\U00000395\U000003B5\U00000395" },
{ 0x000003B6, U"\U00000396\U000003B6\U00000396" },
{ 0x000003B7, U"\U00000397\U000003B7\U00000397" },
{ 0x000003B8, U"\U00000398\U000003B8\U00000398" },
{ 0x000003B9, U"\U00000399\U000003B9\U00000399" },
{ 0x000003BA, U"\U0000039A\U000003BA\U0000039A" },
{ 0x000003BB, U"\U0000039B\U000003BB\U0000039B" },
{ 0x000003BC, U"\U0000039C\U000003BC\U0000039C" },
{ 0x000003BD, U"\U0000039D\U000003BD\U0000039D" },
{ 0x000003BE, U"\U0000039E\U000003BE\U0000039E" },
{ 0x000003BF, U"\U0000039F\U000003BF\U0000039F" },
{ 0x000003C0, U"\U000003A0\U000003C0\U000003A0" },
{ 0x000003C1, U"\U000003A1\U000003C1\U000003A1" },
{ 0x000003C2, U"\U000003A3\U000003C2\U000003A3" },
{ 0x000003C3, U"\U000003A3\U000003C3\U000003A3" },
{ 0x000003C4, U"\U000003A4\U000003C4\U000003A4" },
{ 0x000003C5, U"\U000003A5\U000003C5\U000003A5" },
{ 0x000003C6, U"\U000003A6\U000003C6\U000003A6" },
{ 0x000003C7, U"\U000003A7\U000003C7\U000003A7" },
{ 0x000003C8, U"\U000003A8\U000003C8\U000003A8" },
{ 0x000003C9, U"\U000003A9\U000003C9\U000003A9" },
{ 0x000003CA, U"\U000003AA\U000003CA\U000003AA" },
{ 0x000003CB, U"\U000003AB\U000003CB\U000003AB" },
{ 0x000003CC, U"\U0000038C\U000003CC\U0000038C" },
{ 0x000003CD, U"\U0000038E\U000003CD\U0000038E" },
{ 0x000003CE, U"\U0000038F\U000003CE\U0000038F" },
{ 0x000003CF, U"\U000003CF\U000003D7\U000003CF" },
{ 0x000003D0, U"\U00000392\U000003D0\U00000392" },
{ 0x000003D1, U"\U00000398\U000003D1\U00000398" },
{ 0x000003D5, U"\U000003A6\U000003D5\U000003A6" },
{ 0x000003D6, U"\U000003A0\U000003D6\U000003A0" },
{ 0x000003D7, U"\U000003CF\U000003D7\U000003CF" },
{ 0x000003D8, U"\U000003D8\U000003D9\U000003D8" },
{ 0x000003D9, U"\U000003D8\U000003D9\U000003D8" },
{ 0x000003DA, U"\U000003DA\U000003DB\U000003DA" },
{ 0x000003DB, U"\U000003DA\U000003DB\U000003DA" },
{ 0x000003DC, U"\U000003DC\U000003DD\U000003DC" },
{ 0x000003DD, U"\U000003DC\U000003DD\U000003DC" },
{ 0x000003DE, U"\U000003DE\U000003DF\U000003DE" },
{ 0x000003DF, U"\U000003DE\U000003DF\U000003DE" },
{ 0x000003E0, U"\U000003E0\U000003E1\U000003E0" },
{ 0x000003E1, U"\U000003E0\U000003E1\U000003E0" },
{ 0x000003E2, U"\U000003E2\U000003E3\U000003E2" },
{ 0x000003E3, U"\U000003E2\U000003E3\U000003E2" },
{ 0x000003E4, U"\U000003E4\U000003E5\U000003E4" },
{ 0x000003E5, U"\U000003E4\U000003E5\U000003E4" },
{ 0x000003E6, U"\U000003E6\U000003E7\U000003E6" },
{ 0x000003E7, U"\U000003E6\U000003E7\U000003E6" },
{ 0x000003E8, U"\U000003E8\U000003E9\U000003E8" },
{ 0x000003E9, U"\U000003E8\U000003E9\U000003E8" },
{ 0x000003EA, U"\U000003EA\U000003EB\U000003EA" },
{ 0x000003EB, U"\U000003EA\U000003EB\U000003EA" },
{ 0x000003EC, U"\U000003EC\U000003ED\U000003EC" },
{ 0x000003ED, U"\U000003EC\U000003ED\U000003EC" },
{ 0x000003EE, U"\U000003EE\U000003EF\U000003EE" },
{ 0x000003EF, U"\U000003EE\U000003EF\U000003EE" },
{ 0x000003F0, U"\U0000039A\U000003F0\U0000039A" },
{ 0x000003F1, U"\U000003A1\U000003F1\U000003A1" },
{ 0x000003F2, U"\U000003F9\U000003F2\U000003F9" },
{ 0x000003F3, U"\U0000037F\U000003F3\U0000037F" },
{ 0x000003F4, U"\U000003F4\U000003B8\U000003F4" },
{ 0x000003F5, U"\U00000395\U000003F5\U00000395" },
{ 0x000003F7, U"\U000003F7\U000003F8\U000003F7" },
{ 0x000003F8, U"\U000003F7\U000003F8\U000003F7" },
{ 0x000003F9, U"\U000003F9\U000003F2\U000003F9" },
{ 0x000003FA, U"\U000003FA\U000003FB\U000003FA" },
{ 0x000003FB, U"\U000003FA\U000003FB\U000003FA" },
{ 0x000003FD, U"\U000003FD\U0000037B\U000003FD" },
{ 0x000003FE, U"\U000003FE\U0000037C\U000003FE" },
{ 0x000003FF, U"\U000003FF\U0000037D\U000003FF" },
{ 0x00000400, U"\U00000400\U00000450\U00000400" },
{ 0x00000401, U"\U00000401\U00000451\U00000401" },
{ 0x00000402, U"\U00000402\U00000452\U00000402" },
{ 0x00000403, U"\U00000403\U00000453\U00000403" },
{ 0x00000404, U"\U00000404\U00000454\U00000404" },
{ 0x00000405, U"\U00000405\U00000455\U00000405" },
{ 0x00000406, U"\U00000406\U00000456\U00000406" },
{ 0x00000407, U"\U00000407\U00000457\U00000407" },
{ 0x00000408, U"\U00000408\U00000458\U00000408" },
{ 0x00000409, U"\U00000409\U00000459\U00000409" },
{ 0x0000040A, U"\U0000040A\U0000045A\U0000040A" },
{ 0x0000040B, U"\U0000040B\U0000045B\U0000040B" },
{ 0x0000040C, U"\U0000040C\U0000045C\U0000040C" },
{ 0x0000040D, U"\U0000040D\U0000045D\U0000040D" },
{ 0x0000040E, U"\U0000040E\U0000045E\U0000040E" },
{ 0x0000040F, U"\U0000040F\U0000045F\U0000040F" },
{ 0x00000410, U"\U00000410\U00000430\U00000410" },
{ 0x00000411, U"\U00000411\U00000431\U00000411" },
{ 0x00000412, U"\U00000412\U00000432\U00000412" },
{ 0x00000413, U"\U00000413\U00000433\U00000413" },
{ 0x00000414, U"\U00000414\U00000434\U00000414" },
{ 0x00000415, U"\U00000415\U00000435\U00000415" },
{ 0x00000416, U"\U00000416\U00000436\U00000416" },
{ 0x00000417, U"\U00000417\U00000437\U00000417" },
{ 0x00000418, U"\U00000418\U00000438\U00000418" },
{ 0x00000419, U"\U00000419\U00000439\U00000419" },
{ 0x0000041A, U"\U0000041A\U0000043A\U0000041A" },
{ 0x0000041B, U"\U0000041B\U0000043B\U0000041B" },
{ 0x0000041C, U"\U0000041C\U0000043C\U0000041C" },
{ 0x0000041D, U"\U0000041D\U0000043D\U0000041D" },
{ 0x0000041E, U"\U0000041E\U0000043E\U0000041E" },
{ 0x0000041F, U"\U0000041F\U0000043F\U0000041F" },
{ 0x00000420, U"\U00000420\U00000440\U00000420" },
{ 0x00000421, U"\U00000421\U00000441\U00000421" },
{ 0x00000422, U"\U00000422\U00000442\U00000422" },
{ 0x00000423, U"\U00000423\U00000443\U00000423" },
{ 0x00000424, U"\U00000424\U00000444\U00000424" },
{ 0x00000425, U"\U00000425\U00000445\U00000425" },
{ 0x00000426, U"\U00000426\U00000446\U00000426" },
{ 0x00000427, U"\U00000427\U00000447\U00000427" },
{ 0x00000428, U"\U00000428\U00000448\U00000428" },
{ 0x00000429, U"\U00000429\U00000449\U00000429" },
{ 0x0000042A, U"\U0000042A\U0000044A\U0000042A" },
{ 0x0000042B, U"\U0000042B\U0000044B\U0000042B" },
{ 0x0000042C, U"\U0000042C\U0000044C\U0000042C" },
{ 0x0000042D, U"\U0000042D\U0000044D\U0000042D" },
{ 0x0000042E, U"\U0000042E\U0000044E\U0000042E" },
{ 0x0000042F, U"\U0000042F\U0000044F\U0000042F" },
{ 0x00000430, U"\U00000410\U00000430\U00000410" },
{ 0x00000431, U"\U00000411\U00000431\U00000411" },
{ 0x00000432, U"\U00000412\U00000432\U00000412" },
{ 0x00000433, U"\U00000413\U00000433\U00000413" },
{ 0x00000434, U"\U00000414\U00000434\U00000414" },
{ 0x00000435, U"\U00000415\U00000435\U00000415" },
{ 0x00000436, U"\U00000416\U00000436\U00000416" },
{ 0x00000437, U"\U00000417\U00000437\U00000417" },
{ 0x00000438, U"\U00000418\U00000438\U00000418" },
{ 0x00000439, U"\U00000419\U00000439\U00000419" },
{ 0x0000043A, U"\U0000041A\U0000043A\U0000041A" },
{ 0x0000043B, U"\U0000041B\U0000043B\U0000041B" },
{ 0x0000043C, U"\U0000041C\U0000043C\U0000041C" },
{ 0x0000043D, U"\U0000041D\U0000043D\U0000041D" },
{ 0x0000043E, U"\U0000041E\U0000043E\U0000041E" },
{ 0x0000043F, U"\U0000041F\U0000043F\U0000041F" },
{ 0x00000440, U"\U00000420\U00000440\U00000420" },
{ 0x00000441, U"\U00000421\U00000441\U00000421" },
{ 0x00000442, U"\U00000422\U00000442\U00000422" },
{ 0x00000443, U"\U00000423\U00000443\U00000423" },
{ 0x00000444, U"\U00000424\U00000444\U00000424" },
{ 0x00000445, U"\U00000425\U00000445\U00000425" },
{ 0x00000446, U"\U00000426\U00000446\U00000426" },
{ 0x00000447, U"\U00000427\U00000447\U00000427" },
{ 0x00000448, U"\U00000428\U00000448\U00000428" },
{ 0x00000449, U"\U00000429\U00000449\U00000429" },
{ 0x0000044A, U"\U0000042A\U0000044A\U0000042A" },
{ 0x0000044B, U"\U0000042B\U0000044B\U0000042B" },
{ 0x0000044C, U"\U0000042C\U0000044C\U0000042C" },
{ 0x0000044D, U"\U0000042D\U0000044D\U0000042D" },
{ 0x0000044E, U"\U0000042E\U0000044E\U0000042E" },
{ 0x0000044F, U"\U0000042F\U0000044F\U0000042F" },
{ 0x00000450, U"\U00000400\U00000450\U00000400" },
{ 0x00000451, U"\U00000401\U00000451\U00000401" },
{ 0x00000452, U"\U00000402\U00000452\U00000402" },
{ 0x00000453, U"\U00000403\U00000453\U00000403" },
{ 0x00000454, U"\U00000404\U00000454\U00000404" },
{ 0x00000455, U"\U00000405\U00000455\U00000405" },
{ 0x00000456, U"\U00000406\U00000456\U00000406" },
{ 0x00000457, U"\U00000407\U00000457\U00000407" },
{ 0x00000458, U"\U00000408\U00000458\U00000408" },
{ 0x00000459, U"\U00000409\U00000459\U00000409" },
{ 0x0000045A, U"\U0000040A\U0000045A\U0000040A" },
{ 0x0000045B, U"\U0000040B\U0000045B\U0000040B" },
{ 0x0000045C, U"\U0000040C\U0000045C\U0000040C" },
{ 0x0000045D, U"\U0000040D\U0000045D\U0000040D" },
{ 0x0000045E, U"\U0000040E\U0000045E\U0000040E" },
{ 0x0000045F, U"\U0000040F\U0000045F\U0000040F" },
{ 0x00000460, U"\U00000460\U00000461\U00000460" },
{ 0x00000461, U"\U00000460\U00000461\U00000460" },
{ 0x00000462, U"\U00000462\U00000463\U00000462" },
{ 0x00000463, U"\U00000462\U00000463\U00000462" },
{ 0x00000464, U"\U00000464\U00000465\U00000464" },
{ 0x00000465, U"\U00000464\U00000465\U00000464" },
{ 0x00000466, U"\U00000466\U00000467\U00000466" },
{ 0x00000467, U"\U00000466\U00000467\U00000466" },
{ 0x00000468, U"\U00000468\U00000469\U00000468" },
{ 0x00000469, U"\U00000468\U00000469\U00000468" },
{ 0x0000046A, U"\U0000046A\U0000046B\U0000046A" },
{ 0x0000046B, U"\U0000046A\U0000046B\U0000046A" },
{ 0x0000046C, U"\U0000046C\U0000046D\U0000046C" },
{ 0x0000046D, U"\U0000046C\U0000046D\U0000046C" },
{ 0x0000046E, U"\U0000046E\U0000046F\U0000046E" },
{ 0x0000046F, U"\U0000046E\U0000046F\U0000046E" },
{ 0x00000470, U"\U00000470\U00000471\U00000470" },
{ 0x00000471, U"\U00000470\U00000471\U00000470" },
{ 0x00000472, U"\U00000472\U00000473\U00000472" },
{ 0x00000473, U"\U00000472\U00000473\U00000472" },
{ 0x00000474, U"\U00000474\U00000475\U00000474" },
{ 0x00000475, U"\U00000474\U00000475\U00000474" },
{ 0x00000476, U"\U00000476\U00000477\U00000476" },
{ 0x00000477, U"\U00000476\U00000477\U00000476" },
{ 0x00000478, U"\U00000478\U00000479\U00000478" },
{ 0x00000479, U"\U00000478\U00000479\U00000478" },
{ 0x0000047A, U"\U0000047A\U0000047B\U0000047A" },
{ 0x0000047B, U"\U0000047A\U0000047B\U0000047A" },
{ 0x0000047C, U"\U0000047C\U0000047D\U0000047C" },
{ 0x0000047D, U"\U0000047C\U0000047D\U0000047C" },
{ 0x0000047E, U"\U0000047E\U0000047F\U0000047E" },
{ 0x0000047F, U"\U0000047E\U0000047F\U0000047E" },
{ 0x00000480, U"\U00000480\U00000481\U00000480" },
{ 0x00000481, U"\U00000480\U00000481\U00000480" },
{ 0x0000048A, U"\U0000048A\U0000048B\U0000048A" },
{ 0x0000048B, U"\U0000048A\U0000048B\U0000048A" },
{ 0x0000048C, U"\U0000048C\U0000048D\U0000048C" },
{ 0x0000048D, U"\U0000048C\U0000048D\U0000048C" },
{ 0x0000048E, U"\U0000048E\U0000048F\U0000048E" },
{ 0x0000048F, U"\U0000048E\U0000048F\U0000048E" },
{ 0x00000490, U"\U00000490\U00000491\U00000490" },
{ 0x00000491, U"\U00000490\U00000491\U00000490" },
{ 0x00000492, U"\U00000492\U00000493\U00000492" },
{ 0x00000493, U"\U00000492\U00000493\U00000492" },
{ 0x00000494, U"\U00000494\U00000495\U00000494" },
{ 0x00000495, U"\U00000494\U00000495\U00000494" },
{ 0x00000496, U"\U00000496\U00000497\U00000496" },
{ 0x00000497, U"\U00000496\U00000497\U00000496" },
{ 0x00000498, U"\U00000498\U00000499\U00000498" },
{ 0x00000499, U"\U00000498\U00000499\U00000498" },
{ 0x0000049A, U"\U0000049A\U0000049B\U0000049A" },
{ 0x0000049B, U"\U0000049A\U0000049B\U0000049A" },
{ 0x0000049C, U"\U0000049C\U0000049D\U0000049C" },
{ 0x0000049D, U"\U0000049C\U0000049D\U0000049C" },
{ 0x0000049E, U"\U0000049E\U0000049F\U0000049E" },
{ 0x0000049F, U"\U0000049E\U0000049F\U0000049E" },
{ 0x000004A0, U"\U000004A0\U000004A1\U000004A0" },
{ 0x000004A1, U"\U000004A0\U000004A1\U000004A0" },
{ 0x000004A2, U"\U000004A2\U000004A3\U000004A2" },
{ 0x000004A3, U"\U000004A2\U000004A3\U000004A2" },
{ 0x000004A4, U"\U000004A4\U000004A5\U000004A4" },
{ 0x000004A5, U"\U000004A4\U000004A5\U000004A4" },
{ 0x000004A6, U"\U000004A6\U000004A7\U000004A6" },
{ 0x000004A7, U"\U000004A6\U000004A7\U000004A6" },
{ 0x000004A8, U"\U000004A8\U000004A9\U000004A8" },
{ 0x000004A9, U"\U000004A8\U000004A9\U000004A8" },
{ 0x000004AA, U"\U000004AA\U000004AB\U000004AA" },
{ 0x000004AB, U"\U000004AA\U000004AB\U000004AA" },
{ 0x000004AC, U"\U000004AC\U000004AD\U000004AC" },
{ 0x000004AD, U"\U000004AC\U000004AD\U000004AC" },
{ 0x000004AE, U"\U000004AE\U000004AF\U000004AE" },
{ 0x000004AF, U"\U000004AE\U000004AF\U000004AE" },
{ 0x000004B0, U"\U000004B0\U000004B1\U000004B0" },
{ 0x000004B1, U"\U000004B0\U000004B1\U000004B0" },
{ 0x000004B2, U"\U000004B2\U000004B3\U000004B2" },
{ 0x000004B3, U"\U000004B2\U000004B3\U000004B2" },
{ 0x000004B4, U"\U000004B4\U000004B5\U000004B4" },
{ 0x000004B5, U"\U000004B4\U000004B5\U000004B4" },
{ 0x000004B6, U"\U000004B6\U000004B7\U000004B6" },
{ 0x000004B7, U"\U000004B6\U000004B7\U000004B6" },
{ 0x000004B8, U"\U000004B8\U000004B9\U000004B8" },
{ 0x000004B9, U"\U000004B8\U000004B9\U000004B8" },
{ 0x000004BA, U"\U000004BA\U000004BB\U000004BA" },
{ 0x000004BB, U"\U000004BA\U000004BB\U000004BA" },
{ 0x000004BC, U"\U000004BC\U000004BD\U000004BC" },
{ 0x000004BD, U"\U000004BC\U000004BD\U000004BC" },
{ 0x000004BE, U"\U000004BE\U000004BF\U000004BE" },
{ 0x000004BF, U"\U000004BE\U000004BF\U000004BE" },
{ 0x000004C0, U"\U000004C0\U000004CF\U000004C0" },
{ 0x000004C1, U"\U000004C1\U000004C2\U000004C1" },
{ 0x000004C2, U"\U000004C1\U000004C2\U000004C1" },
{ 0x000004C3, U"\U000004C3\U000004C4\U000004C3" },
{ 0x000004C4, U"\U000004C3\U000004C4\U000004C3" },
{ 0x000004C5, U"\U000004C5\U000004C6\U000004C5" },
{ 0x000004C6, U"\U000004C5\U000004C6\U000004C5" },
{ 0x000004C7, U"\U000004C7\U000004C8\U000004C7" },
{ 0x000004C8, U"\U000004C7\U000004C8\U000004C7" },
{ 0x000004C9, U"\U000004C9\U000004CA\U000004C9" },
{ 0x000004CA, U"\U000004C9\U000004CA\U000004C9" },
{ 0x000004CB, U"\U000004CB\U000004CC\U000004CB" },
{ 0x000004CC, U"\U000004CB\U000004CC\U000004CB" },
{ 0x000004CD, U"\U000004CD\U000004CE\U000004CD" },
{ 0x000004CE, U"\U000004CD\U000004CE\U000004CD" },
{ 0x000004CF, U"\U000004C0\U000004CF\U000004C0" },
{ 0x000004D0, U"\U000004D0\U000004D1\U000004D0" },
{ 0x000004D1, U"\U000004D0\U000004D1\U000004D0" },
{ 0x000004D2, U"\U000004D2\U000004D3\U000004D2" },
{ 0x000004D3, U"\U000004D2\U000004D3\U000004D2" },
{ 0x000004D4, U"\U000004D4\U000004D5\U000004D4" },
{ 0x000004D5, U"\U000004D4\U000004D5\U000004D4" },
{ 0x000004D6, U"\U000004D6\U000004D7\U000004D6" },
{ 0x000004D7, U"\U000004D6\U000004D7\U000004D6" },
{ 0x000004D8, U"\U000004D8\U000004D9\U000004D8" },
{ 0x000004D9, U"\U000004D8\U000004D9\U000004D8" },
{ 0x000004DA, U"\U000004DA\U000004DB\U000004DA" },
{ 0x000004DB, U"\U000004DA\U000004DB\U000004DA" },
{ 0x000004DC, U"\U000004DC\U000004DD\U000004DC" },
{ 0x000004DD, U"\U000004DC\U000004DD\U000004DC" },
{ 0x000004DE, U"\U000004DE\U000004DF\U000004DE" },
{ 0x000004DF, U"\U000004DE\U000004DF\U000004DE" },
{ 0x000004E0, U"\U000004E0\U000004E1\U000004E0" },
{ 0x000004E1, U"\U000004E0\U000004E1\U000004E0" },
{ 0x000004E2, U"\U000004E2\U000004E3\U000004E2" },
{ 0x000004E3, U"\U000004E2\U000004E3\U000004E2" },
{ 0x000004E4, U"\U000004E4\U000004E5\U000004E4" },
{ 0x000004E5, U"\U000004E4\U000004E5\U000004E4" },
{ 0x000004E6, U"\U000004E6\U000004E7\U000004E6" },
{ 0x000004E7, U"\U000004E6\U000004E7\U000004E6" },
{ 0x000004E8, U"\U000004E8\U000004E9\U000004E8" },
{ 0x000004E9, U"\U000004E8\U000004E9\U000004E8" },
{ 0x000004EA, U"\U000004EA\U000004EB\U000004EA" },
{ 0x000004EB, U"\U000004EA\U000004EB\U000004EA" },
{ 0x000004EC, U"\U000004EC\U000004ED\U000004EC" },
{ 0x000004ED, U"\U000004EC\U000004ED\U000004EC" },
{ 0x000004EE, U"\U000004EE\U000004EF\U000004EE" },
{ 0x000004EF, U"\U000004EE\U000004EF\U000004EE" },
{ 0x000004F0, U"\U000004F0\U000004F1\U000004F0" },
{ 0x000004F1, U"\U000004F0\U000004F1\U000004F0" },
{ 0x000004F2, U"\U000004F2\U000004F3\U000004F2" },
{ 0x000004F3, U"\U000004F2\U000004F3\U000004F2" },
{ 0x000004F4, U"\U000004F4\U000004F5\U000004F4" },
{ 0x000004F5, U"\U000004F4\U000004F5\U000004F4" },
{ 0x000004F6, U"\U000004F6\U000004F7\U000004F6" },
{ 0x000004F7, U"\U000004F6\U000004F7\U000004F6" },
{ 0x000004F8, U"\U000004F8\U000004F9\U000004F8" },
{ 0x000004F9, U"\U000004F8\U000004F9\U000004F8" },
{ 0x000004FA, U"\U000004FA\U000004FB\U000004FA" },
{ 0x000004FB, U"\U000004FA\U000004FB\U000004FA" },
{ 0x000004FC, U"\U000004FC\U000004FD\U000004FC" },
{ 0x000004FD, U"\U000004FC\U000004FD\U000004FC" },
{ 0x000004FE, U"\U000004FE\U000004FF\U000004FE" },
{ 0x000004FF, U"\U000004FE\U000004FF\U000004FE" },
{ 0x00000500, U"\U00000500\U00000501\U00000500" },
{ 0x00000501, U"\U00000500\U00000501\U00000500" },
{ 0x00000502, U"\U00000502\U00000503\U00000502" },
{ 0x00000503, U"\U00000502\U00000503\U00000502" },
{ 0x00000504, U"\U00000504\U00000505\U00000504" },
{ 0x00000505, U"\U00000504\U00000505\U00000504" },
{ 0x00000506, U"\U00000506\U00000507\U00000506" },
{ 0x00000507, U"\U00000506\U00000507\U00000506" },
{ 0x00000508, U"\U00000508\U00000509\U00000508" },
{ 0x00000509, U"\U00000508\U00000509\U00000508" },
{ 0x0000050A, U"\U0000050A\U0000050B\U0000050A" },
{ 0x0000050B, U"\U0000050A\U0000050B\U0000050A" },
{ 0x0000050C, U"\U0000050C\U0000050D\U0000050C" },
{ 0x0000050D, U"\U0000050C\U0000050D\U0000050C" },
{ 0x0000050E, U"\U0000050E\U0000050F\U0000050E" },
{ 0x0000050F, U"\U0000050E\U0000050F\U0000050E" },
{ 0x00000510, U"\U00000510\U00000511\U00000510" },
{ 0x00000511, U"\U00000510\U00000511\U00000510" },
{ 0x00000512, U"\U00000512\U00000513\U00000512" },
{ 0x00000513, U"\U00000512\U00000513\U00000512" },
{ 0x00000514, U"\U00000514\U00000515\U00000514" },
{ 0x00000515, U"\U00000514\U00000515\U00000514" },
{ 0x00000516, U"\U00000516\U00000517\U00000516" },
{ 0x00000517, U"\U00000516\U00000517\U00000516" },
{ 0x00000518, U"\U00000518\U00000519\U00000518" },
{ 0x00000519, U"\U00000518\U00000519\U00000518" },
{ 0x0000051A, U"\U0000051A\U0000051B\U0000051A" },
{ 0x0000051B, U"\U0000051A\U0000051B\U0000051A" },
{ 0x0000051C, U"\U0000051C\U0000051D\U0000051C" },
{ 0x0000051D, U"\U0000051C\U0000051D\U0000051C" },
{ 0x0000051E, U"\U0000051E\U0000051F\U0000051E" },
{ 0x0000051F, U"\U0000051E\U0000051F\U0000051E" },
{ 0x00000520, U"\U00000520\U00000521\U00000520" },
{ 0x00000521, U"\U00000520\U00000521\U00000520" },
{ 0x00000522, U"\U00000522\U00000523\U00000522" },
{ 0x00000523, U"\U00000522\U00000523\U00000522" },
{ 0x00000524, U"\U00000524\U00000525\U00000524" },
{ 0x00000525, U"\U00000524\U00000525\U00000524" },
{ 0x00000526, U"\U00000526\U00000527\U00000526" },
{ 0x00000527, U"\U00000526\U00000527\U00000526" },
{ 0x00000528, U"\U00000528\U00000529\U00000528" },
{ 0x00000529, U"\U00000528\U00000529\U00000528" },
{ 0x0000052A, U"\U0000052A\U0000052B\U0000052A" },
{ 0x0000052B, U"\U0000052A\U0000052B\U0000052A" },
{ 0x0000052C, U"\U0000052C\U0000052D\U0000052C" },
{ 0x0000052D, U"\U0000052C\U0000052D\U0000052C" },
{ 0x0000052E, U"\U0000052E\U0000052F\U0000052E" },
{ 0x0000052F, U"\U0000052E\U0000052F\U0000052E" },
{ 0x00000531, U"\U00000531\U00000561\U00000531" },
{ 0x00000532, U"\U00000532\U00000562\U00000532" },
{ 0x00000533, U"\U00000533\U00000563\U00000533" },
{ 0x00000534, U"\U00000534\U00000564\U00000534" },
{ 0x00000535, U"\U00000535\U00000565\U00000535" },
{ 0x00000536, U"\U00000536\U00000566\U00000536" },
{ 0x00000537, U"\U00000537\U00000567\U00000537" },
{ 0x00000538, U"\U00000538\U00000568\U00000538" },
{ 0x00000539, U"\U00000539\U00000569\U00000539" },
{ 0x0000053A, U"\U0000053A\U0000056A\U0000053A" },
{ 0x0000053B, U"\U0000053B\U0000056B\U0000053B" },
{ 0x0000053C, U"\U0000053C\U0000056C\U0000053C" },
{ 0x0000053D, U"\U0000053D\U0000056D\U0000053D" },
{ 0x0000053E, U"\U0000053E\U0000056E\U0000053E" },
{ 0x0000053F, U"\U0000053F\U0000056F\U0000053F" },
{ 0x00000540, U"\U00000540\U00000570\U00000540" },
{ 0x00000541, U"\U00000541\U00000571\U00000541" },
{ 0x00000542, U"\U00000542\U00000572\U00000542" },
{ 0x00000543, U"\U00000543\U00000573\U00000543" },
{ 0x00000544, U"\U00000544\U00000574\U00000544" },
{ 0x00000545, U"\U00000545\U00000575\U00000545" },
{ 0x00000546, U"\U00000546\U00000576\U00000546" },
{ 0x00000547, U"\U00000547\U00000577\U00000547" },
{ 0x00000548, U"\U00000548\U00000578\U00000548" },
{ 0x00000549, U"\U00000549\U00000579\U00000549" },
{ 0x0000054A, U"\U0000054A\U0000057A\U0000054A" },
{ 0x0000054B, U"\U0000054B\U0000057B\U0000054B" },
{ 0x0000054C, U"\U0000054C\U0000057C\U0000054C" },
{ 0x0000054D, U"\U0000054D\U0000057D\U0000054D" },
{ 0x0000054E, U"\U0000054E\U0000057E\U0000054E" },
{ 0x0000054F, U"\U0000054F\U0000057F\U0000054F" },
{ 0x00000550, U"\U00000550\U00000580\U00000550" },
{ 0x00000551, U"\U00000551\U00000581\U00000551" },
{ 0x00000552, U"\U00000552\U00000582\U00000552" },
{ 0x00000553, U"\U00000553\U00000583\U00000553" },
{ 0x00000554, U"\U00000554\U00000584\U00000554" },
{ 0x00000555, U"\U00000555\U00000585\U00000555" },
{ 0x00000556, U"\U00000556\U00000586\U00000556" },
{ 0x00000561, U"\U00000531\U00000561\U00000531" },
{ 0x00000562, U"\U00000532\U00000562\U00000532" },
{ 0x00000563, U"\U00000533\U00000563\U00000533" },
{ 0x00000564, U"\U00000534\U00000564\U00000534" },
{ 0x00000565, U"\U00000535\U00000565\U00000535" },
{ 0x00000566, U"\U00000536\U00000566\U00000536" },
{ 0x00000567, U"\U00000537\U00000567\U00000537" },
{ 0x00000568, U"\U00000538\U00000568\U00000538" },
{ 0x00000569, U"\U00000539\U00000569\U00000539" },
{ 0x0000056A, U"\U0000053A\U0000056A\U0000053A" },
{ 0x0000056B, U"\U0000053B\U0000056B\U0000053B" },
{ 0x0000056C, U"\U0000053C\U0000056C\U0000053C" },
{ 0x0000056D, U"\U0000053D\U0000056D\U0000053D" },
{ 0x0000056E, U"\U0000053E\U0000056E\U0000053E" },
{ 0x0000056F, U"\U0000053F\U0000056F\U0000053F" },
{ 0x00000570, U"\U00000540\U00000570\U00000540" },
{ 0x00000571, U"\U00000541\U00000571\U00000541" },
{ 0x00000572, U"\U00000542\U00000572\U00000542" },
{ 0x00000573, U"\U00000543\U00000573\U00000543" },
{ 0x00000574, U"\U00000544\U00000574\U00000544" },
{ 0x00000575, U"\U00000545\U00000575\U00000545" },
{ 0x00000576, U"\U00000546\U00000576\U00000546" },
{ 0x00000577, U"\U00000547\U00000577\U00000547" },
{ 0x00000578, U"\U00000548\U00000578\U00000548" },
{ 0x00000579, U"\U00000549\U00000579\U00000549" },
{ 0x0000057A, U"\U0000054A\U0000057A\U0000054A" },
{ 0x0000057B, U"\U0000054B\U0000057B\U0000054B" },
{ 0x0000057C, U"\U0000054C\U0000057C\U0000054C" },
{ 0x0000057D, U"\U0000054D\U0000057D\U0000054D" },
{ 0x0000057E, U"\U0000054E\U0000057E\U0000054E" },
{ 0x0000057F, U"\U0000054F\U0000057F\U0000054F" },
{ 0x00000580, U"\U00000550\U00000580\U00000550" },
{ 0x00000581, U"\U00000551\U00000581\U00000551" },
{ 0x00000582, U"\U00000552\U00000582\U00000552" },
{ 0x00000583, U"\U00000553\U00000583\U00000553" },
{ 0x00000584, U"\U00000554\U00000584\U00000554" },
{ 0x00000585, U"\U00000555\U00000585\U00000555" },
{ 0x00000586, U"\U00000556\U00000586\U00000556" },
{ 0x000010A0, U"\U000010A0\U00002D00\U000010A0" },
{ 0x000010A1, U"\U000010A1\U00002D01\U000010A1" },
{ 0x000010A2, U"\U000010A2\U00002D02\U000010A2" },
{ 0x000010A3, U"\U000010A3\U00002D03\U000010A3" },
{ 0x000010A4, U"\U000010A4\U00002D04\U000010A4" },
{ 0x000010A5, U"\U000010A5\U00002D05\U000010A5" },
{ 0x000010A6, U"\U000010A6\U00002D06\U000010A6" },
{ 0x000010A7, U"\U000010A7\U00002D07\U000010A7" },
{ 0x000010A8, U"\U000010A8\U00002D08\U000010A8" },
{ 0x000010A9, U"\U000010A9\U00002D09\U000010A9" },
{ 0x000010AA, U"\U000010AA\U00002D0A\U000010AA" },
{ 0x000010AB, U"\U000010AB\U00002D0B\U000010AB" },
{ 0x000010AC, U"\U000010AC\U00002D0C\U000010AC" },
{ 0x000010AD, U"\U000010AD\U00002D0D\U000010AD" },
{ 0x000010AE, U"\U000010AE\U00002D0E\U000010AE" },
{ 0x000010AF, U"\U000010AF\U00002D0F\U000010AF" },
{ 0x000010B0, U"\U000010B0\U00002D10\U000010B0" },
{ 0x000010B1, U"\U000010B1\U00002D11\U000010B1" },
{ 0x000010B2, U"\U000010B2\U00002D12\U000010B2" },
{ 0x000010B3, U"\U000010B3\U00002D13\U000010B3" },
{ 0x000010B4, U"\U000010B4\U00002D14\U000010B4" },
{ 0x000010B5, U"\U000010B5\U00002D15\U000010B5" },
{ 0x000010B6, U"\U000010B6\U00002D16\U000010B6" },
{ 0x000010B7, U"\U000010B7\U00002D17\U000010B7" },
{ 0x000010B8, U"\U000010B8\U00002D18\U000010B8" },
{ 0x000010B9, U"\U000010B9\U00002D19\U000010B9" },
{ 0x000010BA, U"\U000010BA\U00002D1A\U000010BA" },
{ 0x000010BB, U"\U000010BB\U00002D1B\U000010BB" },
{ 0x000010BC, U"\U000010BC\U00002D1C\U000010BC" },
{ 0x000010BD, U"\U000010BD\U00002D1D\U000010BD" },
{ 0x000010BE, U"\U000010BE\U00002D1E\U000010BE" },
{ 0x000010BF, U"\U000010BF\U00002D1F\U000010BF" },
{ 0x000010C0, U"\U000010C0\U00002D20\U000010C0" },
{ 0x000010C1, U"\U000010C1\U00002D21\U000010C1" },
{ 0x000010C2, U"\U000010C2\U00002D22\U000010C2" },
{ 0x000010C3, U"\U000010C3\U00002D23\U000010C3" },
{ 0x000010C4, U"\U000010C4\U00002D24\U000010C4" },
{ 0x000010C5, U"\U000010C5\U00002D25\U000010C5" },
{ 0x000010C7, U"\U000010C7\U00002D27\U000010C7" },
{ 0x000010CD, U"\U000010CD\U00002D2D\U000010CD" },
{ 0x000010D0, U"\U00001C90\U000010D0\U000010D0" },
{ 0x000010D1, U"\U00001C91\U000010D1\U000010D1" },
{ 0x000010D2, U"\U00001C92\U000010D2\U000010D2" },
{ 0x000010D3, U"\U00001C93\U000010D3\U000010D3" },
{ 0x000010D4, U"\U00001C94\U000010D4\U000010D4" },
{ 0x000010D5, U"\U00001C95\U000010D5\U000010D5" },
{ 0x000010D6, U"\U00001C96\U000010D6\U000010D6" },
{ 0x000010D7, U"\U00001C97\U000010D7\U000010D7" },
{ 0x000010D8, U"\U00001C98\U000010D8\U000010D8" },
{ 0x000010D9, U"\U00001C99\U000010D9\U000010D9" },
{ 0x000010DA, U"\U00001C9A\U000010DA\U000010DA" },
{ 0x000010DB, U"\U00001C9B\U000010DB\U000010DB" },
{ 0x000010DC, U"\U00001C9C\U000010DC\U000010DC" },
{ 0x000010DD, U"\U00001C9D\U000010DD\U000010DD" },
{ 0x000010DE, U"\U00001C9E\U000010DE\U000010DE" },
{ 0x000010DF, U"\U00001C9F\U000010DF\U000010DF" },
{ 0x000010E0, U"\U00001CA0\U000010E0\U000010E0" },
{ 0x000010E1, U"\U00001CA1\U000010E1\U000010E1" },
{ 0x000010E2, U"\U00001CA2\U000010E2\U000010E2" },
{ 0x000010E3, U"\U00001CA3\U000010E3\U000010E3" },
{ 0x000010E4, U"\U00001CA4\U000010E4\U000010E4" },
{ 0x000010E5, U"\U00001CA5\U000010E5\U000010E5" },
{ 0x000010E6, U"\U00001CA6\U000010E6\U000010E6" },
{ 0x000010E7, U"\U00001CA7\U000010E7\U000010E7" },
{ 0x000010E8, U"\U00001CA8\U000010E8\U000010E8" },
{ 0x000010E9, U"\U00001CA9\U000010E9\U000010E9" },
{ 0x000010EA, U"\U00001CAA\U000010EA\U000010EA" },
{ 0x000010EB, U"\U00001CAB\U000010EB\U000010EB" },
{ 0x000010EC, U"\U00001CAC\U000010EC\U000010EC" },
{ 0x000010ED, U"\U00001CAD\U000010ED\U000010ED" },
{ 0x000010EE, U"\U00001CAE\U000010EE\U000010EE" },
{ 0x000010EF, U"\U00001CAF\U000010EF\U000010EF" },
{ 0x000010F0, U"\U00001CB0\U000010F0\U000010F0" },
{ 0x000010F1, U"\U00001CB1\U000010F1\U000010F1" },
{ 0x000010F2, U"\U00001CB2\U000010F2\U000010F2" },
{ 0x000010F3, U"\U00001CB3\U000010F3\U000010F3" },
{ 0x000010F4, U"\U00001CB4\U000010F4\U000010F4" },
{ 0x000010F5, U"\U00001CB5\U000010F5\U000010F5" },
{ 0x000010F6, U"\U00001CB6\U000010F6\U000010F6" },
{ 0x000010F7, U"\U00001CB7\U000010F7\U000010F7" },
{ 0x000010F8, U"\U00001CB8\U000010F8\U000010F8" },
{ 0x000010F9, U"\U00001CB9\U000010F9\U000010F9" },
{ 0x000010FA, U"\U00001CBA\U000010FA\U000010FA" },
{ 0x000010FD, U"\U00001CBD\U000010FD\U000010FD" },
{ 0x000010FE, U"\U00001CBE\U000010FE\U000010FE" },
{ 0x000010FF, U"\U00001CBF\U000010FF\U000010FF" },
{ 0x000013A0, U"\U000013A0\U0000AB70\U000013A0" },
{ 0x000013A1, U"\U000013A1\U0000AB71\U000013A1" },
{ 0x000013A2, U"\U000013A2\U0000AB72\U000013A2" },
{ 0x000013A3, U"\U000013A3\U0000AB73\U000013A3" },
{ 0x000013A4, U"\U000013A4\U0000AB74\U000013A4" },
{ 0x000013A5, U"\U000013A5\U0000AB75\U000013A5" },
{ 0x000013A6, U"\U000013A6\U0000AB76\U000013A6" },
{ 0x000013A7, U"\U000013A7\U0000AB77\U000013A7" },
{ 0x000013A8, U"\U000013A8\U0000AB78\U000013A8" },
{ 0x000013A9, U"\U000013A9\U0000AB79\U000013A9" },
{ 0x000013AA, U"\U000013AA\U0000AB7A\U000013AA" },
{ 0x000013AB, U"\U000013AB\U0000AB7B\U000013AB" },
{ 0x000013AC, U"\U000013AC\U0000AB7C\U000013AC" },
{ 0x000013AD, U"\U000013AD\U0000AB7D\U000013AD" },
{ 0x000013AE, U"\U000013AE\U0000AB7E\U000013AE" },
{ 0x000013AF, U"\U000013AF\U0000AB7F\U000013AF" },
{ 0x000013B0, U"\U000013B0\U0000AB80\U000013B0" },
{ 0x000013B1, U"\U000013B1\U0000AB81\U000013B1" },
{ 0x000013B2, U"\U000013B2\U0000AB82\U000013B2" },
{ 0x000013B3, U"\U000013B3\U0000AB83\U000013B3" },
{ 0x000013B4, U"\U000013B4\U0000AB84\U000013B4" },
{ 0x000013B5, U"\U000013B5\U0000AB85\U000013B5" },
{ 0x000013B6, U"\U000013B6\U0000AB86\U000013B6" },
{ 0x000013B7, U"\U000013B7\U0000AB87\U000013B7" },
{ 0x000013B8, U"\U000013B8\U0000AB88\U000013B8" },
{ 0x000013B9, U"\U000013B9\U0000AB89\U000013B9" },
{ 0x000013BA, U"\U000013BA\U0000AB8A\U000013BA" },
{ 0x000013BB, U"\U000013BB\U0000AB8B\U000013BB" },
{ 0x000013BC, U"\U000013BC\U0000AB8C\U000013BC" },
{ 0x000013BD, U"\U000013BD\U0000AB8D\U000013BD" },
{ 0x000013BE, U"\U000013BE\U0000AB8E\U000013BE" },
{ 0x000013BF, U"\U000013BF\U0000AB8F\U000013BF" },
{ 0x000013C0, U"\U000013C0\U0000AB90\U000013C0" },
{ 0x000013C1, U"\U000013C1\U0000AB91\U000013C1" },
{ 0x000013C2, U"\U000013C2\U0000AB92\U000013C2" },
{ 0x000013C3, U"\U000013C3\U0000AB93\U000013C3" },
{ 0x000013C4, U"\U000013C4\U0000AB94\U000013C4" },
{ 0x000013C5, U"\U000013C5\U0000AB95\U000013C5" },
{ 0x000013C6, U"\U000013C6\U0000AB96\U000013C6" },
{ 0x000013C7, U"\U000013C7\U0000AB97\U000013C7" },
{ 0x000013C8, U"\U000013C8\U0000AB98\U000013C8" },
{ 0x000013C9, U"\U000013C9\U0000AB99\U000013C9" },
{ 0x000013CA, U"\U000013CA\U0000AB9A\U000013CA" },
{ 0x000013CB, U"\U000013CB\U0000AB9B\U000013CB" },
{ 0x000013CC, U"\U000013CC\U0000AB9C\U000013CC" },
{ 0x000013CD, U"\U000013CD\U0000AB9D\U000013CD" },
{ 0x000013CE, U"\U000013CE\U0000AB9E\U000013CE" },
{ 0x000013CF, U"\U000013CF\U0000AB9F\U000013CF" },
{ 0x000013D0, U"\U000013D0\U0000ABA0\U000013D0" },
{ 0x000013D1, U"\U000013D1\U0000ABA1\U000013D1" },
{ 0x000013D2, U"\U000013D2\U0000ABA2\U000013D2" },
{ 0x000013D3, U"\U000013D3\U0000ABA3\U000013D3" },
{ 0x000013D4, U"\U000013D4\U0000ABA4\U000013D4" },
{ 0x000013D5, U"\U000013D5\U0000ABA5\U000013D5" },
{ 0x000013D6, U"\U000013D6\U0000ABA6\U000013D6" },
{ 0x000013D7, U"\U000013D7\U0000ABA7\U000013D7" },
{ 0x000013D8, U"\U000013D8\U0000ABA8\U000013D8" },
{ 0x000013D9, U"\U000013D9\U0000ABA9\U000013D9" },
{ 0x000013DA, U"\U000013DA\U0000ABAA\U000013DA" },
{ 0x000013DB, U"\U000013DB\U0000ABAB\U000013DB" },
{ 0x000013DC, U"\U000013DC\U0000ABAC\U000013DC" },
{ 0x000013DD, U"\U000013DD\U0000ABAD\U000013DD" },
{ 0x000013DE, U"\U000013DE\U0000ABAE\U000013DE" },
{ 0x000013DF, U"\U000013DF\U0000ABAF\U000013DF" },
{ 0x000013E0, U"\U000013E0\U0000ABB0\U000013E0" },
{ 0x000013E1, U"\U000013E1\U0000ABB1\U000013E1" },
{ 0x000013E2, U"\U000013E2\U0000ABB2\U000013E2" },
{ 0x000013E3, U"\U000013E3\U0000ABB3\U000013E3" },
{ 0x000013E4, U"\U000013E4\U0000ABB4\U000013E4" },
{ 0x000013E5, U"\U000013E5\U0000ABB5\U000013E5" },
{ 0x000013E6, U"\U000013E6\U0000ABB6\U000013E6" },
{ 0x000013E7, U"\U000013E7\U0000ABB7\U000013E7" },
{ 0x000013E8, U"\U000013E8\U0000ABB8\U000013E8" },
{ 0x000013E9, U"\U000013E9\U0000ABB9\U000013E9" },
{ 0x000013EA, U"\U000013EA\U0000ABBA\U000013EA" },
{ 0x000013EB, U"\U000013EB\U0000ABBB\U000013EB" },
{ 0x000013EC, U"\U000013EC\U0000ABBC\U000013EC" },
{ 0x000013ED, U"\U000013ED\U0000ABBD\U000013ED" },
{ 0x000013EE, U"\U000013EE\U0000ABBE\U000013EE" },
{ 0x000013EF, U"\U000013EF\U0000ABBF\U000013EF" },
{ 0x000013F0, U"\U000013F0\U000013F8\U000013F0" },
{ 0x000013F1, U"\U000013F1\U000013F9\U000013F1" },
{ 0x000013F2, U"\U000013F2\U000013FA\U000013F2" },
{ 0x000013F3, U"\U000013F3\U000013FB\U000013F3" },
{ 0x000013F4, U"\U000013F4\U000013FC\U000013F4" },
{ 0x000013F5, U"\U000013F5\U000013FD\U000013F5" },
{ 0x000013F8, U"\U000013F0\U000013F8\U000013F0" },
{ 0x000013F9, U"\U000013F1\U000013F9\U000013F1" },
{ 0x000013FA, U"\U000013F2\U000013FA\U000013F2" },
{ 0x000013FB, U"\U000013F3\U000013FB\U000013F3" },
{ 0x000013FC, U"\U000013F4\U000013FC\U000013F4" },
{ 0x000013FD, U"\U000013F5\U000013FD\U000013F5" },
{ 0x00001C80, U"\U00000412\U00001C80\U00000412" },
{ 0x00001C81, U"\U00000414\U00001C81\U00000414" },
{ 0x00001C82, U"\U0000041E\U00001C82\U0000041E" },
{ 0x00001C83, U"\U00000421\U00001C83\U00000421" },
{ 0x00001C84, U"\U00000422\U00001C84\U00000422" },
{ 0x00001C85, U"\U00000422\U00001C85\U00000422" },
{ 0x00001C86, U"\U0000042A\U00001C86\U0000042A" },
{ 0x00001C87, U"\U00000462\U00001C87\U00000462" },
{ 0x00001C88, U"\U0000A64A\U00001C88\U0000A64A" },
{ 0x00001C90, U"\U00001C90\U000010D0\U00001C90" },
{ 0x00001C91, U"\U00001C91\U000010D1\U00001C91" },
{ 0x00001C92, U"\U00001C92\U000010D2\U00001C92" },
{ 0x00001C93, U"\U00001C93\U000010D3\U00001C93" },
{ 0x00001C94, U"\U00001C94\U000010D4\U00001C94" },
{ 0x00001C95, U"\U00001C95\U000010D5\U00001C95" },
{ 0x00001C96, U"\U00001C96\U000010D6\U00001C96" },
{ 0x00001C97, U"\U00001C97\U000010D7\U00001C97" },
{ 0x00001C98, U"\U00001C98\U000010D8\U00001C98" },
{ 0x00001C99, U"\U00001C99\U000010D9\U00001C99" },
{ 0x00001C9A, U"\U00001C9A\U000010DA\U00001C9A" },
{ 0x00001C9B, U"\U00001C9B\U000010DB\U00001C9B" },
{ 0x00001C9C, U"\U00001C9C\U000010DC\U00001C9C" },
{ 0x00001C9D, U"\U00001C9D\U000010DD\U00001C9D" },
{ 0x00001C9E, U"\U00001C9E\U000010DE\U00001C9E" },
{ 0x00001C9F, U"\U00001C9F\U000010DF\U00001C9F" },
{ 0x00001CA0, U"\U00001CA0\U000010E0\U00001CA0" },
{ 0x00001CA1, U"\U00001CA1\U000010E1\U00001CA1" },
{ 0x00001CA2, U"\U00001CA2\U000010E2\U00001CA2" },
{ 0x00001CA3, U"\U00001CA3\U000010E3\U00001CA3" },
{ 0x00001CA4, U"\U00001CA4\U000010E4\U00001CA4" },
{ 0x00001CA5, U"\U00001CA5\U000010E5\U00001CA5" },
{ 0x00001CA6, U"\U00001CA6\U000010E6\U00001CA6" },
{ 0x00001CA7, U"\U00001CA7\U000010E7\U00001CA7" },
{ 0x00001CA8, U"\U00001CA8\U000010E8\U00001CA8" },
{ 0x00001CA9, U"\U00001CA9\U000010E9\U00001CA9" },
{ 0x00001CAA, U"\U00001CAA\U000010EA\U00001CAA" },
{ 0x00001CAB, U"\U00001CAB\U000010EB\U00001CAB" },
{ 0x00001CAC, U"\U00001CAC\U000010EC\U00001CAC" },
{ 0x00001CAD, U"\U00001CAD\U000010ED\U00001CAD" },
{ 0x00001CAE, U"\U00001CAE\U000010EE\U00001CAE" },
{ 0x00001CAF, U"\U00001CAF\U000010EF\U00001CAF" },
{ 0x00001CB0, U"\U00001CB0\U000010F0\U00001CB0" },
{ 0x00001CB1, U"\U00001CB1\U000010F1\U00001CB1" },
{ 0x00001CB2, U"\U00001CB2\U000010F2\U00001CB2" },
{ 0x00001CB3, U"\U00001CB3\U000010F3\U00001CB3" },
{ 0x00001CB4, U"\U00001CB4\U000010F4\U00001CB4" },
{ 0x00001CB5, U"\U00001CB5\U000010F5\U00001CB5" },
{ 0x00001CB6, U"\U00001CB6\U000010F6\U00001CB6" },
{ 0x00001CB7, U"\U00001CB7\U000010F7\U00001CB7" },
{ 0x00001CB8, U"\U00001CB8\U000010F8\U00001CB8" },
{ 0x00001CB9, U"\U00001CB9\U000010F9\U00001CB9" },
{ 0x00001CBA, U"\U00001CBA\U000010FA\U00001CBA" },
{ 0x00001CBD, U"\U00001CBD\U000010FD\U00001CBD" },
{ 0x00001CBE, U"\U00001CBE\U000010FE\U00001CBE" },
{ 0x00001CBF, U"\U00001CBF\U000010FF\U00001CBF" },
{ 0x00001D79, U"\U0000A77D\U00001D79\U0000A77D" },
{ 0x00001D7D, U"\U00002C63\U00001D7D\U00002C63" },
{ 0x00001D8E, U"\U0000A7C6\U00001D8E\U0000A7C6" },
{ 0x00001E00, U"\U00001E00\U00001E01\U00001E00" },
{ 0x00001E01, U"\U00001E00\U00001E01\U00001E00" },
{ 0x00001E02, U"\U00001E02\U00001E03\U00001E02" },
{ 0x00001E03, U"\U00001E02\U00001E03\U00001E02" },
{ 0x00001E04, U"\U00001E04\U00001E05\U00001E04" },
{ 0x00001E05, U"\U00001E04\U00001E05\U00001E04" },
{ 0x00001E06, U"\U00001E06\U00001E07\U00001E06" },
{ 0x00001E07, U"\U00001E06\U00001E07\U00001E06" },
{ 0x00001E08, U"\U00001E08\U00001E09\U00001E08" },
{ 0x00001E09, U"\U00001E08\U00001E09\U00001E08" },
{ 0x00001E0A, U"\U00001E0A\U00001E0B\U00001E0A" },
{ 0x00001E0B, U"\U00001E0A\U00001E0B\U00001E0A" },
{ 0x00001E0C, U"\U00001E0C\U00001E0D\U00001E0C" },
{ 0x00001E0D, U"\U00001E0C\U00001E0D\U00001E0C" },
{ 0x00001E0E, U"\U00001E0E\U00001E0F\U00001E0E" },
{ 0x00001E0F, U"\U00001E0E\U00001E0F\U00001E0E" },
{ 0x00001E10, U"\U00001E10\U00001E11\U00001E10" },
{ 0x00001E11, U"\U00001E10\U00001E11\U00001E10" },
{ 0x00001E12, U"\U00001E12\U00001E13\U00001E12" },
{ 0x00001E13, U"\U00001E12\U00001E13\U00001E12" },
{ 0x00001E14, U"\U00001E14\U00001E15\U00001E14" },
{ 0x00001E15, U"\U00001E14\U00001E15\U00001E14" },
{ 0x00001E16, U"\U00001E16\U00001E17\U00001E16" },
{ 0x00001E17, U"\U00001E16\U00001E17\U00001E16" },
{ 0x00001E18, U"\U00001E18\U00001E19\U00001E18" },
{ 0x00001E19, U"\U00001E18\U00001E19\U00001E18" },
{ 0x00001E1A, U"\U00001E1A\U00001E1B\U00001E1A" },
{ 0x00001E1B, U"\U00001E1A\U00001E1B\U00001E1A" },
{ 0x00001E1C, U"\U00001E1C\U00001E1D\U00001E1C" },
{ 0x00001E1D, U"\U00001E1C\U00001E1D\U00001E1C" },
{ 0x00001E1E, U"\U00001E1E\U00001E1F\U00001E1E" },
{ 0x00001E1F, U"\U00001E1E\U00001E1F\U00001E1E" },
{ 0x00001E20, U"\U00001E20\U00001E21\U00001E20" },
{ 0x00001E21, U"\U00001E20\U00001E21\U00001E20" },
{ 0x00001E22, U"\U00001E22\U00001E23\U00001E22" },
{ 0x00001E23, U"\U00001E22\U00001E23\U00001E22" },
{ 0x00001E24, U"\U00001E24\U00001E25\U00001E24" },
{ 0x00001E25, U"\U00001E24\U00001E25\U00001E24" },
{ 0x00001E26, U"\U00001E26\U00001E27\U00001E26" },
{ 0x00001E27, U"\U00001E26\U00001E27\U00001E26" },
{ 0x00001E28, U"\U00001E28\U00001E29\U00001E28" },
{ 0x00001E29, U"\U00001E28\U00001E29\U00001E28" },
{ 0x00001E2A, U"\U00001E2A\U00001E2B\U00001E2A" },
{ 0x00001E2B, U"\U00001E2A\U00001E2B\U00001E2A" },
{ 0x00001E2C, U"\U00001E2C\U00001E2D\U00001E2C" },
{ 0x00001E2D, U"\U00001E2C\U00001E2D\U00001E2C" },
{ 0x00001E2E, U"\U00001E2E\U00001E2F\U00001E2E" },
{ 0x00001E2F, U"\U00001E2E\U00001E2F\U00001E2E" },
{ 0x00001E30, U"\U00001E30\U00001E31\U00001E30" },
{ 0x00001E31, U"\U00001E30\U00001E31\U00001E30" },
{ 0x00001E32, U"\U00001E32\U00001E33\U00001E32" },
{ 0x00001E33, U"\U00001E32\U00001E33\U00001E32" },
{ 0x00001E34, U"\U00001E34\U00001E35\U00001E34" },
{ 0x00001E35, U"\U00001E34\U00001E35\U00001E34" },
{ 0x00001E36, U"\U00001E36\U00001E37\U00001E36" },
{ 0x00001E37, U"\U00001E36\U00001E37\U00001E36" },
{ 0x00001E38, U"\U00001E38\U00001E39\U00001E38" },
{ 0x00001E39, U"\U00001E38\U00001E39\U00001E38" },
{ 0x00001E3A, U"\U00001E3A\U00001E3B\U00001E3A" },
{ 0x00001E3B, U"\U00001E3A\U00001E3B\U00001E3A" },
{ 0x00001E3C, U"\U00001E3C\U00001E3D\U00001E3C" },
{ 0x00001E3D, U"\U00001E3C\U00001E3D\U00001E3C" },
{ 0x00001E3E, U"\U00001E3E\U00001E3F\U00001E3E" },
{ 0x00001E3F, U"\U00001E3E\U00001E3F\U00001E3E" },
{ 0x00001E40, U"\U00001E40\U00001E41\U00001E40" },
{ 0x00001E41, U"\U00001E40\U00001E41\U00001E40" },
{ 0x00001E42, U"\U00001E42\U00001E43\U00001E42" },
{ 0x00001E43, U"\U00001E42\U00001E43\U00001E42" },
{ 0x00001E44, U"\U00001E44\U00001E45\U00001E44" },
{ 0x00001E45, U"\U00001E44\U00001E45\U00001E44" },
{ 0x00001E46, U"\U00001E46\U00001E47\U00001E46" },
{ 0x00001E47, U"\U00001E46\U00001E47\U00001E46" },
{ 0x00001E48, U"\U00001E48\U00001E49\U00001E48" },
{ 0x00001E49, U"\U00001E48\U00001E49\U00001E48" },
{ 0x00001E4A, U"\U00001E4A\U00001E4B\U00001E4A" },
{ 0x00001E4B, U"\U00001E4A\U00001E4B\U00001E4A" },
{ 0x00001E4C, U"\U00001E4C\U00001E4D\U00001E4C" },
{ 0x00001E4D, U"\U00001E4C\U00001E4D\U00001E4C" },
{ 0x00001E4E, U"\U00001E4E\U00001E4F\U00001E4E" },
{ 0x00001E4F, U"\U00001E4E\U00001E4F\U00001E4E" },
{ 0x00001E50, U"\U00001E50\U00001E51\U00001E50" },
{ 0x00001E51, U"\U00001E50\U00001E51\U00001E50" },
{ 0x00001E52, U"\U00001E52\U00001E53\U00001E52" },
{ 0x00001E53, U"\U00001E52\U00001E53\U00001E52" },
{ 0x00001E54, U"\U00001E54\U00001E55\U00001E54" },
{ 0x00001E55, U"\U00001E54\U00001E55\U00001E54" },
{ 0x00001E56, U"\U00001E56\U00001E57\U00001E56" },
{ 0x00001E57, U"\U00001E56\U00001E57\U00001E56" },
{ 0x00001E58, U"\U00001E58\U00001E59\U00001E58" },
{ 0x00001E59, U"\U00001E58\U00001E59\U00001E58" },
{ 0x00001E5A, U"\U00001E5A\U00001E5B\U00001E5A" },
{ 0x00001E5B, U"\U00001E5A\U00001E5B\U00001E5A" },
{ 0x00001E5C, U"\U00001E5C\U00001E5D\U00001E5C" },
{ 0x00001E5D, U"\U00001E5C\U00001E5D\U00001E5C" },
{ 0x00001E5E, U"\U00001E5E\U00001E5F\U00001E5E" },
{ 0x00001E5F, U"\U00001E5E\U00001E5F\U00001E5E" },
{ 0x00001E60, U"\U00001E60\U00001E61\U00001E60" },
{ 0x00001E61, U"\U00001E60\U00001E61\U00001E60" },
{ 0x00001E62, U"\U00001E62\U00001E63\U00001E62" },
{ 0x00001E63, U"\U00001E62\U00001E63\U00001E62" },
{ 0x00001E64, U"\U00001E64\U00001E65\U00001E64" },
{ 0x00001E65, U"\U00001E64\U00001E65\U00001E64" },
{ 0x00001E66, U"\U00001E66\U00001E67\U00001E66" },
{ 0x00001E67, U"\U00001E66\U00001E67\U00001E66" },
{ 0x00001E68, U"\U00001E68\U00001E69\U00001E68" },
{ 0x00001E69, U"\U00001E68\U00001E69\U00001E68" },
{ 0x00001E6A, U"\U00001E6A\U00001E6B\U00001E6A" },
{ 0x00001E6B, U"\U00001E6A\U00001E6B\U00001E6A" },
{ 0x00001E6C, U"\U00001E6C\U00001E6D\U00001E6C" },
{ 0x00001E6D, U"\U00001E6C\U00001E6D\U00001E6C" },
{ 0x00001E6E, U"\U00001E6E\U00001E6F\U00001E6E" },
{ 0x00001E6F, U"\U00001E6E\U00001E6F\U00001E6E" },
{ 0x00001E70, U"\U00001E70\U00001E71\U00001E70" },
{ 0x00001E71, U"\U00001E70\U00001E71\U00001E70" },
{ 0x00001E72, U"\U00001E72\U00001E73\U00001E72" },
{ 0x00001E73, U"\U00001E72\U00001E73\U00001E72" },
{ 0x00001E74, U"\U00001E74\U00001E75\U00001E74" },
{ 0x00001E75, U"\U00001E74\U00001E75\U00001E74" },
{ 0x00001E76, U"\U00001E76\U00001E77\U00001E76" },
{ 0x00001E77, U"\U00001E76\U00001E77\U00001E76" },
{ 0x00001E78, U"\U00001E78\U00001E79\U00001E78" },
{ 0x00001E79, U"\U00001E78\U00001E79\U00001E78" },
{ 0x00001E7A, U"\U00001E7A\U00001E7B\U00001E7A" },
{ 0x00001E7B, U"\U00001E7A\U00001E7B\U00001E7A" },
{ 0x00001E7C, U"\U00001E7C\U00001E7D\U00001E7C" },
{ 0x00001E7D, U"\U00001E7C\U00001E7D\U00001E7C" },
{ 0x00001E7E, U"\U00001E7E\U00001E7F\U00001E7E" },
{ 0x00001E7F, U"\U00001E7E\U00001E7F\U00001E7E" },
{ 0x00001E80, U"\U00001E80\U00001E81\U00001E80" },
{ 0x00001E81, U"\U00001E80\U00001E81\U00001E80" },
{ 0x00001E82, U"\U00001E82\U00001E83\U00001E82" },
{ 0x00001E83, U"\U00001E82\U00001E83\U00001E82" },
{ 0x00001E84, U"\U00001E84\U00001E85\U00001E84" },
{ 0x00001E85, U"\U00001E84\U00001E85\U00001E84" },
{ 0x00001E86, U"\U00001E86\U00001E87\U00001E86" },
{ 0x00001E87, U"\U00001E86\U00001E87\U00001E86" },
{ 0x00001E88, U"\U00001E88\U00001E89\U00001E88" },
{ 0x00001E89, U"\U00001E88\U00001E89\U00001E88" },
{ 0x00001E8A, U"\U00001E8A\U00001E8B\U00001E8A" },
{ 0x00001E8B, U"\U00001E8A\U00001E8B\U00001E8A" },
{ 0x00001E8C, U"\U00001E8C\U00001E8D\U00001E8C" },
{ 0x00001E8D, U"\U00001E8C\U00001E8D\U00001E8C" },
{ 0x00001E8E, U"\U00001E8E\U00001E8F\U00001E8E" },
{ 0x00001E8F, U"\U00001E8E\U00001E8F\U00001E8E" },
{ 0x00001E90, U"\U00001E90\U00001E91\U00001E90" },
{ 0x00001E91, U"\U00001E90\U00001E91\U00001E90" },
{ 0x00001E92, U"\U00001E92\U00001E93\U00001E92" },
{ 0x00001E93, U"\U00001E92\U00001E93\U00001E92" },
{ 0x00001E94, U"\U00001E94\U00001E95\U00001E94" },
{ 0x00001E95, U"\U00001E94\U00001E95\U00001E94" },
{ 0x00001E9B, U"\U00001E60\U00001E9B\U00001E60" },
{ 0x00001E9E, U"\U00001E9E\U000000DF\U00001E9E" },
{ 0x00001EA0, U"\U00001EA0\U00001EA1\U00001EA0" },
{ 0x00001EA1, U"\U00001EA0\U00001EA1\U00001EA0" },
{ 0x00001EA2, U"\U00001EA2\U00001EA3\U00001EA2" },
{ 0x00001EA3, U"\U00001EA2\U00001EA3\U00001EA2" },
{ 0x00001EA4, U"\U00001EA4\U00001EA5\U00001EA4" },
{ 0x00001EA5, U"\U00001EA4\U00001EA5\U00001EA4" },
{ 0x00001EA6, U"\U00001EA6\U00001EA7\U00001EA6" },
{ 0x00001EA7, U"\U00001EA6\U00001EA7\U00001EA6" },
{ 0x00001EA8, U"\U00001EA8\U00001EA9\U00001EA8" },
{ 0x00001EA9, U"\U00001EA8\U00001EA9\U00001EA8" },
{ 0x00001EAA, U"\U00001EAA\U00001EAB\U00001EAA" },
{ 0x00001EAB, U"\U00001EAA\U00001EAB\U00001EAA" },
{ 0x00001EAC, U"\U00001EAC\U00001EAD\U00001EAC" },
{ 0x00001EAD, U"\U00001EAC\U00001EAD\U00001EAC" },
{ 0x00001EAE, U"\U00001EAE\U00001EAF\U00001EAE" },
{ 0x00001EAF, U"\U00001EAE\U00001EAF\U00001EAE" },
{ 0x00001EB0, U"\U00001EB0\U00001EB1\U00001EB0" },
{ 0x00001EB1, U"\U00001EB0\U00001EB1\U00001EB0" },
{ 0x00001EB2, U"\U00001EB2\U00001EB3\U00001EB2" },
{ 0x00001EB3, U"\U00001EB2\U00001EB3\U00001EB2" },
{ 0x00001EB4, U"\U00001EB4\U00001EB5\U00001EB4" },
{ 0x00001EB5, U"\U00001EB4\U00001EB5\U00001EB4" },
{ 0x00001EB6, U"\U00001EB6\U00001EB7\U00001EB6" },
{ 0x00001EB7, U"\U00001EB6\U00001EB7\U00001EB6" },
{ 0x00001EB8, U"\U00001EB8\U00001EB9\U00001EB8" },
{ 0x00001EB9, U"\U00001EB8\U00001EB9\U00001EB8" },
{ 0x00001EBA, U"\U00001EBA\U00001EBB\U00001EBA" },
{ 0x00001EBB, U"\U00001EBA\U00001EBB\U00001EBA" },
{ 0x00001EBC, U"\U00001EBC\U00001EBD\U00001EBC" },
{ 0x00001EBD, U"\U00001EBC\U00001EBD\U00001EBC" },
{ 0x00001EBE, U"\U00001EBE\U00001EBF\U00001EBE" },
{ 0x00001EBF, U"\U00001EBE\U00001EBF\U00001EBE" },
{ 0x00001EC0, U"\U00001EC0\U00001EC1\U00001EC0" },
{ 0x00001EC1, U"\U00001EC0\U00001EC1\U00001EC0" },
{ 0x00001EC2, U"\U00001EC2\U00001EC3\U00001EC2" },
{ 0x00001EC3, U"\U00001EC2\U00001EC3\U00001EC2" },
{ 0x00001EC4, U"\U00001EC4\U00001EC5\U00001EC4" },
{ 0x00001EC5, U"\U00001EC4\U00001EC5\U00001EC4" },
{ 0x00001EC6, U"\U00001EC6\U00001EC7\U00001EC6" },
{ 0x00001EC7, U"\U00001EC6\U00001EC7\U00001EC6" },
{ 0x00001EC8, U"\U00001EC8\U00001EC9\U00001EC8" },
{ 0x00001EC9, U"\U00001EC8\U00001EC9\U00001EC8" },
{ 0x00001ECA, U"\U00001ECA\U00001ECB\U00001ECA" },
{ 0x00001ECB, U"\U00001ECA\U00001ECB\U00001ECA" },
{ 0x00001ECC, U"\U00001ECC\U00001ECD\U00001ECC" },
{ 0x00001ECD, U"\U00001ECC\U00001ECD\U00001ECC" },
{ 0x00001ECE, U"\U00001ECE\U00001ECF\U00001ECE" },
{ 0x00001ECF, U"\U00001ECE\U00001ECF\U00001ECE" },
{ 0x00001ED0, U"\U00001ED0\U00001ED1\U00001ED0" },
{ 0x00001ED1, U"\U00001ED0\U00001ED1\U00001ED0" },
{ 0x00001ED2, U"\U00001ED2\U00001ED3\U00001ED2" },
{ 0x00001ED3, U"\U00001ED2\U00001ED3\U00001ED2" },
{ 0x00001ED4, U"\U00001ED4\U00001ED5\U00001ED4" },
{ 0x00001ED5, U"\U00001ED4\U00001ED5\U00001ED4" },
{ 0x00001ED6, U"\U00001ED6\U00001ED7\U00001ED6" },
{ 0x00001ED7, U"\U00001ED6\U00001ED7\U00001ED6" },
{ 0x00001ED8, U"\U00001ED8\U00001ED9\U00001ED8" },
{ 0x00001ED9, U"\U00001ED8\U00001ED9\U00001ED8" },
{ 0x00001EDA, U"\U00001EDA\U00001EDB\U00001EDA" },
{ 0x00001EDB, U"\U00001EDA\U00001EDB\U00001EDA" },
{ 0x00001EDC, U"\U00001EDC\U00001EDD\U00001EDC" },
{ 0x00001EDD, U"\U00001EDC\U00001EDD\U00001EDC" },
{ 0x00001EDE, U"\U00001EDE\U00001EDF\U00001EDE" },
{ 0x00001EDF, U"\U00001EDE\U00001EDF\U00001EDE" },
{ 0x00001EE0, U"\U00001EE0\U00001EE1\U00001EE0" },
{ 0x00001EE1, U"\U00001EE0\U00001EE1\U00001EE0" },
{ 0x00001EE2, U"\U00001EE2\U00001EE3\U00001EE2" },
{ 0x00001EE3, U"\U00001EE2\U00001EE3\U00001EE2" },
{ 0x00001EE4, U"\U00001EE4\U00001EE5\U00001EE4" },
{ 0x00001EE5, U"\U00001EE4\U00001EE5\U00001EE4" },
{ 0x00001EE6, U"\U00001EE6\U00001EE7\U00001EE6" },
{ 0x00001EE7, U"\U00001EE6\U00001EE7\U00001EE6" },
{ 0x00001EE8, U"\U00001EE8\U00001EE9\U00001EE8" },
{ 0x00001EE9, U"\U00001EE8\U00001EE9\U00001EE8" },
{ 0x00001EEA, U"\U00001EEA\U00001EEB\U00001EEA" },
{ 0x00001EEB, U"\U00001EEA\U00001EEB\U00001EEA" },
{ 0x00001EEC, U"\U00001EEC\U00001EED\U00001EEC" },
{ 0x00001EED, U"\U00001EEC\U00001EED\U00001EEC" },
{ 0x00001EEE, U"\U00001EEE\U00001EEF\U00001EEE" },
{ 0x00001EEF, U"\U00001EEE\U00001EEF\U00001EEE" },
{ 0x00001EF0, U"\U00001EF0\U00001EF1\U00001EF0" },
{ 0x00001EF1, U"\U00001EF0\U00001EF1\U00001EF0" },
{ 0x00001EF2, U"\U00001EF2\U00001EF3\U00001EF2" },
{ 0x00001EF3, U"\U00001EF2\U00001EF3\U00001EF2" },
{ 0x00001EF4, U"\U00001EF4\U00001EF5\U00001EF4" },
{ 0x00001EF5, U"\U00001EF4\U00001EF5\U00001EF4" },
{ 0x00001EF6, U"\U00001EF6\U00001EF7\U00001EF6" },
{ 0x00001EF7, U"\U00001EF6\U00001EF7\U00001EF6" },
{ 0x00001EF8, U"\U00001EF8\U00001EF9\U00001EF8" },
{ 0x00001EF9, U"\U00001EF8\U00001EF9\U00001EF8" },
{ 0x00001EFA, U"\U00001EFA\U00001EFB\U00001EFA" },
{ 0x00001EFB, U"\U00001EFA\U00001EFB\U00001EFA" },
{ 0x00001EFC, U"\U00001EFC\U00001EFD\U00001EFC" },
{ 0x00001EFD, U"\U00001EFC\U00001EFD\U00001EFC" },
{ 0x00001EFE, U"\U00001EFE\U00001EFF\U00001EFE" },
{ 0x00001EFF, U"\U00001EFE\U00001EFF\U00001EFE" },
{ 0x00001F00, U"\U00001F08\U00001F00\U00001F08" },
{ 0x00001F01, U"\U00001F09\U00001F01\U00001F09" },
{ 0x00001F02, U"\U00001F0A\U00001F02\U00001F0A" },
{ 0x00001F03, U"\U00001F0B\U00001F03\U00001F0B" },
{ 0x00001F04, U"\U00001F0C\U00001F04\U00001F0C" },
{ 0x00001F05, U"\U00001F0D\U00001F05\U00001F0D" },
{ 0x00001F06, U"\U00001F0E\U00001F06\U00001F0E" },
{ 0x00001F07, U"\U00001F0F\U00001F07\U00001F0F" },
{ 0x00001F08, U"\U00001F08\U00001F00\U00001F08" },
{ 0x00001F09, U"\U00001F09\U00001F01\U00001F09" },
{ 0x00001F0A, U"\U00001F0A\U00001F02\U00001F0A" },
{ 0x00001F0B, U"\U00001F0B\U00001F03\U00001F0B" },
{ 0x00001F0C, U"\U00001F0C\U00001F04\U00001F0C" },
{ 0x00001F0D, U"\U00001F0D\U00001F05\U00001F0D" },
{ 0x00001F0E, U"\U00001F0E\U00001F06\U00001F0E" },
{ 0x00001F0F, U"\U00001F0F\U00001F07\U00001F0F" },
{ 0x00001F10, U"\U00001F18\U00001F10\U00001F18" },
{ 0x00001F11, U"\U00001F19\U00001F11\U00001F19" },
{ 0x00001F12, U"\U00001F1A\U00001F12\U00001F1A" },
{ 0x00001F13, U"\U00001F1B\U00001F13\U00001F1B" },
{ 0x00001F14, U"\U00001F1C\U00001F14\U00001F1C" },
{ 0x00001F15, U"\U00001F1D\U00001F15\U00001F1D" },
{ 0x00001F18, U"\U00001F18\U00001F10\U00001F18" },
{ 0x00001F19, U"\U00001F19\U00001F11\U00001F19" },
{ 0x00001F1A, U"\U00001F1A\U00001F12\U00001F1A" },
{ 0x00001F1B, U"\U00001F1B\U00001F13\U00001F1B" },
{ 0x00001F1C, U"\U00001F1C\U00001F14\U00001F1C" },
{ 0x00001F1D, U"\U00001F1D\U00001F15\U00001F1D" },
{ 0x00001F20, U"\U00001F28\U00001F20\U00001F28" },
{ 0x00001F21, U"\U00001F29\U00001F21\U00001F29" },
{ 0x00001F22, U"\U00001F2A\U00001F22\U00001F2A" },
{ 0x00001F23, U"\U00001F2B\U00001F23\U00001F2B" },
{ 0x00001F24, U"\U00001F2C\U00001F24\U00001F2C" },
{ 0x00001F25, U"\U00001F2D\U00001F25\U00001F2D" },
{ 0x00001F26, U"\U00001F2E\U00001F26\U00001F2E" },
{ 0x00001F27, U"\U00001F2F\U00001F27\U00001F2F" },
{ 0x00001F28, U"\U00001F28\U00001F20\U00001F28" },
{ 0x00001F29, U"\U00001F29\U00001F21\U00001F29" },
{ 0x00001F2A, U"\U00001F2A\U00001F22\U00001F2A" },
{ 0x00001F2B, U"\U00001F2B\U00001F23\U00001F2B" },
{ 0x00001F2C, U"\U00001F2C\U00001F24\U00001F2C" },
{ 0x00001F2D, U"\U00001F2D\U00001F25\U00001F2D" },
{ 0x00001F2E, U"\U00001F2E\U00001F26\U00001F2E" },
{ 0x00001F2F, U"\U00001F2F\U00001F27\U00001F2F" },
{ 0x00001F30, U"\U00001F38\U00001F30\U00001F38" },
{ 0x00001F31, U"\U00001F39\U00001F31\U00001F39" },
{ 0x00001F32, U"\U00001F3A\U00001F32\U00001F3A" },
{ 0x00001F33, U"\U00001F3B\U00001F33\U00001F3B" },
{ 0x00001F34, U"\U00001F3C\U00001F34\U00001F3C" },
{ 0x00001F35, U"\U00001F3D\U00001F35\U00001F3D" },
{ 0x00001F36, U"\U00001F3E\U00001F36\U00001F3E" },
{ 0x00001F37, U"\U00001F3F\U00001F37\U00001F3F" },
{ 0x00001F38, U"\U00001F38\U00001F30\U00001F38" },
{ 0x00001F39, U"\U00001F39\U00001F31\U00001F39" },
{ 0x00001F3A, U"\U00001F3A\U00001F32\U00001F3A" },
{ 0x00001F3B, U"\U00001F3B\U00001F33\U00001F3B" },
{ 0x00001F3C, U"\U00001F3C\U00001F34\U00001F3C" },
{ 0x00001F3D, U"\U00001F3D\U00001F35\U00001F3D" },
{ 0x00001F3E, U"\U00001F3E\U00001F36\U00001F3E" },
{ 0x00001F3F, U"\U00001F3F\U00001F37\U00001F3F" },
{ 0x00001F40, U"\U00001F48\U00001F40\U00001F48" },
{ 0x00001F41, U"\U00001F49\U00001F41\U00001F49" },
{ 0x00001F42, U"\U00001F4A\U00001F42\U00001F4A" },
{ 0x00001F43, U"\U00001F4B\U00001F43\U00001F4B" },
{ 0x00001F44, U"\U00001F4C\U00001F44\U00001F4C" },
{ 0x00001F45, U"\U00001F4D\U00001F45\U00001F4D" },
{ 0x00001F48, U"\U00001F48\U00001F40\U00001F48" },
{ 0x00001F49, U"\U00001F49\U00001F41\U00001F49" },
{ 0x00001F4A, U"\U00001F4A\U00001F42\U00001F4A" },
{ 0x00001F4B, U"\U00001F4B\U00001F43\U00001F4B" },
{ 0x00001F4C, U"\U00001F4C\U00001F44\U00001F4C" },
{ 0x00001F4D, U"\U00001F4D\U00001F45\U00001F4D" },
{ 0x00001F51, U"\U00001F59\U00001F51\U00001F59" },
{ 0x00001F53, U"\U00001F5B\U00001F53\U00001F5B" },
{ 0x00001F55, U"\U00001F5D\U00001F55\U00001F5D" },
{ 0x00001F57, U"\U00001F5F\U00001F57\U00001F5F" },
{ 0x00001F59, U"\U00001F59\U00001F51\U00001F59" },
{ 0x00001F5B, U"\U00001F5B\U00001F53\U00001F5B" },
{ 0x00001F5D, U"\U00001F5D\U00001F55\U00001F5D" },
{ 0x00001F5F, U"\U00001F5F\U00001F57\U00001F5F" },
{ 0x00001F60, U"\U00001F68\U00001F60\U00001F68" },
{ 0x00001F61, U"\U00001F69\U00001F61\U00001F69" },
{ 0x00001F62, U"\U00001F6A\U00001F62\U00001F6A" },
{ 0x00001F63, U"\U00001F6B\U00001F63\U00001F6B" },
{ 0x00001F64, U"\U00001F6C\U00001F64\U00001F6C" },
{ 0x00001F65, U"\U00001F6D\U00001F65\U00001F6D" },
{ 0x00001F66, U"\U00001F6E\U00001F66\U00001F6E" },
{ 0x00001F67, U"\U00001F6F\U00001F67\U00001F6F" },
{ 0x00001F68, U"\U00001F68\U00001F60\U00001F68" },
{ 0x00001F69, U"\U00001F69\U00001F61\U00001F69" },
{ 0x00001F6A, U"\U00001F6A\U00001F62\U00001F6A" },
{ 0x00001F6B, U"\U00001F6B\U00001F63\U00001F6B" },
{ 0x00001F6C, U"\U00001F6C\U00001F64\U00001F6C" },
{ 0x00001F6D, U"\U00001F6D\U00001F65\U00001F6D" },
{ 0x00001F6E, U"\U00001F6E\U00001F66\U00001F6E" },
{ 0x00001F6F, U"\U00001F6F\U00001F67\U00001F6F" },
{ 0x00001F70, U"\U00001FBA\U00001F70\U00001FBA" },
{ 0x00001F71, U"\U00001FBB\U00001F71\U00001FBB" },
{ 0x00001F72, U"\U00001FC8\U00001F72\U00001FC8" },
{ 0x00001F73, U"\U00001FC9\U00001F73\U00001FC9" },
{ 0x00001F74, U"\U00001FCA\U00001F74\U00001FCA" },
{ 0x00001F75, U"\U00001FCB\U00001F75\U00001FCB" },
{ 0x00001F76, U"\U00001FDA\U00001F76\U00001FDA" },
{ 0x00001F77, U"\U00001FDB\U00001F77\U00001FDB" },
{ 0x00001F78, U"\U00001FF8\U00001F78\U00001FF8" },
{ 0x00001F79, U"\U00001FF9\U00001F79\U00001FF9" },
{ 0x00001F7A, U"\U00001FEA\U00001F7A\U00001FEA" },
{ 0x00001F7B, U"\U00001FEB\U00001F7B\U00001FEB" },
{ 0x00001F7C, U"\U00001FFA\U00001F7C\U00001FFA" },
{ 0x00001F7D, U"\U00001FFB\U00001F7D\U00001FFB" },
{ 0x00001F80, U"\U00001F88\U00001F80\U00001F88" },
{ 0x00001F81, U"\U00001F89\U00001F81\U00001F89" },
{ 0x00001F82, U"\U00001F8A\U00001F82\U00001F8A" },
{ 0x00001F83, U"\U00001F8B\U00001F83\U00001F8B" },
{ 0x00001F84, U"\U00001F8C\U00001F84\U00001F8C" },
{ 0x00001F85, U"\U00001F8D\U00001F85\U00001F8D" },
{ 0x00001F86, U"\U00001F8E\U00001F86\U00001F8E" },
{ 0x00001F87, U"\U00001F8F\U00001F87\U00001F8F" },
{ 0x00001F88, U"\U00001F88\U00001F80\U00001F88" },
{ 0x00001F89, U"\U00001F89\U00001F81\U00001F89" },
{ 0x00001F8A, U"\U00001F8A\U00001F82\U00001F8A" },
{ 0x00001F8B, U"\U00001F8B\U00001F83\U00001F8B" },
{ 0x00001F8C, U"\U00001F8C\U00001F84\U00001F8C" },
{ 0x00001F8D, U"\U00001F8D\U00001F85\U00001F8D" },
{ 0x00001F8E, U"\U00001F8E\U00001F86\U00001F8E" },
{ 0x00001F8F, U"\U00001F8F\U00001F87\U00001F8F" },
{ 0x00001F90, U"\U00001F98\U00001F90\U00001F98" },
{ 0x00001F91, U"\U00001F99\U00001F91\U00001F99" },
{ 0x00001F92, U"\U00001F9A\U00001F92\U00001F9A" },
{ 0x00001F93, U"\U00001F9B\U00001F93\U00001F9B" },
{ 0x00001F94, U"\U00001F9C\U00001F94\U00001F9C" },
{ 0x00001F95, U"\U00001F9D\U00001F95\U00001F9D" },
{ 0x00001F96, U"\U00001F9E\U00001F96\U00001F9E" },
{ 0x00001F97, U"\U00001F9F\U00001F97\U00001F9F" },
{ 0x00001F98, U"\U00001F98\U00001F90\U00001F98" },
{ 0x00001F99, U"\U00001F99\U00001F91\U00001F99" },
{ 0x00001F9A, U"\U00001F9A\U00001F92\U00001F9A" },
{ 0x00001F9B, U"\U00001F9B\U00001F93\U00001F9B" },
{ 0x00001F9C, U"\U00001F9C\U00001F94\U00001F9C" },
{ 0x00001F9D, U"\U00001F9D\U00001F95\U00001F9D" },
{ 0x00001F9E, U"\U00001F9E\U00001F96\U00001F9E" },
{ 0x00001F9F, U"\U00001F9F\U00001F97\U00001F9F" },
{ 0x00001FA0, U"\U00001FA8\U00001FA0\U00001FA8" },
{ 0x00001FA1, U"\U00001FA9\U00001FA1\U00001FA9" },
{ 0x00001FA2, U"\U00001FAA\U00001FA2\U00001FAA" },
{ 0x00001FA3, U"\U00001FAB\U00001FA3\U00001FAB" },
{ 0x00001FA4, U"\U00001FAC\U00001FA4\U00001FAC" },
{ 0x00001FA5, U"\U00001FAD\U00001FA5\U00001FAD" },
{ 0x00001FA6, U"\U00001FAE\U00001FA6\U00001FAE" },
{ 0x00001FA7, U"\U00001FAF\U00001FA7\U00001FAF" },
{ 0x00001FA8, U"\U00001FA8\U00001FA0\U00001FA8" },
{ 0x00001FA9, U"\U00001FA9\U00001FA1\U00001FA9" },
{ 0x00001FAA, U"\U00001FAA\U00001FA2\U00001FAA" },
{ 0x00001FAB, U"\U00001FAB\U00001FA3\U00001FAB" },
{ 0x00001FAC, U"\U00001FAC\U00001FA4\U00001FAC" },
{ 0x00001FAD, U"\U00001FAD\U00001FA5\U00001FAD" },
{ 0x00001FAE, U"\U00001FAE\U00001FA6\U00001FAE" },
{ 0x00001FAF, U"\U00001FAF\U00001FA7\U00001FAF" },
{ 0x00001FB0, U"\U00001FB8\U00001FB0\U00001FB8" },
{ 0x00001FB1, U"\U00001FB9\U00001FB1\U00001FB9" },
{ 0x00001FB3, U"\U00001FBC\U00001FB3\U00001FBC" },
{ 0x00001FB8, U"\U00001FB8\U00001FB0\U00001FB8" },
{ 0x00001FB9, U"\U00001FB9\U00001FB1\U00001FB9" },
{ 0x00001FBA, U"\U00001FBA\U00001F70\U00001FBA" },
{ 0x00001FBB, U"\U00001FBB\U00001F71\U00001FBB" },
{ 0x00001FBC, U"\U00001FBC\U00001FB3\U00001FBC" },
{ 0x00001FBE, U"\U00000399\U00001FBE\U00000399" },
{ 0x00001FC3, U"\U00001FCC\U00001FC3\U00001FCC" },
{ 0x00001FC8, U"\U00001FC8\U00001F72\U00001FC8" },
{ 0x00001FC9, U"\U00001FC9\U00001F73\U00001FC9" },
{ 0x00001FCA, U"\U00001FCA\U00001F74\U00001FCA" },
{ 0x00001FCB, U"\U00001FCB\U00001F75\U00001FCB" },
{ 0x00001FCC, U"\U00001FCC\U00001FC3\U00001FCC" },
{ 0x00001FD0, U"\U00001FD8\U00001FD0\U00001FD8" },
{ 0x00001FD1, U"\U00001FD9\U00001FD1\U00001FD9" },
{ 0x00001FD8, U"\U00001FD8\U00001FD0\U00001FD8" },
{ 0x00001FD9, U"\U00001FD9\U00001FD1\U00001FD9" },
{ 0x00001FDA, U"\U00001FDA\U00001F76\U00001FDA" },
{ 0x00001FDB, U"\U00001FDB\U00001F77\U00001FDB" },
{ 0x00001FE0, U"\U00001FE8\U00001FE0\U00001FE8" },
{ 0x00001FE1, U"\U00001FE9\U00001FE1\U00001FE9" },
{ 0x00001FE5, U"\U00001FEC\U00001FE5\U00001FEC" },
{ 0x00001FE8, U"\U00001FE8\U00001FE0\U00001FE8" },
{ 0x00001FE9, U"\U00001FE9\U00001FE1\U00001FE9" },
{ 0x00001FEA, U"\U00001FEA\U00001F7A\U00001FEA" },
{ 0x00001FEB, U"\U00001FEB\U00001F7B\U00001FEB" },
{ 0x00001FEC, U"\U00001FEC\U00001FE5\U00001FEC" },
{ 0x00001FF3, U"\U00001FFC\U00001FF3\U00001FFC" },
{ 0x00001FF8, U"\U00001FF8\U00001F78\U00001FF8" },
{ 0x00001FF9, U"\U00001FF9\U00001F79\U00001FF9" },
{ 0x00001FFA, U"\U00001FFA\U00001F7C\U00001FFA" },
{ 0x00001FFB, U"\U00001FFB\U00001F7D\U00001FFB" },
{ 0x00001FFC, U"\U00001FFC\U00001FF3\U00001FFC" },
{ 0x00002126, U"\U00002126\U000003C9\U00002126" },
{ 0x0000212A, U"\U0000212A\U0000006B\U0000212A" },
{ 0x0000212B, U"\U0000212B\U000000E5\U0000212B" },
{ 0x00002132, U"\U00002132\U0000214E\U00002132" },
{ 0x0000214E, U"\U00002132\U0000214E\U00002132" },
{ 0x00002160, U"\U00002160\U00002170\U00002160" },
{ 0x00002161, U"\U00002161\U00002171\U00002161" },
{ 0x00002162, U"\U00002162\U00002172\U00002162" },
{ 0x00002163, U"\U00002163\U00002173\U00002163" },
{ 0x00002164, U"\U00002164\U00002174\U00002164" },
{ 0x00002165, U"\U00002165\U00002175\U00002165" },
{ 0x00002166, U"\U00002166\U00002176\U00002166" },
{ 0x00002167, U"\U00002167\U00002177\U00002167" },
{ 0x00002168, U"\U00002168\U00002178\U00002168" },
{ 0x00002169, U"\U00002169\U00002179\U00002169" },
{ 0x0000216A, U"\U0000216A\U0000217A\U0000216A" },
{ 0x0000216B, U"\U0000216B\U0000217B\U0000216B" },
{ 0x0000216C, U"\U0000216C\U0000217C\U0000216C" },
{ 0x0000216D, U"\U0000216D\U0000217D\U0000216D" },
{ 0x0000216E, U"\U0000216E\U0000217E\U0000216E" },
{ 0x0000216F, U"\U0000216F\U0000217F\U0000216F" },
{ 0x00002170, U"\U00002160\U00002170\U00002160" },
{ 0x00002171, U"\U00002161\U00002171\U00002161" },
{ 0x00002172, U"\U00002162\U00002172\U00002162" },
{ 0x00002173, U"\U00002163\U00002173\U00002163" },
{ 0x00002174, U"\U00002164\U00002174\U00002164" },
{ 0x00002175, U"\U00002165\U00002175\U00002165" },
{ 0x00002176, U"\U00002166\U00002176\U00002166" },
{ 0x00002177, U"\U00002167\U00002177\U00002167" },
{ 0x00002178, U"\U00002168\U00002178\U00002168" },
{ 0x00002179, U"\U00002169\U00002179\U00002169" },
{ 0x0000217A, U"\U0000216A\U0000217A\U0000216A" },
{ 0x0000217B, U"\U0000216B\U0000217B\U0000216B" },
{ 0x0000217C, U"\U0000216C\U0000217C\U0000216C" },
{ 0x0000217D, U"\U0000216D\U0000217D\U0000216D" },
{ 0x0000217E, U"\U0000216E\U0000217E\U0000216E" },
{ 0x0000217F, U"\U0000216F\U0000217F\U0000216F" },
{ 0x00002183, U"\U00002183\U00002184\U00002183" },
{ 0x00002184, U"\U00002183\U00002184\U00002183" },
{ 0x000024B6, U"\U000024B6\U000024D0\U000024B6" },
{ 0x000024B7, U"\U000024B7\U000024D1\U000024B7" },
{ 0x000024B8, U"\U000024B8\U000024D2\U000024B8" },
{ 0x000024B9, U"\U000024B9\U000024D3\U000024B9" },
{ 0x000024BA, U"\U000024BA\U000024D4\U000024BA" },
{ 0x000024BB, U"\U000024BB\U000024D5\U000024BB" },
{ 0x000024BC, U"\U000024BC\U000024D6\U000024BC" },
{ 0x000024BD, U"\U000024BD\U000024D7\U000024BD" },
{ 0x000024BE, U"\U000024BE\U000024D8\U000024BE" },
{ 0x000024BF, U"\U000024BF\U000024D9\U000024BF" },
{ 0x000024C0, U"\U000024C0\U000024DA\U000024C0" },
{ 0x000024C1, U"\U000024C1\U000024DB\U000024C1" },
{ 0x000024C2, U"\U000024C2\U000024DC\U000024C2" },
{ 0x000024C3, U"\U000024C3\U000024DD\U000024C3" },
{ 0x000024C4, U"\U000024C4\U000024DE\U000024C4" },
{ 0x000024C5, U"\U000024C5\U000024DF\U000024C5" },
{ 0x000024C6, U"\U000024C6\U000024E0\U000024C6" },
{ 0x000024C7, U"\U000024C7\U000024E1\U000024C7" },
{ 0x000024C8, U"\U000024C8\U000024E2\U000024C8" },
{ 0x000024C9, U"\U000024C9\U000024E3\U000024C9" },
{ 0x000024CA, U"\U000024CA\U000024E4\U000024CA" },
{ 0x000024CB, U"\U000024CB\U000024E5\U000024CB" },
{ 0x000024CC, U"\U000024CC\U000024E6\U000024CC" },
{ 0x000024CD, U"\U000024CD\U000024E7\U000024CD" },
{ 0x000024CE, U"\U000024CE\U000024E8\U000024CE" },
{ 0x000024CF, U"\U000024CF\U000024E9\U000024CF" },
{ 0x000024D0, U"\U000024B6\U000024D0\U000024B6" },
{ 0x000024D1, U"\U000024B7\U000024D1\U000024B7" },
{ 0x000024D2, U"\U000024B8\U000024D2\U000024B8" },
{ 0x000024D3, U"\U000024B9\U000024D3\U000024B9" },
{ 0x000024D4, U"\U000024BA\U000024D4\U000024BA" },
{ 0x000024D5, U"\U000024BB\U000024D5\U000024BB" },
{ 0x000024D6, U"\U000024BC\U000024D6\U000024BC" },
{ 0x000024D7, U"\U000024BD\U000024D7\U000024BD" },
{ 0x000024D8, U"\U000024BE\U000024D8\U000024BE" },
{ 0x000024D9, U"\U000024BF\U000024D9\U000024BF" },
{ 0x000024DA, U"\U000024C0\U000024DA\U000024C0" },
{ 0x000024DB, U"\U000024C1\U000024DB\U000024C1" },
{ 0x000024DC, U"\U000024C2\U000024DC\U000024C2" },
{ 0x000024DD, U"\U000024C3\U000024DD\U000024C3" },
{ 0x000024DE, U"\U000024C4\U000024DE\U000024C4" },
{ 0x000024DF, U"\U000024C5\U000024DF\U000024C5" },
{ 0x000024E0, U"\U000024C6\U000024E0\U000024C6" },
{ 0x000024E1, U"\U000024C7\U000024E1\U000024C7" },
{ 0x000024E2, U"\U000024C8\U000024E2\U000024C8" },
{ 0x000024E3, U"\U000024C9\U000024E3\U000024C9" },
{ 0x000024E4, U"\U000024CA\U000024E4\U000024CA" },
{ 0x000024E5, U"\U000024CB\U000024E5\U000024CB" },
{ 0x000024E6, U"\U000024CC\U000024E6\U000024CC" },
{ 0x000024E7, U"\U000024CD\U000024E7\U000024CD" },
{ 0x000024E8, U"\U000024CE\U000024E8\U000024CE" },
{ 0x000024E9, U"\U000024CF\U000024E9\U000024CF" },
{ 0x00002C00, U"\U00002C00\U00002C30\U00002C00" },
{ 0x00002C01, U"\U00002C01\U00002C31\U00002C01" },
{ 0x00002C02, U"\U00002C02\U00002C32\U00002C02" },
{ 0x00002C03, U"\U00002C03\U00002C33\U00002C03" },
{ 0x00002C04, U"\U00002C04\U00002C34\U00002C04" },
{ 0x00002C05, U"\U00002C05\U00002C35\U00002C05" },
{ 0x00002C06, U"\U00002C06\U00002C36\U00002C06" },
{ 0x00002C07, U"\U00002C07\U00002C37\U00002C07" },
{ 0x00002C08, U"\U00002C08\U00002C38\U00002C08" },
{ 0x00002C09, U"\U00002C09\U00002C39\U00002C09" },
{ 0x00002C0A, U"\U00002C0A\U00002C3A\U00002C0A" },
{ 0x00002C0B, U"\U00002C0B\U00002C3B\U00002C0B" },
{ 0x00002C0C, U"\U00002C0C\U00002C3C\U00002C0C" },
{ 0x00002C0D, U"\U00002C0D\U00002C3D\U00002C0D" },
{ 0x00002C0E, U"\U00002C0E\U00002C3E\U00002C0E" },
{ 0x00002C0F, U"\U00002C0F\U00002C3F\U00002C0F" },
{ 0x00002C10, U"\U00002C10\U00002C40\U00002C10" },
{ 0x00002C11, U"\U00002C11\U00002C41\U00002C11" },
{ 0x00002C12, U"\U00002C12\U00002C42\U00002C12" },
{ 0x00002C13, U"\U00002C13\U00002C43\U00002C13" },
{ 0x00002C14, U"\U00002C14\U00002C44\U00002C14" },
{ 0x00002C15, U"\U00002C15\U00002C45\U00002C15" },
{ 0x00002C16, U"\U00002C16\U00002C46\U00002C16" },
{ 0x00002C17, U"\U00002C17\U00002C47\U00002C17" },
{ 0x00002C18, U"\U00002C18\U00002C48\U00002C18" },
{ 0x00002C19, U"\U00002C19\U00002C49\U00002C19" },
{ 0x00002C1A, U"\U00002C1A\U00002C4A\U00002C1A" },
{ 0x00002C1B, U"\U00002C1B\U00002C4B\U00002C1B" },
{ 0x00002C1C, U"\U00002C1C\U00002C4C\U00002C1C" },
{ 0x00002C1D, U"\U00002C1D\U00002C4D\U00002C1D" },
{ 0x00002C1E, U"\U00002C1E\U00002C4E\U00002C1E" },
{ 0x00002C1F, U"\U00002C1F\U00002C4F\U00002C1F" },
{ 0x00002C20, U"\U00002C20\U00002C50\U00002C20" },
{ 0x00002C21, U"\U00002C21\U00002C51\U00002C21" },
{ 0x00002C22, U"\U00002C22\U00002C52\U00002C22" },
{ 0x00002C23, U"\U00002C23\U00002C53\U00002C23" },
{ 0x00002C24, U"\U00002C24\U00002C54\U00002C24" },
{ 0x00002C25, U"\U00002C25\U00002C55\U00002C25" },
{ 0x00002C26, U"\U00002C26\U00002C56\U00002C26" },
{ 0x00002C27, U"\U00002C27\U00002C57\U00002C27" },
{ 0x00002C28, U"\U00002C28\U00002C58\U00002C28" },
{ 0x00002C29, U"\U00002C29\U00002C59\U00002C29" },
{ 0x00002C2A, U"\U00002C2A\U00002C5A\U00002C2A" },
{ 0x00002C2B, U"\U00002C2B\U00002C5B\U00002C2B" },
{ 0x00002C2C, U"\U00002C2C\U00002C5C\U00002C2C" },
{ 0x00002C2D, U"\U00002C2D\U00002C5D\U00002C2D" },
{ 0x00002C2E, U"\U00002C2E\U00002C5E\U00002C2E" },
{ 0x00002C30, U"\U00002C00\U00002C30\U00002C00" },
{ 0x00002C31, U"\U00002C01\U00002C31\U00002C01" },
{ 0x00002C32, U"\U00002C02\U00002C32\U00002C02" },
{ 0x00002C33, U"\U00002C03\U00002C33\U00002C03" },
{ 0x00002C34, U"\U00002C04\U00002C34\U00002C04" },
{ 0x00002C35, U"\U00002C05\U00002C35\U00002C05" },
{ 0x00002C36, U"\U00002C06\U00002C36\U00002C06" },
{ 0x00002C37, U"\U00002C07\U00002C37\U00002C07" },
{ 0x00002C38, U"\U00002C08\U00002C38\U00002C08" },
{ 0x00002C39, U"\U00002C09\U00002C39\U00002C09" },
{ 0x00002C3A, U"\U00002C0A\U00002C3A\U00002C0A" },
{ 0x00002C3B, U"\U00002C0B\U00002C3B\U00002C0B" },
{ 0x00002C3C, U"\U00002C0C\U00002C3C\U00002C0C" },
{ 0x00002C3D, U"\U00002C0D\U00002C3D\U00002C0D" },
{ 0x00002C3E, U"\U00002C0E\U00002C3E\U00002C0E" },
{ 0x00002C3F, U"\U00002C0F\U00002C3F\U00002C0F" },
{ 0x00002C40, U"\U00002C10\U00002C40\U00002C10" },
{ 0x00002C41, U"\U00002C11\U00002C41\U00002C11" },
{ 0x00002C42, U"\U00002C12\U00002C42\U00002C12" },
{ 0x00002C43, U"\U00002C13\U00002C43\U00002C13" },
{ 0x00002C44, U"\U00002C14\U00002C44\U00002C14" },
{ 0x00002C45, U"\U00002C15\U00002C45\U00002C15" },
{ 0x00002C46, U"\U00002C16\U00002C46\U00002C16" },
{ 0x00002C47, U"\U00002C17\U00002C47\U00002C17" },
{ 0x00002C48, U"\U00002C18\U00002C48\U00002C18" },
{ 0x00002C49, U"\U00002C19\U00002C49\U00002C19" },
{ 0x00002C4A, U"\U00002C1A\U00002C4A\U00002C1A" },
{ 0x00002C4B, U"\U00002C1B\U00002C4B\U00002C1B" },
{ 0x00002C4C, U"\U00002C1C\U00002C4C\U00002C1C" },
{ 0x00002C4D, U"\U00002C1D\U00002C4D\U00002C1D" },
{ 0x00002C4E, U"\U00002C1E\U00002C4E\U00002C1E" },
{ 0x00002C4F, U"\U00002C1F\U00002C4F\U00002C1F" },
{ 0x00002C50, U"\U00002C20\U00002C50\U00002C20" },
{ 0x00002C51, U"\U00002C21\U00002C51\U00002C21" },
{ 0x00002C52, U"\U00002C22\U00002C52\U00002C22" },
{ 0x00002C53, U"\U00002C23\U00002C53\U00002C23" },
{ 0x00002C54, U"\U00002C24\U00002C54\U00002C24" },
{ 0x00002C55, U"\U00002C25\U00002C55\U00002C25" },
{ 0x00002C56, U"\U00002C26\U00002C56\U00002C26" },
{ 0x00002C57, U"\U00002C27\U00002C57\U00002C27" },
{ 0x00002C58, U"\U00002C28\U00002C58\U00002C28" },
{ 0x00002C59, U"\U00002C29\U00002C59\U00002C29" },
{ 0x00002C5A, U"\U00002C2A\U00002C5A\U00002C2A" },
{ 0x00002C5B, U"\U00002C2B\U00002C5B\U00002C2B" },
{ 0x00002C5C, U"\U00002C2C\U00002C5C\U00002C2C" },
{ 0x00002C5D, U"\U00002C2D\U00002C5D\U00002C2D" },
{ 0x00002C5E, U"\U00002C2E\U00002C5E\U00002C2E" },
{ 0x00002C60, U"\U00002C60\U00002C61\U00002C60" },
{ 0x00002C61, U"\U00002C60\U00002C61\U00002C60" },
{ 0x00002C62, U"\U00002C62\U0000026B\U00002C62" },
{ 0x00002C63, U"\U00002C63\U00001D7D\U00002C63" },
{ 0x00002C64, U"\U00002C64\U0000027D\U00002C64" },
{ 0x00002C65, U"\U0000023A\U00002C65\U0000023A" },
{ 0x00002C66, U"\U0000023E\U00002C66\U0000023E" },
{ 0x00002C67, U"\U00002C67\U00002C68\U00002C67" },
{ 0x00002C68, U"\U00002C67\U00002C68\U00002C67" },
{ 0x00002C69, U"\U00002C69\U00002C6A\U00002C69" },
{ 0x00002C6A, U"\U00002C69\U00002C6A\U00002C69" },
{ 0x00002C6B, U"\U00002C6B\U00002C6C\U00002C6B" },
{ 0x00002C6C, U"\U00002C6B\U00002C6C\U00002C6B" },
{ 0x00002C6D, U"\U00002C6D\U00000251\U00002C6D" },
{ 0x00002C6E, U"\U00002C6E\U00000271\U00002C6E" },
{ 0x00002C6F, U"\U00002C6F\U00000250\U00002C6F" },
{ 0x00002C70, U"\U00002C70\U00000252\U00002C70" },
{ 0x00002C72, U"\U00002C72\U00002C73\U00002C72" },
{ 0x00002C73, U"\U00002C72\U00002C73\U00002C72" },
{ 0x00002C75, U"\U00002C75\U00002C76\U00002C75" },
{ 0x00002C76, U"\U00002C75\U00002C76\U00002C75" },
{ 0x00002C7E, U"\U00002C7E\U0000023F\U00002C7E" },
{ 0x00002C7F, U"\U00002C7F\U00000240\U00002C7F" },
{ 0x00002C80, U"\U00002C80\U00002C81\U00002C80" },
{ 0x00002C81, U"\U00002C80\U00002C81\U00002C80" },
{ 0x00002C82, U"\U00002C82\U00002C83\U00002C82" },
{ 0x00002C83, U"\U00002C82\U00002C83\U00002C82" },
{ 0x00002C84, U"\U00002C84\U00002C85\U00002C84" },
{ 0x00002C85, U"\U00002C84\U00002C85\U00002C84" },
{ 0x00002C86, U"\U00002C86\U00002C87\U00002C86" },
{ 0x00002C87, U"\U00002C86\U00002C87\U00002C86" },
{ 0x00002C88, U"\U00002C88\U00002C89\U00002C88" },
{ 0x00002C89, U"\U00002C88\U00002C89\U00002C88" },
{ 0x00002C8A, U"\U00002C8A\U00002C8B\U00002C8A" },
{ 0x00002C8B, U"\U00002C8A\U00002C8B\U00002C8A" },
{ 0x00002C8C, U"\U00002C8C\U00002C8D\U00002C8C" },
{ 0x00002C8D, U"\U00002C8C\U00002C8D\U00002C8C" },
{ 0x00002C8E, U"\U00002C8E\U00002C8F\U00002C8E" },
{ 0x00002C8F, U"\U00002C8E\U00002C8F\U00002C8E" },
{ 0x00002C90, U"\U00002C90\U00002C91\U00002C90" },
{ 0x00002C91, U"\U00002C90\U00002C91\U00002C90" },
{ 0x00002C92, U"\U00002C92\U00002C93\U00002C92" },
{ 0x00002C93, U"\U00002C92\U00002C93\U00002C92" },
{ 0x00002C94, U"\U00002C94\U00002C95\U00002C94" },
{ 0x00002C95, U"\U00002C94\U00002C95\U00002C94" },
{ 0x00002C96, U"\U00002C96\U00002C97\U00002C96" },
{ 0x00002C97, U"\U00002C96\U00002C97\U00002C96" },
{ 0x00002C98, U"\U00002C98\U00002C99\U00002C98" },
{ 0x00002C99, U"\U00002C98\U00002C99\U00002C98" },
{ 0x00002C9A, U"\U00002C9A\U00002C9B\U00002C9A" },
{ 0x00002C9B, U"\U00002C9A\U00002C9B\U00002C9A" },
{ 0x00002C9C, U"\U00002C9C\U00002C9D\U00002C9C" },
{ 0x00002C9D, U"\U00002C9C\U00002C9D\U00002C9C" },
{ 0x00002C9E, U"\U00002C9E\U00002C9F\U00002C9E" },
{ 0x00002C9F, U"\U00002C9E\U00002C9F\U00002C9E" },
{ 0x00002CA0, U"\U00002CA0\U00002CA1\U00002CA0" },
{ 0x00002CA1, U"\U00002CA0\U00002CA1\U00002CA0" },
{ 0x00002CA2, U"\U00002CA2\U00002CA3\U00002CA2" },
{ 0x00002CA3, U"\U00002CA2\U00002CA3\U00002CA2" },
{ 0x00002CA4, U"\U00002CA4\U00002CA5\U00002CA4" },
{ 0x00002CA5, U"\U00002CA4\U00002CA5\U00002CA4" },
{ 0x00002CA6, U"\U00002CA6\U00002CA7\U00002CA6" },
{ 0x00002CA7, U"\U00002CA6\U00002CA7\U00002CA6" },
{ 0x00002CA8, U"\U00002CA8\U00002CA9\U00002CA8" },
{ 0x00002CA9, U"\U00002CA8\U00002CA9\U00002CA8" },
{ 0x00002CAA, U"\U00002CAA\U00002CAB\U00002CAA" },
{ 0x00002CAB, U"\U00002CAA\U00002CAB\U00002CAA" },
{ 0x00002CAC, U"\U00002CAC\U00002CAD\U00002CAC" },
{ 0x00002CAD, U"\U00002CAC\U00002CAD\U00002CAC" },
{ 0x00002CAE, U"\U00002CAE\U00002CAF\U00002CAE" },
{ 0x00002CAF, U"\U00002CAE\U00002CAF\U00002CAE" },
{ 0x00002CB0, U"\U00002CB0\U00002CB1\U00002CB0" },
{ 0x00002CB1, U"\U00002CB0\U00002CB1\U00002CB0" },
{ 0x00002CB2, U"\U00002CB2\U00002CB3\U00002CB2" },
{ 0x00002CB3, U"\U00002CB2\U00002CB3\U00002CB2" },
{ 0x00002CB4, U"\U00002CB4\U00002CB5\U00002CB4" },
{ 0x00002CB5, U"\U00002CB4\U00002CB5\U00002CB4" },
{ 0x00002CB6, U"\U00002CB6\U00002CB7\U00002CB6" },
{ 0x00002CB7, U"\U00002CB6\U00002CB7\U00002CB6" },
{ 0x00002CB8, U"\U00002CB8\U00002CB9\U00002CB8" },
{ 0x00002CB9, U"\U00002CB8\U00002CB9\U00002CB8" },
{ 0x00002CBA, U"\U00002CBA\U00002CBB\U00002CBA" },
{ 0x00002CBB, U"\U00002CBA\U00002CBB\U00002CBA" },
{ 0x00002CBC, U"\U00002CBC\U00002CBD\U00002CBC" },
{ 0x00002CBD, U"\U00002CBC\U00002CBD\U00002CBC" },
{ 0x00002CBE, U"\U00002CBE\U00002CBF\U00002CBE" },
{ 0x00002CBF, U"\U00002CBE\U00002CBF\U00002CBE" },
{ 0x00002CC0, U"\U00002CC0\U00002CC1\U00002CC0" },
{ 0x00002CC1, U"\U00002CC0\U00002CC1\U00002CC0" },
{ 0x00002CC2, U"\U00002CC2\U00002CC3\U00002CC2" },
{ 0x00002CC3, U"\U00002CC2\U00002CC3\U00002CC2" },
{ 0x00002CC4, U"\U00002CC4\U00002CC5\U00002CC4" },
{ 0x00002CC5, U"\U00002CC4\U00002CC5\U00002CC4" },
{ 0x00002CC6, U"\U00002CC6\U00002CC7\U00002CC6" },
{ 0x00002CC7, U"\U00002CC6\U00002CC7\U00002CC6" },
{ 0x00002CC8, U"\U00002CC8\U00002CC9\U00002CC8" },
{ 0x00002CC9, U"\U00002CC8\U00002CC9\U00002CC8" },
{ 0x00002CCA, U"\U00002CCA\U00002CCB\U00002CCA" },
{ 0x00002CCB, U"\U00002CCA\U00002CCB\U00002CCA" },
{ 0x00002CCC, U"\U00002CCC\U00002CCD\U00002CCC" },
{ 0x00002CCD, U"\U00002CCC\U00002CCD\U00002CCC" },
{ 0x00002CCE, U"\U00002CCE\U00002CCF\U00002CCE" },
{ 0x00002CCF, U"\U00002CCE\U00002CCF\U00002CCE" },
{ 0x00002CD0, U"\U00002CD0\U00002CD1\U00002CD0" },
{ 0x00002CD1, U"\U00002CD0\U00002CD1\U00002CD0" },
{ 0x00002CD2, U"\U00002CD2\U00002CD3\U00002CD2" },
{ 0x00002CD3, U"\U00002CD2\U00002CD3\U00002CD2" },
{ 0x00002CD4, U"\U00002CD4\U00002CD5\U00002CD4" },
{ 0x00002CD5, U"\U00002CD4\U00002CD5\U00002CD4" },
{ 0x00002CD6, U"\U00002CD6\U00002CD7\U00002CD6" },
{ 0x00002CD7, U"\U00002CD6\U00002CD7\U00002CD6" },
{ 0x00002CD8, U"\U00002CD8\U00002CD9\U00002CD8" },
{ 0x00002CD9, U"\U00002CD8\U00002CD9\U00002CD8" },
{ 0x00002CDA, U"\U00002CDA\U00002CDB\U00002CDA" },
{ 0x00002CDB, U"\U00002CDA\U00002CDB\U00002CDA" },
{ 0x00002CDC, U"\U00002CDC\U00002CDD\U00002CDC" },
{ 0x00002CDD, U"\U00002CDC\U00002CDD\U00002CDC" },
{ 0x00002CDE, U"\U00002CDE\U00002CDF\U00002CDE" },
{ 0x00002CDF, U"\U00002CDE\U00002CDF\U00002CDE" },
{ 0x00002CE0, U"\U00002CE0\U00002CE1\U00002CE0" },
{ 0x00002CE1, U"\U00002CE0\U00002CE1\U00002CE0" },
{ 0x00002CE2, U"\U00002CE2\U00002CE3\U00002CE2" },
{ 0x00002CE3, U"\U00002CE2\U00002CE3\U00002CE2" },
{ 0x00002CEB, U"\U00002CEB\U00002CEC\U00002CEB" },
{ 0x00002CEC, U"\U00002CEB\U00002CEC\U00002CEB" },
{ 0x00002CED, U"\U00002CED\U00002CEE\U00002CED" },
{ 0x00002CEE, U"\U00002CED\U00002CEE\U00002CED" },
{ 0x00002CF2, U"\U00002CF2\U00002CF3\U00002CF2" },
{ 0x00002CF3, U"\U00002CF2\U00002CF3\U00002CF2" },
{ 0x00002D00, U"\U000010A0\U00002D00\U000010A0" },
{ 0x00002D01, U"\U000010A1\U00002D01\U000010A1" },
{ 0x00002D02, U"\U000010A2\U00002D02\U000010A2" },
{ 0x00002D03, U"\U000010A3\U00002D03\U000010A3" },
{ 0x00002D04, U"\U000010A4\U00002D04\U000010A4" },
{ 0x00002D05, U"\U000010A5\U00002D05\U000010A5" },
{ 0x00002D06, U"\U000010A6\U00002D06\U000010A6" },
{ 0x00002D07, U"\U000010A7\U00002D07\U000010A7" },
{ 0x00002D08, U"\U000010A8\U00002D08\U000010A8" },
{ 0x00002D09, U"\U000010A9\U00002D09\U000010A9" },
{ 0x00002D0A, U"\U000010AA\U00002D0A\U000010AA" },
{ 0x00002D0B, U"\U000010AB\U00002D0B\U000010AB" },
{ 0x00002D0C, U"\U000010AC\U00002D0C\U000010AC" },
{ 0x00002D0D, U"\U000010AD\U00002D0D\U000010AD" },
{ 0x00002D0E, U"\U000010AE\U00002D0E\U000010AE" },
{ 0x00002D0F, U"\U000010AF\U00002D0F\U000010AF" },
{ 0x00002D10, U"\U000010B0\U00002D10\U000010B0" },
{ 0x00002D11, U"\U000010B1\U00002D11\U000010B1" },
{ 0x00002D12, U"\U000010B2\U00002D12\U000010B2" },
{ 0x00002D13, U"\U000010B3\U00002D13\U000010B3" },
{ 0x00002D14, U"\U000010B4\U00002D14\U000010B4" },
{ 0x00002D15, U"\U000010B5\U00002D15\U000010B5" },
{ 0x00002D16, U"\U000010B6\U00002D16\U000010B6" },
{ 0x00002D17, U"\U000010B7\U00002D17\U000010B7" },
{ 0x00002D18, U"\U000010B8\U00002D18\U000010B8" },
{ 0x00002D19, U"\U000010B9\U00002D19\U000010B9" },
{ 0x00002D1A, U"\U000010BA\U00002D1A\U000010BA" },
{ 0x00002D1B, U"\U000010BB\U00002D1B\U000010BB" },
{ 0x00002D1C, U"\U000010BC\U00002D1C\U000010BC" },
{ 0x00002D1D, U"\U000010BD\U00002D1D\U000010BD" },
{ 0x00002D1E, U"\U000010BE\U00002D1E\U000010BE" },
{ 0x00002D1F, U"\U000010BF\U00002D1F\U000010BF" },
{ 0x00002D20, U"\U000010C0\U00002D20\U000010C0" },
{ 0x00002D21, U"\U000010C1\U00002D21\U000010C1" },
{ 0x00002D22, U"\U000010C2\U00002D22\U000010C2" },
{ 0x00002D23, U"\U000010C3\U00002D23\U000010C3" },
{ 0x00002D24, U"\U000010C4\U00002D24\U000010C4" },
{ 0x00002D25, U"\U000010C5\U00002D25\U000010C5" },
{ 0x00002D27, U"\U000010C7\U00002D27\U000010C7" },
{ 0x00002D2D, U"\U000010CD\U00002D2D\U000010CD" },
{ 0x0000A640, U"\U0000A640\U0000A641\U0000A640" },
{ 0x0000A641, U"\U0000A640\U0000A641\U0000A640" },
{ 0x0000A642, U"\U0000A642\U0000A643\U0000A642" },
{ 0x0000A643, U"\U0000A642\U0000A643\U0000A642" },
{ 0x0000A644, U"\U0000A644\U0000A645\U0000A644" },
{ 0x0000A645, U"\U0000A644\U0000A645\U0000A644" },
{ 0x0000A646, U"\U0000A646\U0000A647\U0000A646" },
{ 0x0000A647, U"\U0000A646\U0000A647\U0000A646" },
{ 0x0000A648, U"\U0000A648\U0000A649\U0000A648" },
{ 0x0000A649, U"\U0000A648\U0000A649\U0000A648" },
{ 0x0000A64A, U"\U0000A64A\U0000A64B\U0000A64A" },
{ 0x0000A64B, U"\U0000A64A\U0000A64B\U0000A64A" },
{ 0x0000A64C, U"\U0000A64C\U0000A64D\U0000A64C" },
{ 0x0000A64D, U"\U0000A64C\U0000A64D\U0000A64C" },
{ 0x0000A64E, U"\U0000A64E\U0000A64F\U0000A64E" },
{ 0x0000A64F, U"\U0000A64E\U0000A64F\U0000A64E" },
{ 0x0000A650, U"\U0000A650\U0000A651\U0000A650" },
{ 0x0000A651, U"\U0000A650\U0000A651\U0000A650" },
{ 0x0000A652, U"\U0000A652\U0000A653\U0000A652" },
{ 0x0000A653, U"\U0000A652\U0000A653\U0000A652" },
{ 0x0000A654, U"\U0000A654\U0000A655\U0000A654" },
{ 0x0000A655, U"\U0000A654\U0000A655\U0000A654" },
{ 0x0000A656, U"\U0000A656\U0000A657\U0000A656" },
{ 0x0000A657, U"\U0000A656\U0000A657\U0000A656" },
{ 0x0000A658, U"\U0000A658\U0000A659\U0000A658" },
{ 0x0000A659, U"\U0000A658\U0000A659\U0000A658" },
{ 0x0000A65A, U"\U0000A65A\U0000A65B\U0000A65A" },
{ 0x0000A65B, U"\U0000A65A\U0000A65B\U0000A65A" },
{ 0x0000A65C, U"\U0000A65C\U0000A65D\U0000A65C" },
{ 0x0000A65D, U"\U0000A65C\U0000A65D\U0000A65C" },
{ 0x0000A65E, U"\U0000A65E\U0000A65F\U0000A65E" },
{ 0x0000A65F, U"\U0000A65E\U0000A65F\U0000A65E" },
{ 0x0000A660, U"\U0000A660\U0000A661\U0000A660" },
{ 0x0000A661, U"\U0000A660\U0000A661\U0000A660" },
{ 0x0000A662, U"\U0000A662\U0000A663\U0000A662" },
{ 0x0000A663, U"\U0000A662\U0000A663\U0000A662" },
{ 0x0000A664, U"\U0000A664\U0000A665\U0000A664" },
{ 0x0000A665, U"\U0000A664\U0000A665\U0000A664" },
{ 0x0000A666, U"\U0000A666\U0000A667\U0000A666" },
{ 0x0000A667, U"\U0000A666\U0000A667\U0000A666" },
{ 0x0000A668, U"\U0000A668\U0000A669\U0000A668" },
{ 0x0000A669, U"\U0000A668\U0000A669\U0000A668" },
{ 0x0000A66A, U"\U0000A66A\U0000A66B\U0000A66A" },
{ 0x0000A66B, U"\U0000A66A\U0000A66B\U0000A66A" },
{ 0x0000A66C, U"\U0000A66C\U0000A66D\U0000A66C" },
{ 0x0000A66D, U"\U0000A66C\U0000A66D\U0000A66C" },
{ 0x0000A680, U"\U0000A680\U0000A681\U0000A680" },
{ 0x0000A681, U"\U0000A680\U0000A681\U0000A680" },
{ 0x0000A682, U"\U0000A682\U0000A683\U0000A682" },
{ 0x0000A683, U"\U0000A682\U0000A683\U0000A682" },
{ 0x0000A684, U"\U0000A684\U0000A685\U0000A684" },
{ 0x0000A685, U"\U0000A684\U0000A685\U0000A684" },
{ 0x0000A686, U"\U0000A686\U0000A687\U0000A686" },
{ 0x0000A687, U"\U0000A686\U0000A687\U0000A686" },
{ 0x0000A688, U"\U0000A688\U0000A689\U0000A688" },
{ 0x0000A689, U"\U0000A688\U0000A689\U0000A688" },
{ 0x0000A68A, U"\U0000A68A\U0000A68B\U0000A68A" },
{ 0x0000A68B, U"\U0000A68A\U0000A68B\U0000A68A" },
{ 0x0000A68C, U"\U0000A68C\U0000A68D\U0000A68C" },
{ 0x0000A68D, U"\U0000A68C\U0000A68D\U0000A68C" },
{ 0x0000A68E, U"\U0000A68E\U0000A68F\U0000A68E" },
{ 0x0000A68F, U"\U0000A68E\U0000A68F\U0000A68E" },
{ 0x0000A690, U"\U0000A690\U0000A691\U0000A690" },
{ 0x0000A691, U"\U0000A690\U0000A691\U0000A690" },
{ 0x0000A692, U"\U0000A692\U0000A693\U0000A692" },
{ 0x0000A693, U"\U0000A692\U0000A693\U0000A692" },
{ 0x0000A694, U"\U0000A694\U0000A695\U0000A694" },
{ 0x0000A695, U"\U0000A694\U0000A695\U0000A694" },
{ 0x0000A696, U"\U0000A696\U0000A697\U0000A696" },
{ 0x0000A697, U"\U0000A696\U0000A697\U0000A696" },
{ 0x0000A698, U"\U0000A698\U0000A699\U0000A698" },
{ 0x0000A699, U"\U0000A698\U0000A699\U0000A698" },
{ 0x0000A69A, U"\U0000A69A\U0000A69B\U0000A69A" },
{ 0x0000A69B, U"\U0000A69A\U0000A69B\U0000A69A" },
{ 0x0000A722, U"\U0000A722\U0000A723\U0000A722" },
{ 0x0000A723, U"\U0000A722\U0000A723\U0000A722" },
{ 0x0000A724, U"\U0000A724\U0000A725\U0000A724" },
{ 0x0000A725, U"\U0000A724\U0000A725\U0000A724" },
{ 0x0000A726, U"\U0000A726\U0000A727\U0000A726" },
{ 0x0000A727, U"\U0000A726\U0000A727\U0000A726" },
{ 0x0000A728, U"\U0000A728\U0000A729\U0000A728" },
{ 0x0000A729, U"\U0000A728\U0000A729\U0000A728" },
{ 0x0000A72A, U"\U0000A72A\U0000A72B\U0000A72A" },
{ 0x0000A72B, U"\U0000A72A\U0000A72B\U0000A72A" },
{ 0x0000A72C, U"\U0000A72C\U0000A72D\U0000A72C" },
{ 0x0000A72D, U"\U0000A72C\U0000A72D\U0000A72C" },
{ 0x0000A72E, U"\U0000A72E\U0000A72F\U0000A72E" },
{ 0x0000A72F, U"\U0000A72E\U0000A72F\U0000A72E" },
{ 0x0000A732, U"\U0000A732\U0000A733\U0000A732" },
{ 0x0000A733, U"\U0000A732\U0000A733\U0000A732" },
{ 0x0000A734, U"\U0000A734\U0000A735\U0000A734" },
{ 0x0000A735, U"\U0000A734\U0000A735\U0000A734" },
{ 0x0000A736, U"\U0000A736\U0000A737\U0000A736" },
{ 0x0000A737, U"\U0000A736\U0000A737\U0000A736" },
{ 0x0000A738, U"\U0000A738\U0000A739\U0000A738" },
{ 0x0000A739, U"\U0000A738\U0000A739\U0000A738" },
{ 0x0000A73A, U"\U0000A73A\U0000A73B\U0000A73A" },
{ 0x0000A73B, U"\U0000A73A\U0000A73B\U0000A73A" },
{ 0x0000A73C, U"\U0000A73C\U0000A73D\U0000A73C" },
{ 0x0000A73D, U"\U0000A73C\U0000A73D\U0000A73C" },
{ 0x0000A73E, U"\U0000A73E\U0000A73F\U0000A73E" },
{ 0x0000A73F, U"\U0000A73E\U0000A73F\U0000A73E" },
{ 0x0000A740, U"\U0000A740\U0000A741\U0000A740" },
{ 0x0000A741, U"\U0000A740\U0000A741\U0000A740" },
{ 0x0000A742, U"\U0000A742\U0000A743\U0000A742" },
{ 0x0000A743, U"\U0000A742\U0000A743\U0000A742" },
{ 0x0000A744, U"\U0000A744\U0000A745\U0000A744" },
{ 0x0000A745, U"\U0000A744\U0000A745\U0000A744" },
{ 0x0000A746, U"\U0000A746\U0000A747\U0000A746" },
{ 0x0000A747, U"\U0000A746\U0000A747\U0000A746" },
{ 0x0000A748, U"\U0000A748\U0000A749\U0000A748" },
{ 0x0000A749, U"\U0000A748\U0000A749\U0000A748" },
{ 0x0000A74A, U"\U0000A74A\U0000A74B\U0000A74A" },
{ 0x0000A74B, U"\U0000A74A\U0000A74B\U0000A74A" },
{ 0x0000A74C, U"\U0000A74C\U0000A74D\U0000A74C" },
{ 0x0000A74D, U"\U0000A74C\U0000A74D\U0000A74C" },
{ 0x0000A74E, U"\U0000A74E\U0000A74F\U0000A74E" },
{ 0x0000A74F, U"\U0000A74E\U0000A74F\U0000A74E" },
{ 0x0000A750, U"\U0000A750\U0000A751\U0000A750" },
{ 0x0000A751, U"\U0000A750\U0000A751\U0000A750" },
{ 0x0000A752, U"\U0000A752\U0000A753\U0000A752" },
{ 0x0000A753, U"\U0000A752\U0000A753\U0000A752" },
{ 0x0000A754, U"\U0000A754\U0000A755\U0000A754" },
{ 0x0000A755, U"\U0000A754\U0000A755\U0000A754" },
{ 0x0000A756, U"\U0000A756\U0000A757\U0000A756" },
{ 0x0000A757, U"\U0000A756\U0000A757\U0000A756" },
{ 0x0000A758, U"\U0000A758\U0000A759\U0000A758" },
{ 0x0000A759, U"\U0000A758\U0000A759\U0000A758" },
{ 0x0000A75A, U"\U0000A75A\U0000A75B\U0000A75A" },
{ 0x0000A75B, U"\U0000A75A\U0000A75B\U0000A75A" },
{ 0x0000A75C, U"\U0000A75C\U0000A75D\U0000A75C" },
{ 0x0000A75D, U"\U0000A75C\U0000A75D\U0000A75C" },
{ 0x0000A75E, U"\U0000A75E\U0000A75F\U0000A75E" },
{ 0x0000A75F, U"\U0000A75E\U0000A75F\U0000A75E" },
{ 0x0000A760, U"\U0000A760\U0000A761\U0000A760" },
{ 0x0000A761, U"\U0000A760\U0000A761\U0000A760" },
{ 0x0000A762, U"\U0000A762\U0000A763\U0000A762" },
{ 0x0000A763, U"\U0000A762\U0000A763\U0000A762" },
{ 0x0000A764, U"\U0000A764\U0000A765\U0000A764" },
{ 0x0000A765, U"\U0000A764\U0000A765\U0000A764" },
{ 0x0000A766, U"\U0000A766\U0000A767\U0000A766" },
{ 0x0000A767, U"\U0000A766\U0000A767\U0000A766" },
{ 0x0000A768, U"\U0000A768\U0000A769\U0000A768" },
{ 0x0000A769, U"\U0000A768\U0000A769\U0000A768" },
{ 0x0000A76A, U"\U0000A76A\U0000A76B\U0000A76A" },
{ 0x0000A76B, U"\U0000A76A\U0000A76B\U0000A76A" },
{ 0x0000A76C, U"\U0000A76C\U0000A76D\U0000A76C" },
{ 0x0000A76D, U"\U0000A76C\U0000A76D\U0000A76C" },
{ 0x0000A76E, U"\U0000A76E\U0000A76F\U0000A76E" },
{ 0x0000A76F, U"\U0000A76E\U0000A76F\U0000A76E" },
{ 0x0000A779, U"\U0000A779\U0000A77A\U0000A779" },
{ 0x0000A77A, U"\U0000A779\U0000A77A\U0000A779" },
{ 0x0000A77B, U"\U0000A77B\U0000A77C\U0000A77B" },
{ 0x0000A77C, U"\U0000A77B\U0000A77C\U0000A77B" },
{ 0x0000A77D, U"\U0000A77D\U00001D79\U0000A77D" },
{ 0x0000A77E, U"\U0000A77E\U0000A77F\U0000A77E" },
{ 0x0000A77F, U"\U0000A77E\U0000A77F\U0000A77E" },
{ 0x0000A780, U"\U0000A780\U0000A781\U0000A780" },
{ 0x0000A781, U"\U0000A780\U0000A781\U0000A780" },
{ 0x0000A782, U"\U0000A782\U0000A783\U0000A782" },
{ 0x0000A783, U"\U0000A782\U0000A783\U0000A782" },
{ 0x0000A784, U"\U0000A784\U0000A785\U0000A784" },
{ 0x0000A785, U"\U0000A784\U0000A785\U0000A784" },
{ 0x0000A786, U"\U0000A786\U0000A787\U0000A786" },
{ 0x0000A787, U"\U0000A786\U0000A787\U0000A786" },
{ 0x0000A78B, U"\U0000A78B\U0000A78C\U0000A78B" },
{ 0x0000A78C, U"\U0000A78B\U0000A78C\U0000A78B" },
{ 0x0000A78D, U"\U0000A78D\U00000265\U0000A78D" },
{ 0x0000A790, U"\U0000A790\U0000A791\U0000A790" },
{ 0x0000A791, U"\U0000A790\U0000A791\U0000A790" },
{ 0x0000A792, U"\U0000A792\U0000A793\U0000A792" },
{ 0x0000A793, U"\U0000A792\U0000A793\U0000A792" },
{ 0x0000A794, U"\U0000A7C4\U0000A794\U0000A7C4" },
{ 0x0000A796, U"\U0000A796\U0000A797\U0000A796" },
{ 0x0000A797, U"\U0000A796\U0000A797\U0000A796" },
{ 0x0000A798, U"\U0000A798\U0000A799\U0000A798" },
{ 0x0000A799, U"\U0000A798\U0000A799\U0000A798" },
{ 0x0000A79A, U"\U0000A79A\U0000A79B\U0000A79A" },
{ 0x0000A79B, U"\U0000A79A\U0000A79B\U0000A79A" },
{ 0x0000A79C, U"\U0000A79C\U0000A79D\U0000A79C" },
{ 0x0000A79D, U"\U0000A79C\U0000A79D\U0000A79C" },
{ 0x0000A79E, U"\U0000A79E\U0000A79F\U0000A79E" },
{ 0x0000A79F, U"\U0000A79E\U0000A79F\U0000A79E" },
{ 0x0000A7A0, U"\U0000A7A0\U0000A7A1\U0000A7A0" },
{ 0x0000A7A1, U"\U0000A7A0\U0000A7A1\U0000A7A0" },
{ 0x0000A7A2, U"\U0000A7A2\U0000A7A3\U0000A7A2" },
{ 0x0000A7A3, U"\U0000A7A2\U0000A7A3\U0000A7A2" },
{ 0x0000A7A4, U"\U0000A7A4\U0000A7A5\U0000A7A4" },
{ 0x0000A7A5, U"\U0000A7A4\U0000A7A5\U0000A7A4" },
{ 0x0000A7A6, U"\U0000A7A6\U0000A7A7\U0000A7A6" },
{ 0x0000A7A7, U"\U0000A7A6\U0000A7A7\U0000A7A6" },
{ 0x0000A7A8, U"\U0000A7A8\U0000A7A9\U0000A7A8" },
{ 0x0000A7A9, U"\U0000A7A8\U0000A7A9\U0000A7A8" },
{ 0x0000A7AA, U"\U0000A7AA\U00000266\U0000A7AA" },
{ 0x0000A7AB, U"\U0000A7AB\U0000025C\U0000A7AB" },
{ 0x0000A7AC, U"\U0000A7AC\U00000261\U0000A7AC" },
{ 0x0000A7AD, U"\U0000A7AD\U0000026C\U0000A7AD" },
{ 0x0000A7AE, U"\U0000A7AE\U0000026A\U0000A7AE" },
{ 0x0000A7B0, U"\U0000A7B0\U0000029E\U0000A7B0" },
{ 0x0000A7B1, U"\U0000A7B1\U00000287\U0000A7B1" },
{ 0x0000A7B2, U"\U0000A7B2\U0000029D\U0000A7B2" },
{ 0x0000A7B3, U"\U0000A7B3\U0000AB53\U0000A7B3" },
{ 0x0000A7B4, U"\U0000A7B4\U0000A7B5\U0000A7B4" },
{ 0x0000A7B5, U"\U0000A7B4\U0000A7B5\U0000A7B4" },
{ 0x0000A7B6, U"\U0000A7B6\U0000A7B7\U0000A7B6" },
{ 0x0000A7B7, U"\U0000A7B6\U0000A7B7\U0000A7B6" },
{ 0x0000A7B8, U"\U0000A7B8\U0000A7B9\U0000A7B8" },
{ 0x0000A7B9, U"\U0000A7B8\U0000A7B9\U0000A7B8" },
{ 0x0000A7BA, U"\U0000A7BA\U0000A7BB\U0000A7BA" },
{ 0x0000A7BB, U"\U0000A7BA\U0000A7BB\U0000A7BA" },
{ 0x0000A7BC, U"\U0000A7BC\U0000A7BD\U0000A7BC" },
{ 0x0000A7BD, U"\U0000A7BC\U0000A7BD\U0000A7BC" },
{ 0x0000A7BE, U"\U0000A7BE\U0000A7BF\U0000A7BE" },
{ 0x0000A7BF, U"\U0000A7BE\U0000A7BF\U0000A7BE" },
{ 0x0000A7C2, U"\U0000A7C2\U0000A7C3\U0000A7C2" },
{ 0x0000A7C3, U"\U0000A7C2\U0000A7C3\U0000A7C2" },
{ 0x0000A7C4, U"\U0000A7C4\U0000A794\U0000A7C4" },
{ 0x0000A7C5, U"\U0000A7C5\U00000282\U0000A7C5" },
{ 0x0000A7C6, U"\U0000A7C6\U00001D8E\U0000A7C6" },
{ 0x0000A7C7, U"\U0000A7C7\U0000A7C8\U0000A7C7" },
{ 0x0000A7C8, U"\U0000A7C7\U0000A7C8\U0000A7C7" },
{ 0x0000A7C9, U"\U0000A7C9\U0000A7CA\U0000A7C9" },
{ 0x0000A7CA, U"\U0000A7C9\U0000A7CA\U0000A7C9" },
{ 0x0000A7F5, U"\U0000A7F5\U0000A7F6\U0000A7F5" },
{ 0x0000A7F6, U"\U0000A7F5\U0000A7F6\U0000A7F5" },
{ 0x0000AB53, U"\U0000A7B3\U0000AB53\U0000A7B3" },
{ 0x0000AB70, U"\U000013A0\U0000AB70\U000013A0" },
{ 0x0000AB71, U"\U000013A1\U0000AB71\U000013A1" },
{ 0x0000AB72, U"\U000013A2\U0000AB72\U000013A2" },
{ 0x0000AB73, U"\U000013A3\U0000AB73\U000013A3" },
{ 0x0000AB74, U"\U000013A4\U0000AB74\U000013A4" },
{ 0x0000AB75, U"\U000013A5\U0000AB75\U000013A5" },
{ 0x0000AB76, U"\U000013A6\U0000AB76\U000013A6" },
{ 0x0000AB77, U"\U000013A7\U0000AB77\U000013A7" },
{ 0x0000AB78, U"\U000013A8\U0000AB78\U000013A8" },
{ 0x0000AB79, U"\U000013A9\U0000AB79\U000013A9" },
{ 0x0000AB7A, U"\U000013AA\U0000AB7A\U000013AA" },
{ 0x0000AB7B, U"\U000013AB\U0000AB7B\U000013AB" },
{ 0x0000AB7C, U"\U000013AC\U0000AB7C\U000013AC" },
{ 0x0000AB7D, U"\U000013AD\U0000AB7D\U000013AD" },
{ 0x0000AB7E, U"\U000013AE\U0000AB7E\U000013AE" },
{ 0x0000AB7F, U"\U000013AF\U0000AB7F\U000013AF" },
{ 0x0000AB80, U"\U000013B0\U0000AB80\U000013B0" },
{ 0x0000AB81, U"\U000013B1\U0000AB81\U000013B1" },
{ 0x0000AB82, U"\U000013B2\U0000AB82\U000013B2" },
{ 0x0000AB83, U"\U000013B3\U0000AB83\U000013B3" },
{ 0x0000AB84, U"\U000013B4\U0000AB84\U000013B4" },
{ 0x0000AB85, U"\U000013B5\U0000AB85\U000013B5" },
{ 0x0000AB86, U"\U000013B6\U0000AB86\U000013B6" },
{ 0x0000AB87, U"\U000013B7\U0000AB87\U000013B7" },
{ 0x0000AB88, U"\U000013B8\U0000AB88\U000013B8" },
{ 0x0000AB89, U"\U000013B9\U0000AB89\U000013B9" },
{ 0x0000AB8A, U"\U000013BA\U0000AB8A\U000013BA" },
{ 0x0000AB8B, U"\U000013BB\U0000AB8B\U000013BB" },
{ 0x0000AB8C, U"\U000013BC\U0000AB8C\U000013BC" },
{ 0x0000AB8D, U"\U000013BD\U0000AB8D\U000013BD" },
{ 0x0000AB8E, U"\U000013BE\U0000AB8E\U000013BE" },
{ 0x0000AB8F, U"\U000013BF\U0000AB8F\U000013BF" },
{ 0x0000AB90, U"\U000013C0\U0000AB90\U000013C0" },
{ 0x0000AB91, U"\U000013C1\U0000AB91\U000013C1" },
{ 0x0000AB92, U"\U000013C2\U0000AB92\U000013C2" },
{ 0x0000AB93, U"\U000013C3\U0000AB93\U000013C3" },
{ 0x0000AB94, U"\U000013C4\U0000AB94\U000013C4" },
{ 0x0000AB95, U"\U000013C5\U0000AB95\U000013C5" },
{ 0x0000AB96, U"\U000013C6\U0000AB96\U000013C6" },
{ 0x0000AB97, U"\U000013C7\U0000AB97\U000013C7" },
{ 0x0000AB98, U"\U000013C8\U0000AB98\U000013C8" },
{ 0x0000AB99, U"\U000013C9\U0000AB99\U000013C9" },
{ 0x0000AB9A, U"\U000013CA\U0000AB9A\U000013CA" },
{ 0x0000AB9B, U"\U000013CB\U0000AB9B\U000013CB" },
{ 0x0000AB9C, U"\U000013CC\U0000AB9C\U000013CC" },
{ 0x0000AB9D, U"\U000013CD\U0000AB9D\U000013CD" },
{ 0x0000AB9E, U"\U000013CE\U0000AB9E\U000013CE" },
{ 0x0000AB9F, U"\U000013CF\U0000AB9F\U000013CF" },
{ 0x0000ABA0, U"\U000013D0\U0000ABA0\U000013D0" },
{ 0x0000ABA1, U"\U000013D1\U0000ABA1\U000013D1" },
{ 0x0000ABA2, U"\U000013D2\U0000ABA2\U000013D2" },
{ 0x0000ABA3, U"\U000013D3\U0000ABA3\U000013D3" },
{ 0x0000ABA4, U"\U000013D4\U0000ABA4\U000013D4" },
{ 0x0000ABA5, U"\U000013D5\U0000ABA5\U000013D5" },
{ 0x0000ABA6, U"\U000013D6\U0000ABA6\U000013D6" },
{ 0x0000ABA7, U"\U000013D7\U0000ABA7\U000013D7" },
{ 0x0000ABA8, U"\U000013D8\U0000ABA8\U000013D8" },
{ 0x0000ABA9, U"\U000013D9\U0000ABA9\U000013D9" },
{ 0x0000ABAA, U"\U000013DA\U0000ABAA\U000013DA" },
{ 0x0000ABAB, U"\U000013DB\U0000ABAB\U000013DB" },
{ 0x0000ABAC, U"\U000013DC\U0000ABAC\U000013DC" },
{ 0x0000ABAD, U"\U000013DD\U0000ABAD\U000013DD" },
{ 0x0000ABAE, U"\U000013DE\U0000ABAE\U000013DE" },
{ 0x0000ABAF, U"\U000013DF\U0000ABAF\U000013DF" },
{ 0x0000ABB0, U"\U000013E0\U0000ABB0\U000013E0" },
{ 0x0000ABB1, U"\U000013E1\U0000ABB1\U000013E1" },
{ 0x0000ABB2, U"\U000013E2\U0000ABB2\U000013E2" },
{ 0x0000ABB3, U"\U000013E3\U0000ABB3\U000013E3" },
{ 0x0000ABB4, U"\U000013E4\U0000ABB4\U000013E4" },
{ 0x0000ABB5, U"\U000013E5\U0000ABB5\U000013E5" },
{ 0x0000ABB6, U"\U000013E6\U0000ABB6\U000013E6" },
{ 0x0000ABB7, U"\U000013E7\U0000ABB7\U000013E7" },
{ 0x0000ABB8, U"\U000013E8\U0000ABB8\U000013E8" },
{ 0x0000ABB9, U"\U000013E9\U0000ABB9\U000013E9" },
{ 0x0000ABBA, U"\U000013EA\U0000ABBA\U000013EA" },
{ 0x0000ABBB, U"\U000013EB\U0000ABBB\U000013EB" },
{ 0x0000ABBC, U"\U000013EC\U0000ABBC\U000013EC" },
{ 0x0000ABBD, U"\U000013ED\U0000ABBD\U000013ED" },
{ 0x0000ABBE, U"\U000013EE\U0000ABBE\U000013EE" },
{ 0x0000ABBF, U"\U000013EF\U0000ABBF\U000013EF" },
{ 0x0000FF21, U"\U0000FF21\U0000FF41\U0000FF21" },
{ 0x0000FF22, U"\U0000FF22\U0000FF42\U0000FF22" },
{ 0x0000FF23, U"\U0000FF23\U0000FF43\U0000FF23" },
{ 0x0000FF24, U"\U0000FF24\U0000FF44\U0000FF24" },
{ 0x0000FF25, U"\U0000FF25\U0000FF45\U0000FF25" },
{ 0x0000FF26, U"\U0000FF26\U0000FF46\U0000FF26" },
{ 0x0000FF27, U"\U0000FF27\U0000FF47\U0000FF27" },
{ 0x0000FF28, U"\U0000FF28\U0000FF48\U0000FF28" },
{ 0x0000FF29, U"\U0000FF29\U0000FF49\U0000FF29" },
{ 0x0000FF2A, U"\U0000FF2A\U0000FF4A\U0000FF2A" },
{ 0x0000FF2B, U"\U0000FF2B\U0000FF4B\U0000FF2B" },
{ 0x0000FF2C, U"\U0000FF2C\U0000FF4C\U0000FF2C" },
{ 0x0000FF2D, U"\U0000FF2D\U0000FF4D\U0000FF2D" },
{ 0x0000FF2E, U"\U0000FF2E\U0000FF4E\U0000FF2E" },
{ 0x0000FF2F, U"\U0000FF2F\U0000FF4F\U0000FF2F" },
{ 0x0000FF30, U"\U0000FF30\U0000FF50\U0000FF30" },
{ 0x0000FF31, U"\U0000FF31\U0000FF51\U0000FF31" },
{ 0x0000FF32, U"\U0000FF32\U0000FF52\U0000FF32" },
{ 0x0000FF33, U"\U0000FF33\U0000FF53\U0000FF33" },
{ 0x0000FF34, U"\U0000FF34\U0000FF54\U0000FF34" },
{ 0x0000FF35, U"\U0000FF35\U0000FF55\U0000FF35" },
{ 0x0000FF36, U"\U0000FF36\U0000FF56\U0000FF36" },
{ 0x0000FF37, U"\U0000FF37\U0000FF57\U0000FF37" },
{ 0x0000FF38, U"\U0000FF38\U0000FF58\U0000FF38" },
{ 0x0000FF39, U"\U0000FF39\U0000FF59\U0000FF39" },
{ 0x0000FF3A, U"\U0000FF3A\U0000FF5A\U0000FF3A" },
{ 0x0000FF41, U"\U0000FF21\U0000FF41\U0000FF21" },
{ 0x0000FF42, U"\U0000FF22\U0000FF42\U0000FF22" },
{ 0x0000FF43, U"\U0000FF23\U0000FF43\U0000FF23" },
{ 0x0000FF44, U"\U0000FF24\U0000FF44\U0000FF24" },
{ 0x0000FF45, U"\U0000FF25\U0000FF45\U0000FF25" },
{ 0x0000FF46, U"\U0000FF26\U0000FF46\U0000FF26" },
{ 0x0000FF47, U"\U0000FF27\U0000FF47\U0000FF27" },
{ 0x0000FF48, U"\U0000FF28\U0000FF48\U0000FF28" },
{ 0x0000FF49, U"\U0000FF29\U0000FF49\U0000FF29" },
{ 0x0000FF4A, U"\U0000FF2A\U0000FF4A\U0000FF2A" },
{ 0x0000FF4B, U"\U0000FF2B\U0000FF4B\U0000FF2B" },
{ 0x0000FF4C, U"\U0000FF2C\U0000FF4C\U0000FF2C" },
{ 0x0000FF4D, U"\U0000FF2D\U0000FF4D\U0000FF2D" },
{ 0x0000FF4E, U"\U0000FF2E\U0000FF4E\U0000FF2E" },
{ 0x0000FF4F, U"\U0000FF2F\U0000FF4F\U0000FF2F" },
{ 0x0000FF50, U"\U0000FF30\U0000FF50\U0000FF30" },
{ 0x0000FF51, U"\U0000FF31\U0000FF51\U0000FF31" },
{ 0x0000FF52, U"\U0000FF32\U0000FF52\U0000FF32" },
{ 0x0000FF53, U"\U0000FF33\U0000FF53\U0000FF33" },
{ 0x0000FF54, U"\U0000FF34\U0000FF54\U0000FF34" },
{ 0x0000FF55, U"\U0000FF35\U0000FF55\U0000FF35" },
{ 0x0000FF56, U"\U0000FF36\U0000FF56\U0000FF36" },
{ 0x0000FF57, U"\U0000FF37\U0000FF57\U0000FF37" },
{ 0x0000FF58, U"\U0000FF38\U0000FF58\U0000FF38" },
{ 0x0000FF59, U"\U0000FF39\U0000FF59\U0000FF39" },
{ 0x0000FF5A, U"\U0000FF3A\U0000FF5A\U0000FF3A" },
{ 0x00010400, U"\U00010400\U00010428\U00010400" },
{ 0x00010401, U"\U00010401\U00010429\U00010401" },
{ 0x00010402, U"\U00010402\U0001042A\U00010402" },
{ 0x00010403, U"\U00010403\U0001042B\U00010403" },
{ 0x00010404, U"\U00010404\U0001042C\U00010404" },
{ 0x00010405, U"\U00010405\U0001042D\U00010405" },
{ 0x00010406, U"\U00010406\U0001042E\U00010406" },
{ 0x00010407, U"\U00010407\U0001042F\U00010407" },
{ 0x00010408, U"\U00010408\U00010430\U00010408" },
{ 0x00010409, U"\U00010409\U00010431\U00010409" },
{ 0x0001040A, U"\U0001040A\U00010432\U0001040A" },
{ 0x0001040B, U"\U0001040B\U00010433\U0001040B" },
{ 0x0001040C, U"\U0001040C\U00010434\U0001040C" },
{ 0x0001040D, U"\U0001040D\U00010435\U0001040D" },
{ 0x0001040E, U"\U0001040E\U00010436\U0001040E" },
{ 0x0001040F, U"\U0001040F\U00010437\U0001040F" },
{ 0x00010410, U"\U00010410\U00010438\U00010410" },
{ 0x00010411, U"\U00010411\U00010439\U00010411" },
{ 0x00010412, U"\U00010412\U0001043A\U00010412" },
{ 0x00010413, U"\U00010413\U0001043B\U00010413" },
{ 0x00010414, U"\U00010414\U0001043C\U00010414" },
{ 0x00010415, U"\U00010415\U0001043D\U00010415" },
{ 0x00010416, U"\U00010416\U0001043E\U00010416" },
{ 0x00010417, U"\U00010417\U0001043F\U00010417" },
{ 0x00010418, U"\U00010418\U00010440\U00010418" },
{ 0x00010419, U"\U00010419\U00010441\U00010419" },
{ 0x0001041A, U"\U0001041A\U00010442\U0001041A" },
{ 0x0001041B, U"\U0001041B\U00010443\U0001041B" },
{ 0x0001041C, U"\U0001041C\U00010444\U0001041C" },
{ 0x0001041D, U"\U0001041D\U00010445\U0001041D" },
{ 0x0001041E, U"\U0001041E\U00010446\U0001041E" },
{ 0x0001041F, U"\U0001041F\U00010447\U0001041F" },
{ 0x00010420, U"\U00010420\U00010448\U00010420" },
{ 0x00010421, U"\U00010421\U00010449\U00010421" },
{ 0x00010422, U"\U00010422\U0001044A\U00010422" },
{ 0x00010423, U"\U00010423\U0001044B\U00010423" },
{ 0x00010424, U"\U00010424\U0001044C\U00010424" },
{ 0x00010425, U"\U00010425\U0001044D\U00010425" },
{ 0x00010426, U"\U00010426\U0001044E\U00010426" },
{ 0x00010427, U"\U00010427\U0001044F\U00010427" },
{ 0x00010428, U"\U00010400\U00010428\U00010400" },
{ 0x00010429, U"\U00010401\U00010429\U00010401" },
{ 0x0001042A, U"\U00010402\U0001042A\U00010402" },
{ 0x0001042B, U"\U00010403\U0001042B\U00010403" },
{ 0x0001042C, U"\U00010404\U0001042C\U00010404" },
{ 0x0001042D, U"\U00010405\U0001042D\U00010405" },
{ 0x0001042E, U"\U00010406\U0001042E\U00010406" },
{ 0x0001042F, U"\U00010407\U0001042F\U00010407" },
{ 0x00010430, U"\U00010408\U00010430\U00010408" },
{ 0x00010431, U"\U00010409\U00010431\U00010409" },
{ 0x00010432, U"\U0001040A\U00010432\U0001040A" },
{ 0x00010433, U"\U0001040B\U00010433\U0001040B" },
{ 0x00010434, U"\U0001040C\U00010434\U0001040C" },
{ 0x00010435, U"\U0001040D\U00010435\U0001040D" },
{ 0x00010436, U"\U0001040E\U00010436\U0001040E" },
{ 0x00010437, U"\U0001040F\U00010437\U0001040F" },
{ 0x00010438, U"\U00010410\U00010438\U00010410" },
{ 0x00010439, U"\U00010411\U00010439\U00010411" },
{ 0x0001043A, U"\U00010412\U0001043A\U00010412" },
{ 0x0001043B, U"\U00010413\U0001043B\U00010413" },
{ 0x0001043C, U"\U00010414\U0001043C\U00010414" },
{ 0x0001043D, U"\U00010415\U0001043D\U00010415" },
{ 0x0001043E, U"\U00010416\U0001043E\U00010416" },
{ 0x0001043F, U"\U00010417\U0001043F\U00010417" },
{ 0x00010440, U"\U00010418\U00010440\U00010418" },
{ 0x00010441, U"\U00010419\U00010441\U00010419" },
{ 0x00010442, U"\U0001041A\U00010442\U0001041A" },
{ 0x00010443, U"\U0001041B\U00010443\U0001041B" },
{ 0x00010444, U"\U0001041C\U00010444\U0001041C" },
{ 0x00010445, U"\U0001041D\U00010445\U0001041D" },
{ 0x00010446, U"\U0001041E\U00010446\U0001041E" },
{ 0x00010447, U"\U0001041F\U00010447\U0001041F" },
{ 0x00010448, U"\U00010420\U00010448\U00010420" },
{ 0x00010449, U"\U00010421\U00010449\U00010421" },
{ 0x0001044A, U"\U00010422\U0001044A\U00010422" },
{ 0x0001044B, U"\U00010423\U0001044B\U00010423" },
{ 0x0001044C, U"\U00010424\U0001044C\U00010424" },
{ 0x0001044D, U"\U00010425\U0001044D\U00010425" },
{ 0x0001044E, U"\U00010426\U0001044E\U00010426" },
{ 0x0001044F, U"\U00010427\U0001044F\U00010427" },
{ 0x000104B0, U"\U000104B0\U000104D8\U000104B0" },
{ 0x000104B1, U"\U000104B1\U000104D9\U000104B1" },
{ 0x000104B2, U"\U000104B2\U000104DA\U000104B2" },
{ 0x000104B3, U"\U000104B3\U000104DB\U000104B3" },
{ 0x000104B4, U"\U000104B4\U000104DC\U000104B4" },
{ 0x000104B5, U"\U000104B5\U000104DD\U000104B5" },
{ 0x000104B6, U"\U000104B6\U000104DE\U000104B6" },
{ 0x000104B7, U"\U000104B7\U000104DF\U000104B7" },
{ 0x000104B8, U"\U000104B8\U000104E0\U000104B8" },
{ 0x000104B9, U"\U000104B9\U000104E1\U000104B9" },
{ 0x000104BA, U"\U000104BA\U000104E2\U000104BA" },
{ 0x000104BB, U"\U000104BB\U000104E3\U000104BB" },
{ 0x000104BC, U"\U000104BC\U000104E4\U000104BC" },
{ 0x000104BD, U"\U000104BD\U000104E5\U000104BD" },
{ 0x000104BE, U"\U000104BE\U000104E6\U000104BE" },
{ 0x000104BF, U"\U000104BF\U000104E7\U000104BF" },
{ 0x000104C0, U"\U000104C0\U000104E8\U000104C0" },
{ 0x000104C1, U"\U000104C1\U000104E9\U000104C1" },
{ 0x000104C2, U"\U000104C2\U000104EA\U000104C2" },
{ 0x000104C3, U"\U000104C3\U000104EB\U000104C3" },
{ 0x000104C4, U"\U000104C4\U000104EC\U000104C4" },
{ 0x000104C5, U"\U000104C5\U000104ED\U000104C5" },
{ 0x000104C6, U"\U000104C6\U000104EE\U000104C6" },
{ 0x000104C7, U"\U000104C7\U000104EF\U000104C7" },
{ 0x000104C8, U"\U000104C8\U000104F0\U000104C8" },
{ 0x000104C9, U"\U000104C9\U000104F1\U000104C9" },
{ 0x000104CA, U"\U000104CA\U000104F2\U000104CA" },
{ 0x000104CB, U"\U000104CB\U000104F3\U000104CB" },
{ 0x000104CC, U"\U000104CC\U000104F4\U000104CC" },
{ 0x000104CD, U"\U000104CD\U000104F5\U000104CD" },
{ 0x000104CE, U"\U000104CE\U000104F6\U000104CE" },
{ 0x000104CF, U"\U000104CF\U000104F7\U000104CF" },
{ 0x000104D0, U"\U000104D0\U000104F8\U000104D0" },
{ 0x000104D1, U"\U000104D1\U000104F9\U000104D1" },
{ 0x000104D2, U"\U000104D2\U000104FA\U000104D2" },
{ 0x000104D3, U"\U000104D3\U000104FB\U000104D3" },
{ 0x000104D8, U"\U000104B0\U000104D8\U000104B0" },
{ 0x000104D9, U"\U000104B1\U000104D9\U000104B1" },
{ 0x000104DA, U"\U000104B2\U000104DA\U000104B2" },
{ 0x000104DB, U"\U000104B3\U000104DB\U000104B3" },
{ 0x000104DC, U"\U000104B4\U000104DC\U000104B4" },
{ 0x000104DD, U"\U000104B5\U000104DD\U000104B5" },
{ 0x000104DE, U"\U000104B6\U000104DE\U000104B6" },
{ 0x000104DF, U"\U000104B7\U000104DF\U000104B7" },
{ 0x000104E0, U"\U000104B8\U000104E0\U000104B8" },
{ 0x000104E1, U"\U000104B9\U000104E1\U000104B9" },
{ 0x000104E2, U"\U000104BA\U000104E2\U000104BA" },
{ 0x000104E3, U"\U000104BB\U000104E3\U000104BB" },
{ 0x000104E4, U"\U000104BC\U000104E4\U000104BC" },
{ 0x000104E5, U"\U000104BD\U000104E5\U000104BD" },
{ 0x000104E6, U"\U000104BE\U000104E6\U000104BE" },
{ 0x000104E7, U"\U000104BF\U000104E7\U000104BF" },
{ 0x000104E8, U"\U000104C0\U000104E8\U000104C0" },
{ 0x000104E9, U"\U000104C1\U000104E9\U000104C1" },
{ 0x000104EA, U"\U000104C2\U000104EA\U000104C2" },
{ 0x000104EB, U"\U000104C3\U000104EB\U000104C3" },
{ 0x000104EC, U"\U000104C4\U000104EC\U000104C4" },
{ 0x000104ED, U"\U000104C5\U000104ED\U000104C5" },
{ 0x000104EE, U"\U000104C6\U000104EE\U000104C6" },
{ 0x000104EF, U"\U000104C7\U000104EF\U000104C7" },
{ 0x000104F0, U"\U000104C8\U000104F0\U000104C8" },
{ 0x000104F1, U"\U000104C9\U000104F1\U000104C9" },
{ 0x000104F2, U"\U000104CA\U000104F2\U000104CA" },
{ 0x000104F3, U"\U000104CB\U000104F3\U000104CB" },
{ 0x000104F4, U"\U000104CC\U000104F4\U000104CC" },
{ 0x000104F5, U"\U000104CD\U000104F5\U000104CD" },
{ 0x000104F6, U"\U000104CE\U000104F6\U000104CE" },
{ 0x000104F7, U"\U000104CF\U000104F7\U000104CF" },
{ 0x000104F8, U"\U000104D0\U000104F8\U000104D0" },
{ 0x000104F9, U"\U000104D1\U000104F9\U000104D1" },
{ 0x000104FA, U"\U000104D2\U000104FA\U000104D2" },
{ 0x000104FB, U"\U000104D3\U000104FB\U000104D3" },
{ 0x00010C80, U"\U00010C80\U00010CC0\U00010C80" },
{ 0x00010C81, U"\U00010C81\U00010CC1\U00010C81" },
{ 0x00010C82, U"\U00010C82\U00010CC2\U00010C82" },
{ 0x00010C83, U"\U00010C83\U00010CC3\U00010C83" },
{ 0x00010C84, U"\U00010C84\U00010CC4\U00010C84" },
{ 0x00010C85, U"\U00010C85\U00010CC5\U00010C85" },
{ 0x00010C86, U"\U00010C86\U00010CC6\U00010C86" },
{ 0x00010C87, U"\U00010C87\U00010CC7\U00010C87" },
{ 0x00010C88, U"\U00010C88\U00010CC8\U00010C88" },
{ 0x00010C89, U"\U00010C89\U00010CC9\U00010C89" },
{ 0x00010C8A, U"\U00010C8A\U00010CCA\U00010C8A" },
{ 0x00010C8B, U"\U00010C8B\U00010CCB\U00010C8B" },
{ 0x00010C8C, U"\U00010C8C\U00010CCC\U00010C8C" },
{ 0x00010C8D, U"\U00010C8D\U00010CCD\U00010C8D" },
{ 0x00010C8E, U"\U00010C8E\U00010CCE\U00010C8E" },
{ 0x00010C8F, U"\U00010C8F\U00010CCF\U00010C8F" },
{ 0x00010C90, U"\U00010C90\U00010CD0\U00010C90" },
{ 0x00010C91, U"\U00010C91\U00010CD1\U00010C91" },
{ 0x00010C92, U"\U00010C92\U00010CD2\U00010C92" },
{ 0x00010C93, U"\U00010C93\U00010CD3\U00010C93" },
{ 0x00010C94, U"\U00010C94\U00010CD4\U00010C94" },
{ 0x00010C95, U"\U00010C95\U00010CD5\U00010C95" },
{ 0x00010C96, U"\U00010C96\U00010CD6\U00010C96" },
{ 0x00010C97, U"\U00010C97\U00010CD7\U00010C97" },
{ 0x00010C98, U"\U00010C98\U00010CD8\U00010C98" },
{ 0x00010C99, U"\U00010C99\U00010CD9\U00010C99" },
{ 0x00010C9A, U"\U00010C9A\U00010CDA\U00010C9A" },
{ 0x00010C9B, U"\U00010C9B\U00010CDB\U00010C9B" },
{ 0x00010C9C, U"\U00010C9C\U00010CDC\U00010C9C" },
{ 0x00010C9D, U"\U00010C9D\U00010CDD\U00010C9D" },
{ 0x00010C9E, U"\U00010C9E\U00010CDE\U00010C9E" },
{ 0x00010C9F, U"\U00010C9F\U00010CDF\U00010C9F" },
{ 0x00010CA0, U"\U00010CA0\U00010CE0\U00010CA0" },
{ 0x00010CA1, U"\U00010CA1\U00010CE1\U00010CA1" },
{ 0x00010CA2, U"\U00010CA2\U00010CE2\U00010CA2" },
{ 0x00010CA3, U"\U00010CA3\U00010CE3\U00010CA3" },
{ 0x00010CA4, U"\U00010CA4\U00010CE4\U00010CA4" },
{ 0x00010CA5, U"\U00010CA5\U00010CE5\U00010CA5" },
{ 0x00010CA6, U"\U00010CA6\U00010CE6\U00010CA6" },
{ 0x00010CA7, U"\U00010CA7\U00010CE7\U00010CA7" },
{ 0x00010CA8, U"\U00010CA8\U00010CE8\U00010CA8" },
{ 0x00010CA9, U"\U00010CA9\U00010CE9\U00010CA9" },
{ 0x00010CAA, U"\U00010CAA\U00010CEA\U00010CAA" },
{ 0x00010CAB, U"\U00010CAB\U00010CEB\U00010CAB" },
{ 0x00010CAC, U"\U00010CAC\U00010CEC\U00010CAC" },
{ 0x00010CAD, U"\U00010CAD\U00010CED\U00010CAD" },
{ 0x00010CAE, U"\U00010CAE\U00010CEE\U00010CAE" },
{ 0x00010CAF, U"\U00010CAF\U00010CEF\U00010CAF" },
{ 0x00010CB0, U"\U00010CB0\U00010CF0\U00010CB0" },
{ 0x00010CB1, U"\U00010CB1\U00010CF1\U00010CB1" },
{ 0x00010CB2, U"\U00010CB2\U00010CF2\U00010CB2" },
{ 0x00010CC0, U"\U00010C80\U00010CC0\U00010C80" },
{ 0x00010CC1, U"\U00010C81\U00010CC1\U00010C81" },
{ 0x00010CC2, U"\U00010C82\U00010CC2\U00010C82" },
{ 0x00010CC3, U"\U00010C83\U00010CC3\U00010C83" },
{ 0x00010CC4, U"\U00010C84\U00010CC4\U00010C84" },
{ 0x00010CC5, U"\U00010C85\U00010CC5\U00010C85" },
{ 0x00010CC6, U"\U00010C86\U00010CC6\U00010C86" },
{ 0x00010CC7, U"\U00010C87\U00010CC7\U00010C87" },
{ 0x00010CC8, U"\U00010C88\U00010CC8\U00010C88" },
{ 0x00010CC9, U"\U00010C89\U00010CC9\U00010C89" },
{ 0x00010CCA, U"\U00010C8A\U00010CCA\U00010C8A" },
{ 0x00010CCB, U"\U00010C8B\U00010CCB\U00010C8B" },
{ 0x00010CCC, U"\U00010C8C\U00010CCC\U00010C8C" },
{ 0x00010CCD, U"\U00010C8D\U00010CCD\U00010C8D" },
{ 0x00010CCE, U"\U00010C8E\U00010CCE\U00010C8E" },
{ 0x00010CCF, U"\U00010C8F\U00010CCF\U00010C8F" },
{ 0x00010CD0, U"\U00010C90\U00010CD0\U00010C90" },
{ 0x00010CD1, U"\U00010C91\U00010CD1\U00010C91" },
{ 0x00010CD2, U"\U00010C92\U00010CD2\U00010C92" },
{ 0x00010CD3, U"\U00010C93\U00010CD3\U00010C93" },
{ 0x00010CD4, U"\U00010C94\U00010CD4\U00010C94" },
{ 0x00010CD5, U"\U00010C95\U00010CD5\U00010C95" },
{ 0x00010CD6, U"\U00010C96\U00010CD6\U00010C96" },
{ 0x00010CD7, U"\U00010C97\U00010CD7\U00010C97" },
{ 0x00010CD8, U"\U00010C98\U00010CD8\U00010C98" },
{ 0x00010CD9, U"\U00010C99\U00010CD9\U00010C99" },
{ 0x00010CDA, U"\U00010C9A\U00010CDA\U00010C9A" },
{ 0x00010CDB, U"\U00010C9B\U00010CDB\U00010C9B" },
{ 0x00010CDC, U"\U00010C9C\U00010CDC\U00010C9C" },
{ 0x00010CDD, U"\U00010C9D\U00010CDD\U00010C9D" },
{ 0x00010CDE, U"\U00010C9E\U00010CDE\U00010C9E" },
{ 0x00010CDF, U"\U00010C9F\U00010CDF\U00010C9F" },
{ 0x00010CE0, U"\U00010CA0\U00010CE0\U00010CA0" },
{ 0x00010CE1, U"\U00010CA1\U00010CE1\U00010CA1" },
{ 0x00010CE2, U"\U00010CA2\U00010CE2\U00010CA2" },
{ 0x00010CE3, U"\U00010CA3\U00010CE3\U00010CA3" },
{ 0x00010CE4, U"\U00010CA4\U00010CE4\U00010CA4" },
{ 0x00010CE5, U"\U00010CA5\U00010CE5\U00010CA5" },
{ 0x00010CE6, U"\U00010CA6\U00010CE6\U00010CA6" },
{ 0x00010CE7, U"\U00010CA7\U00010CE7\U00010CA7" },
{ 0x00010CE8, U"\U00010CA8\U00010CE8\U00010CA8" },
{ 0x00010CE9, U"\U00010CA9\U00010CE9\U00010CA9" },
{ 0x00010CEA, U"\U00010CAA\U00010CEA\U00010CAA" },
{ 0x00010CEB, U"\U00010CAB\U00010CEB\U00010CAB" },
{ 0x00010CEC, U"\U00010CAC\U00010CEC\U00010CAC" },
{ 0x00010CED, U"\U00010CAD\U00010CED\U00010CAD" },
{ 0x00010CEE, U"\U00010CAE\U00010CEE\U00010CAE" },
{ 0x00010CEF, U"\U00010CAF\U00010CEF\U00010CAF" },
{ 0x00010CF0, U"\U00010CB0\U00010CF0\U00010CB0" },
{ 0x00010CF1, U"\U00010CB1\U00010CF1\U00010CB1" },
{ 0x00010CF2, U"\U00010CB2\U00010CF2\U00010CB2" },
{ 0x000118A0, U"\U000118A0\U000118C0\U000118A0" },
{ 0x000118A1, U"\U000118A1\U000118C1\U000118A1" },
{ 0x000118A2, U"\U000118A2\U000118C2\U000118A2" },
{ 0x000118A3, U"\U000118A3\U000118C3\U000118A3" },
{ 0x000118A4, U"\U000118A4\U000118C4\U000118A4" },
{ 0x000118A5, U"\U000118A5\U000118C5\U000118A5" },
{ 0x000118A6, U"\U000118A6\U000118C6\U000118A6" },
{ 0x000118A7, U"\U000118A7\U000118C7\U000118A7" },
{ 0x000118A8, U"\U000118A8\U000118C8\U000118A8" },
{ 0x000118A9, U"\U000118A9\U000118C9\U000118A9" },
{ 0x000118AA, U"\U000118AA\U000118CA\U000118AA" },
{ 0x000118AB, U"\U000118AB\U000118CB\U000118AB" },
{ 0x000118AC, U"\U000118AC\U000118CC\U000118AC" },
{ 0x000118AD, U"\U000118AD\U000118CD\U000118AD" },
{ 0x000118AE, U"\U000118AE\U000118CE\U000118AE" },
{ 0x000118AF, U"\U000118AF\U000118CF\U000118AF" },
{ 0x000118B0, U"\U000118B0\U000118D0\U000118B0" },
{ 0x000118B1, U"\U000118B1\U000118D1\U000118B1" },
{ 0x000118B2, U"\U000118B2\U000118D2\U000118B2" },
{ 0x000118B3, U"\U000118B3\U000118D3\U000118B3" },
{ 0x000118B4, U"\U000118B4\U000118D4\U000118B4" },
{ 0x000118B5, U"\U000118B5\U000118D5\U000118B5" },
{ 0x000118B6, U"\U000118B6\U000118D6\U000118B6" },
{ 0x000118B7, U"\U000118B7\U000118D7\U000118B7" },
{ 0x000118B8, U"\U000118B8\U000118D8\U000118B8" },
{ 0x000118B9, U"\U000118B9\U000118D9\U000118B9" },
{ 0x000118BA, U"\U000118BA\U000118DA\U000118BA" },
{ 0x000118BB, U"\U000118BB\U000118DB\U000118BB" },
{ 0x000118BC, U"\U000118BC\U000118DC\U000118BC" },
{ 0x000118BD, U"\U000118BD\U000118DD\U000118BD" },
{ 0x000118BE, U"\U000118BE\U000118DE\U000118BE" },
{ 0x000118BF, U"\U000118BF\U000118DF\U000118BF" },
{ 0x000118C0, U"\U000118A0\U000118C0\U000118A0" },
{ 0x000118C1, U"\U000118A1\U000118C1\U000118A1" },
{ 0x000118C2, U"\U000118A2\U000118C2\U000118A2" },
{ 0x000118C3, U"\U000118A3\U000118C3\U000118A3" },
{ 0x000118C4, U"\U000118A4\U000118C4\U000118A4" },
{ 0x000118C5, U"\U000118A5\U000118C5\U000118A5" },
{ 0x000118C6, U"\U000118A6\U000118C6\U000118A6" },
{ 0x000118C7, U"\U000118A7\U000118C7\U000118A7" },
{ 0x000118C8, U"\U000118A8\U000118C8\U000118A8" },
{ 0x000118C9, U"\U000118A9\U000118C9\U000118A9" },
{ 0x000118CA, U"\U000118AA\U000118CA\U000118AA" },
{ 0x000118CB, U"\U000118AB\U000118CB\U000118AB" },
{ 0x000118CC, U"\U000118AC\U000118CC\U000118AC" },
{ 0x000118CD, U"\U000118AD\U000118CD\U000118AD" },
{ 0x000118CE, U"\U000118AE\U000118CE\U000118AE" },
{ 0x000118CF, U"\U000118AF\U000118CF\U000118AF" },
{ 0x000118D0, U"\U000118B0\U000118D0\U000118B0" },
{ 0x000118D1, U"\U000118B1\U000118D1\U000118B1" },
{ 0x000118D2, U"\U000118B2\U000118D2\U000118B2" },
{ 0x000118D3, U"\U000118B3\U000118D3\U000118B3" },
{ 0x000118D4, U"\U000118B4\U000118D4\U000118B4" },
{ 0x000118D5, U"\U000118B5\U000118D5\U000118B5" },
{ 0x000118D6, U"\U000118B6\U000118D6\U000118B6" },
{ 0x000118D7, U"\U000118B7\U000118D7\U000118B7" },
{ 0x000118D8, U"\U000118B8\U000118D8\U000118B8" },
{ 0x000118D9, U"\U000118B9\U000118D9\U000118B9" },
{ 0x000118DA, U"\U000118BA\U000118DA\U000118BA" },
{ 0x000118DB, U"\U000118BB\U000118DB\U000118BB" },
{ 0x000118DC, U"\U000118BC\U000118DC\U000118BC" },
{ 0x000118DD, U"\U000118BD\U000118DD\U000118BD" },
{ 0x000118DE, U"\U000118BE\U000118DE\U000118BE" },
{ 0x000118DF, U"\U000118BF\U000118DF\U000118BF" },
{ 0x00016E40, U"\U00016E40\U00016E60\U00016E40" },
{ 0x00016E41, U"\U00016E41\U00016E61\U00016E41" },
{ 0x00016E42, U"\U00016E42\U00016E62\U00016E42" },
{ 0x00016E43, U"\U00016E43\U00016E63\U00016E43" },
{ 0x00016E44, U"\U00016E44\U00016E64\U00016E44" },
{ 0x00016E45, U"\U00016E45\U00016E65\U00016E45" },
{ 0x00016E46, U"\U00016E46\U00016E66\U00016E46" },
{ 0x00016E47, U"\U00016E47\U00016E67\U00016E47" },
{ 0x00016E48, U"\U00016E48\U00016E68\U00016E48" },
{ 0x00016E49, U"\U00016E49\U00016E69\U00016E49" },
{ 0x00016E4A, U"\U00016E4A\U00016E6A\U00016E4A" },
{ 0x00016E4B, U"\U00016E4B\U00016E6B\U00016E4B" },
{ 0x00016E4C, U"\U00016E4C\U00016E6C\U00016E4C" },
{ 0x00016E4D, U"\U00016E4D\U00016E6D\U00016E4D" },
{ 0x00016E4E, U"\U00016E4E\U00016E6E\U00016E4E" },
{ 0x00016E4F, U"\U00016E4F\U00016E6F\U00016E4F" },
{ 0x00016E50, U"\U00016E50\U00016E70\U00016E50" },
{ 0x00016E51, U"\U00016E51\U00016E71\U00016E51" },
{ 0x00016E52, U"\U00016E52\U00016E72\U00016E52" },
{ 0x00016E53, U"\U00016E53\U00016E73\U00016E53" },
{ 0x00016E54, U"\U00016E54\U00016E74\U00016E54" },
{ 0x00016E55, U"\U00016E55\U00016E75\U00016E55" },
{ 0x00016E56, U"\U00016E56\U00016E76\U00016E56" },
{ 0x00016E57, U"\U00016E57\U00016E77\U00016E57" },
{ 0x00016E58, U"\U00016E58\U00016E78\U00016E58" },
{ 0x00016E59, U"\U00016E59\U00016E79\U00016E59" },
{ 0x00016E5A, U"\U00016E5A\U00016E7A\U00016E5A" },
{ 0x00016E5B, U"\U00016E5B\U00016E7B\U00016E5B" },
{ 0x00016E5C, U"\U00016E5C\U00016E7C\U00016E5C" },
{ 0x00016E5D, U"\U00016E5D\U00016E7D\U00016E5D" },
{ 0x00016E5E, U"\U00016E5E\U00016E7E\U00016E5E" },
{ 0x00016E5F, U"\U00016E5F\U00016E7F\U00016E5F" },
{ 0x00016E60, U"\U00016E40\U00016E60\U00016E40" },
{ 0x00016E61, U"\U00016E41\U00016E61\U00016E41" },
{ 0x00016E62, U"\U00016E42\U00016E62\U00016E42" },
{ 0x00016E63, U"\U00016E43\U00016E63\U00016E43" },
{ 0x00016E64, U"\U00016E44\U00016E64\U00016E44" },
{ 0x00016E65, U"\U00016E45\U00016E65\U00016E45" },
{ 0x00016E66, U"\U00016E46\U00016E66\U00016E46" },
{ 0x00016E67, U"\U00016E47\U00016E67\U00016E47" },
{ 0x00016E68, U"\U00016E48\U00016E68\U00016E48" },
{ 0x00016E69, U"\U00016E49\U00016E69\U00016E49" },
{ 0x00016E6A, U"\U00016E4A\U00016E6A\U00016E4A" },
{ 0x00016E6B, U"\U00016E4B\U00016E6B\U00016E4B" },
{ 0x00016E6C, U"\U00016E4C\U00016E6C\U00016E4C" },
{ 0x00016E6D, U"\U00016E4D\U00016E6D\U00016E4D" },
{ 0x00016E6E, U"\U00016E4E\U00016E6E\U00016E4E" },
{ 0x00016E6F, U"\U00016E4F\U00016E6F\U00016E4F" },
{ 0x00016E70, U"\U00016E50\U00016E70\U00016E50" },
{ 0x00016E71, U"\U00016E51\U00016E71\U00016E51" },
{ 0x00016E72, U"\U00016E52\U00016E72\U00016E52" },
{ 0x00016E73, U"\U00016E53\U00016E73\U00016E53" },
{ 0x00016E74, U"\U00016E54\U00016E74\U00016E54" },
{ 0x00016E75, U"\U00016E55\U00016E75\U00016E55" },
{ 0x00016E76, U"\U00016E56\U00016E76\U00016E56" },
{ 0x00016E77, U"\U00016E57\U00016E77\U00016E57" },
{ 0x00016E78, U"\U00016E58\U00016E78\U00016E58" },
{ 0x00016E79, U"\U00016E59\U00016E79\U00016E59" },
{ 0x00016E7A, U"\U00016E5A\U00016E7A\U00016E5A" },
{ 0x00016E7B, U"\U00016E5B\U00016E7B\U00016E5B" },
{ 0x00016E7C, U"\U00016E5C\U00016E7C\U00016E5C" },
{ 0x00016E7D, U"\U00016E5D\U00016E7D\U00016E5D" },
{ 0x00016E7E, U"\U00016E5E\U00016E7E\U00016E5E" },
{ 0x00016E7F, U"\U00016E5F\U00016E7F\U00016E5F" },
{ 0x0001E900, U"\U0001E900\U0001E922\U0001E900" },
{ 0x0001E901, U"\U0001E901\U0001E923\U0001E901" },
{ 0x0001E902, U"\U0001E902\U0001E924\U0001E902" },
{ 0x0001E903, U"\U0001E903\U0001E925\U0001E903" },
{ 0x0001E904, U"\U0001E904\U0001E926\U0001E904" },
{ 0x0001E905, U"\U0001E905\U0001E927\U0001E905" },
{ 0x0001E906, U"\U0001E906\U0001E928\U0001E906" },
{ 0x0001E907, U"\U0001E907\U0001E929\U0001E907" },
{ 0x0001E908, U"\U0001E908\U0001E92A\U0001E908" },
{ 0x0001E909, U"\U0001E909\U0001E92B\U0001E909" },
{ 0x0001E90A, U"\U0001E90A\U0001E92C\U0001E90A" },
{ 0x0001E90B, U"\U0001E90B\U0001E92D\U0001E90B" },
{ 0x0001E90C, U"\U0001E90C\U0001E92E\U0001E90C" },
{ 0x0001E90D, U"\U0001E90D\U0001E92F\U0001E90D" },
{ 0x0001E90E, U"\U0001E90E\U0001E930\U0001E90E" },
{ 0x0001E90F, U"\U0001E90F\U0001E931\U0001E90F" },
{ 0x0001E910, U"\U0001E910\U0001E932\U0001E910" },
{ 0x0001E911, U"\U0001E911\U0001E933\U0001E911" },
{ 0x0001E912, U"\U0001E912\U0001E934\U0001E912" },
{ 0x0001E913, U"\U0001E913\U0001E935\U0001E913" },
{ 0x0001E914, U"\U0001E914\U0001E936\U0001E914" },
{ 0x0001E915, U"\U0001E915\U0001E937\U0001E915" },
{ 0x0001E916, U"\U0001E916\U0001E938\U0001E916" },
{ 0x0001E917, U"\U0001E917\U0001E939\U0001E917" },
{ 0x0001E918, U"\U0001E918\U0001E93A\U0001E918" },
{ 0x0001E919, U"\U0001E919\U0001E93B\U0001E919" },
{ 0x0001E91A, U"\U0001E91A\U0001E93C\U0001E91A" },
{ 0x0001E91B, U"\U0001E91B\U0001E93D\U0001E91B" },
{ 0x0001E91C, U"\U0001E91C\U0001E93E\U0001E91C" },
{ 0x0001E91D, U"\U0001E91D\U0001E93F\U0001E91D" },
{ 0x0001E91E, U"\U0001E91E\U0001E940\U0001E91E" },
{ 0x0001E91F, U"\U0001E91F\U0001E941\U0001E91F" },
{ 0x0001E920, U"\U0001E920\U0001E942\U0001E920" },
{ 0x0001E921, U"\U0001E921\U0001E943\U0001E921" },
{ 0x0001E922, U"\U0001E900\U0001E922\U0001E900" },
{ 0x0001E923, U"\U0001E901\U0001E923\U0001E901" },
{ 0x0001E924, U"\U0001E902\U0001E924\U0001E902" },
{ 0x0001E925, U"\U0001E903\U0001E925\U0001E903" },
{ 0x0001E926, U"\U0001E904\U0001E926\U0001E904" },
{ 0x0001E927, U"\U0001E905\U0001E927\U0001E905" },
{ 0x0001E928, U"\U0001E906\U0001E928\U0001E906" },
{ 0x0001E929, U"\U0001E907\U0001E929\U0001E907" },
{ 0x0001E92A, U"\U0001E908\U0001E92A\U0001E908" },
{ 0x0001E92B, U"\U0001E909\U0001E92B\U0001E909" },
{ 0x0001E92C, U"\U0001E90A\U0001E92C\U0001E90A" },
{ 0x0001E92D, U"\U0001E90B\U0001E92D\U0001E90B" },
{ 0x0001E92E, U"\U0001E90C\U0001E92E\U0001E90C" },
{ 0x0001E92F, U"\U0001E90D\U0001E92F\U0001E90D" },
{ 0x0001E930, U"\U0001E90E\U0001E930\U0001E90E" },
{ 0x0001E931, U"\U0001E90F\U0001E931\U0001E90F" },
{ 0x0001E932, U"\U0001E910\U0001E932\U0001E910" },
{ 0x0001E933, U"\U0001E911\U0001E933\U0001E911" },
{ 0x0001E934, U"\U0001E912\U0001E934\U0001E912" },
{ 0x0001E935, U"\U0001E913\U0001E935\U0001E913" },
{ 0x0001E936, U"\U0001E914\U0001E936\U0001E914" },
{ 0x0001E937, U"\U0001E915\U0001E937\U0001E915" },
{ 0x0001E938, U"\U0001E916\U0001E938\U0001E916" },
{ 0x0001E939, U"\U0001E917\U0001E939\U0001E917" },
{ 0x0001E93A, U"\U0001E918\U0001E93A\U0001E918" },
{ 0x0001E93B, U"\U0001E919\U0001E93B\U0001E919" },
{ 0x0001E93C, U"\U0001E91A\U0001E93C\U0001E91A" },
{ 0x0001E93D, U"\U0001E91B\U0001E93D\U0001E91B" },
{ 0x0001E93E, U"\U0001E91C\U0001E93E\U0001E91C" },
{ 0x0001E93F, U"\U0001E91D\U0001E93F\U0001E91D" },
{ 0x0001E940, U"\U0001E91E\U0001E940\U0001E91E" },
{ 0x0001E941, U"\U0001E91F\U0001E941\U0001E91F" },
{ 0x0001E942, U"\U0001E920\U0001E942\U0001E920" },
{ 0x0001E943, U"\U0001E921\U0001E943\U0001E921" },
};
const std::unordered_multimap<char32_t, SpecialCasing> _special_case_mappings = {
{ 0x000003A3, { U"\U000003C2", U"\U000003A3", U"\U000003A3", 0, SpecialCasingContext::Final_Sigma } },
{ 0x00000307, { U"\U00000307", 0, 0, "lt", SpecialCasingContext::After_Soft_Dotted } },
{ 0x00000049, { U"\U00000069\U00000307", U"\U00000049", U"\U00000049", "lt", SpecialCasingContext::More_Above } },
{ 0x0000004A, { U"\U0000006A\U00000307", U"\U0000004A", U"\U0000004A", "lt", SpecialCasingContext::More_Above } },
{ 0x0000012E, { U"\U0000012F\U00000307", U"\U0000012E", U"\U0000012E", "lt", SpecialCasingContext::More_Above } },
{ 0x000000CC, { U"\U00000069\U00000307\U00000300", U"\U000000CC", U"\U000000CC", "lt", SpecialCasingContext::Unassigned } },
{ 0x000000CD, { U"\U00000069\U00000307\U00000301", U"\U000000CD", U"\U000000CD", "lt", SpecialCasingContext::Unassigned } },
{ 0x00000128, { U"\U00000069\U00000307\U00000303", U"\U00000128", U"\U00000128", "lt", SpecialCasingContext::Unassigned } },
{ 0x00000130, { U"\U00000069", U"\U00000130", U"\U00000130", "tr", SpecialCasingContext::Unassigned } },
{ 0x00000130, { U"\U00000069", U"\U00000130", U"\U00000130", "az", SpecialCasingContext::Unassigned } },
{ 0x00000307, { 0, U"\U00000307", U"\U00000307", "tr", SpecialCasingContext::After_I } },
{ 0x00000307, { 0, U"\U00000307", U"\U00000307", "az", SpecialCasingContext::After_I } },
{ 0x00000049, { U"\U00000131", U"\U00000049", U"\U00000049", "tr", SpecialCasingContext::Not_Before_Dot } },
{ 0x00000049, { U"\U00000131", U"\U00000049", U"\U00000049", "az", SpecialCasingContext::Not_Before_Dot } },
{ 0x00000069, { U"\U00000069", U"\U00000130", U"\U00000130", "tr", SpecialCasingContext::Unassigned } },
{ 0x00000069, { U"\U00000069", U"\U00000130", U"\U00000130", "az", SpecialCasingContext::Unassigned } },
};
const std::unordered_multimap<char32_t, SpecialCasing> _special_case_mappings_default = {
{ 0x000000DF, { U"\U000000DF", U"\U00000053\U00000073", U"\U00000053\U00000053", 0, SpecialCasingContext::Unassigned } },
{ 0x00000130, { U"\U00000069\U00000307", U"\U00000130", U"\U00000130", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB00, { U"\U0000FB00", U"\U00000046\U00000066", U"\U00000046\U00000046", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB01, { U"\U0000FB01", U"\U00000046\U00000069", U"\U00000046\U00000049", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB02, { U"\U0000FB02", U"\U00000046\U0000006C", U"\U00000046\U0000004C", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB03, { U"\U0000FB03", U"\U00000046\U00000066\U00000069", U"\U00000046\U00000046\U00000049", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB04, { U"\U0000FB04", U"\U00000046\U00000066\U0000006C", U"\U00000046\U00000046\U0000004C", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB05, { U"\U0000FB05", U"\U00000053\U00000074", U"\U00000053\U00000054", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB06, { U"\U0000FB06", U"\U00000053\U00000074", U"\U00000053\U00000054", 0, SpecialCasingContext::Unassigned } },
{ 0x00000587, { U"\U00000587", U"\U00000535\U00000582", U"\U00000535\U00000552", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB13, { U"\U0000FB13", U"\U00000544\U00000576", U"\U00000544\U00000546", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB14, { U"\U0000FB14", U"\U00000544\U00000565", U"\U00000544\U00000535", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB15, { U"\U0000FB15", U"\U00000544\U0000056B", U"\U00000544\U0000053B", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB16, { U"\U0000FB16", U"\U0000054E\U00000576", U"\U0000054E\U00000546", 0, SpecialCasingContext::Unassigned } },
{ 0x0000FB17, { U"\U0000FB17", U"\U00000544\U0000056D", U"\U00000544\U0000053D", 0, SpecialCasingContext::Unassigned } },
{ 0x00000149, { U"\U00000149", U"\U000002BC\U0000004E", U"\U000002BC\U0000004E", 0, SpecialCasingContext::Unassigned } },
{ 0x00000390, { U"\U00000390", U"\U00000399\U00000308\U00000301", U"\U00000399\U00000308\U00000301", 0, SpecialCasingContext::Unassigned } },
{ 0x000003B0, { U"\U000003B0", U"\U000003A5\U00000308\U00000301", U"\U000003A5\U00000308\U00000301", 0, SpecialCasingContext::Unassigned } },
{ 0x000001F0, { U"\U000001F0", U"\U0000004A\U0000030C", U"\U0000004A\U0000030C", 0, SpecialCasingContext::Unassigned } },
{ 0x00001E96, { U"\U00001E96", U"\U00000048\U00000331", U"\U00000048\U00000331", 0, SpecialCasingContext::Unassigned } },
{ 0x00001E97, { U"\U00001E97", U"\U00000054\U00000308", U"\U00000054\U00000308", 0, SpecialCasingContext::Unassigned } },
{ 0x00001E98, { U"\U00001E98", U"\U00000057\U0000030A", U"\U00000057\U0000030A", 0, SpecialCasingContext::Unassigned } },
{ 0x00001E99, { U"\U00001E99", U"\U00000059\U0000030A", U"\U00000059\U0000030A", 0, SpecialCasingContext::Unassigned } },
{ 0x00001E9A, { U"\U00001E9A", U"\U00000041\U000002BE", U"\U00000041\U000002BE", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F50, { U"\U00001F50", U"\U000003A5\U00000313", U"\U000003A5\U00000313", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F52, { U"\U00001F52", U"\U000003A5\U00000313\U00000300", U"\U000003A5\U00000313\U00000300", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F54, { U"\U00001F54", U"\U000003A5\U00000313\U00000301", U"\U000003A5\U00000313\U00000301", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F56, { U"\U00001F56", U"\U000003A5\U00000313\U00000342", U"\U000003A5\U00000313\U00000342", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FB6, { U"\U00001FB6", U"\U00000391\U00000342", U"\U00000391\U00000342", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FC6, { U"\U00001FC6", U"\U00000397\U00000342", U"\U00000397\U00000342", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FD2, { U"\U00001FD2", U"\U00000399\U00000308\U00000300", U"\U00000399\U00000308\U00000300", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FD3, { U"\U00001FD3", U"\U00000399\U00000308\U00000301", U"\U00000399\U00000308\U00000301", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FD6, { U"\U00001FD6", U"\U00000399\U00000342", U"\U00000399\U00000342", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FD7, { U"\U00001FD7", U"\U00000399\U00000308\U00000342", U"\U00000399\U00000308\U00000342", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FE2, { U"\U00001FE2", U"\U000003A5\U00000308\U00000300", U"\U000003A5\U00000308\U00000300", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FE3, { U"\U00001FE3", U"\U000003A5\U00000308\U00000301", U"\U000003A5\U00000308\U00000301", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FE4, { U"\U00001FE4", U"\U000003A1\U00000313", U"\U000003A1\U00000313", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FE6, { U"\U00001FE6", U"\U000003A5\U00000342", U"\U000003A5\U00000342", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FE7, { U"\U00001FE7", U"\U000003A5\U00000308\U00000342", U"\U000003A5\U00000308\U00000342", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FF6, { U"\U00001FF6", U"\U000003A9\U00000342", U"\U000003A9\U00000342", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F80, { U"\U00001F80", U"\U00001F88", U"\U00001F08\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F81, { U"\U00001F81", U"\U00001F89", U"\U00001F09\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F82, { U"\U00001F82", U"\U00001F8A", U"\U00001F0A\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F83, { U"\U00001F83", U"\U00001F8B", U"\U00001F0B\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F84, { U"\U00001F84", U"\U00001F8C", U"\U00001F0C\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F85, { U"\U00001F85", U"\U00001F8D", U"\U00001F0D\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F86, { U"\U00001F86", U"\U00001F8E", U"\U00001F0E\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F87, { U"\U00001F87", U"\U00001F8F", U"\U00001F0F\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F88, { U"\U00001F80", U"\U00001F88", U"\U00001F08\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F89, { U"\U00001F81", U"\U00001F89", U"\U00001F09\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F8A, { U"\U00001F82", U"\U00001F8A", U"\U00001F0A\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F8B, { U"\U00001F83", U"\U00001F8B", U"\U00001F0B\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F8C, { U"\U00001F84", U"\U00001F8C", U"\U00001F0C\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F8D, { U"\U00001F85", U"\U00001F8D", U"\U00001F0D\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F8E, { U"\U00001F86", U"\U00001F8E", U"\U00001F0E\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F8F, { U"\U00001F87", U"\U00001F8F", U"\U00001F0F\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F90, { U"\U00001F90", U"\U00001F98", U"\U00001F28\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F91, { U"\U00001F91", U"\U00001F99", U"\U00001F29\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F92, { U"\U00001F92", U"\U00001F9A", U"\U00001F2A\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F93, { U"\U00001F93", U"\U00001F9B", U"\U00001F2B\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F94, { U"\U00001F94", U"\U00001F9C", U"\U00001F2C\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F95, { U"\U00001F95", U"\U00001F9D", U"\U00001F2D\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F96, { U"\U00001F96", U"\U00001F9E", U"\U00001F2E\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F97, { U"\U00001F97", U"\U00001F9F", U"\U00001F2F\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F98, { U"\U00001F90", U"\U00001F98", U"\U00001F28\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F99, { U"\U00001F91", U"\U00001F99", U"\U00001F29\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F9A, { U"\U00001F92", U"\U00001F9A", U"\U00001F2A\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F9B, { U"\U00001F93", U"\U00001F9B", U"\U00001F2B\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F9C, { U"\U00001F94", U"\U00001F9C", U"\U00001F2C\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F9D, { U"\U00001F95", U"\U00001F9D", U"\U00001F2D\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F9E, { U"\U00001F96", U"\U00001F9E", U"\U00001F2E\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001F9F, { U"\U00001F97", U"\U00001F9F", U"\U00001F2F\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA0, { U"\U00001FA0", U"\U00001FA8", U"\U00001F68\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA1, { U"\U00001FA1", U"\U00001FA9", U"\U00001F69\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA2, { U"\U00001FA2", U"\U00001FAA", U"\U00001F6A\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA3, { U"\U00001FA3", U"\U00001FAB", U"\U00001F6B\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA4, { U"\U00001FA4", U"\U00001FAC", U"\U00001F6C\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA5, { U"\U00001FA5", U"\U00001FAD", U"\U00001F6D\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA6, { U"\U00001FA6", U"\U00001FAE", U"\U00001F6E\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA7, { U"\U00001FA7", U"\U00001FAF", U"\U00001F6F\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA8, { U"\U00001FA0", U"\U00001FA8", U"\U00001F68\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FA9, { U"\U00001FA1", U"\U00001FA9", U"\U00001F69\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FAA, { U"\U00001FA2", U"\U00001FAA", U"\U00001F6A\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FAB, { U"\U00001FA3", U"\U00001FAB", U"\U00001F6B\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FAC, { U"\U00001FA4", U"\U00001FAC", U"\U00001F6C\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FAD, { U"\U00001FA5", U"\U00001FAD", U"\U00001F6D\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FAE, { U"\U00001FA6", U"\U00001FAE", U"\U00001F6E\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FAF, { U"\U00001FA7", U"\U00001FAF", U"\U00001F6F\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FB3, { U"\U00001FB3", U"\U00001FBC", U"\U00000391\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FBC, { U"\U00001FB3", U"\U00001FBC", U"\U00000391\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FC3, { U"\U00001FC3", U"\U00001FCC", U"\U00000397\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FCC, { U"\U00001FC3", U"\U00001FCC", U"\U00000397\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FF3, { U"\U00001FF3", U"\U00001FFC", U"\U000003A9\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FFC, { U"\U00001FF3", U"\U00001FFC", U"\U000003A9\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FB2, { U"\U00001FB2", U"\U00001FBA\U00000345", U"\U00001FBA\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FB4, { U"\U00001FB4", U"\U00000386\U00000345", U"\U00000386\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FC2, { U"\U00001FC2", U"\U00001FCA\U00000345", U"\U00001FCA\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FC4, { U"\U00001FC4", U"\U00000389\U00000345", U"\U00000389\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FF2, { U"\U00001FF2", U"\U00001FFA\U00000345", U"\U00001FFA\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FF4, { U"\U00001FF4", U"\U0000038F\U00000345", U"\U0000038F\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FB7, { U"\U00001FB7", U"\U00000391\U00000342\U00000345", U"\U00000391\U00000342\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FC7, { U"\U00001FC7", U"\U00000397\U00000342\U00000345", U"\U00000397\U00000342\U00000399", 0, SpecialCasingContext::Unassigned } },
{ 0x00001FF7, { U"\U00001FF7", U"\U000003A9\U00000342\U00000345", U"\U000003A9\U00000342\U00000399", 0, SpecialCasingContext::Unassigned } },
};
const std::unordered_map<char32_t, CaseFolding> _case_foldings = {
{ 0x00000041, { 0x00000061,  0x00000000, 0, 0x00000000 } },
{ 0x00000042, { 0x00000062,  0x00000000, 0, 0x00000000 } },
{ 0x00000043, { 0x00000063,  0x00000000, 0, 0x00000000 } },
{ 0x00000044, { 0x00000064,  0x00000000, 0, 0x00000000 } },
{ 0x00000045, { 0x00000065,  0x00000000, 0, 0x00000000 } },
{ 0x00000046, { 0x00000066,  0x00000000, 0, 0x00000000 } },
{ 0x00000047, { 0x00000067,  0x00000000, 0, 0x00000000 } },
{ 0x00000048, { 0x00000068,  0x00000000, 0, 0x00000000 } },
{ 0x00000049, { 0x00000069,  0x00000000, 0, 0x00000131 } },
{ 0x0000004A, { 0x0000006A,  0x00000000, 0, 0x00000000 } },
{ 0x0000004B, { 0x0000006B,  0x00000000, 0, 0x00000000 } },
{ 0x0000004C, { 0x0000006C,  0x00000000, 0, 0x00000000 } },
{ 0x0000004D, { 0x0000006D,  0x00000000, 0, 0x00000000 } },
{ 0x0000004E, { 0x0000006E,  0x00000000, 0, 0x00000000 } },
{ 0x0000004F, { 0x0000006F,  0x00000000, 0, 0x00000000 } },
{ 0x00000050, { 0x00000070,  0x00000000, 0, 0x00000000 } },
{ 0x00000051, { 0x00000071,  0x00000000, 0, 0x00000000 } },
{ 0x00000052, { 0x00000072,  0x00000000, 0, 0x00000000 } },
{ 0x00000053, { 0x00000073,  0x00000000, 0, 0x00000000 } },
{ 0x00000054, { 0x00000074,  0x00000000, 0, 0x00000000 } },
{ 0x00000055, { 0x00000075,  0x00000000, 0, 0x00000000 } },
{ 0x00000056, { 0x00000076,  0x00000000, 0, 0x00000000 } },
{ 0x00000057, { 0x00000077,  0x00000000, 0, 0x00000000 } },
{ 0x00000058, { 0x00000078,  0x00000000, 0, 0x00000000 } },
{ 0x00000059, { 0x00000079,  0x00000000, 0, 0x00000000 } },
{ 0x0000005A, { 0x0000007A,  0x00000000, 0, 0x00000000 } },
{ 0x000000B5, { 0x000003BC,  0x00000000, 0, 0x00000000 } },
{ 0x000000C0, { 0x000000E0,  0x00000000, 0, 0x00000000 } },
{ 0x000000C1, { 0x000000E1,  0x00000000, 0, 0x00000000 } },
{ 0x000000C2, { 0x000000E2,  0x00000000, 0, 0x00000000 } },
{ 0x000000C3, { 0x000000E3,  0x00000000, 0, 0x00000000 } },
{ 0x000000C4, { 0x000000E4,  0x00000000, 0, 0x00000000 } },
{ 0x000000C5, { 0x000000E5,  0x00000000, 0, 0x00000000 } },
{ 0x000000C6, { 0x000000E6,  0x00000000, 0, 0x00000000 } },
{ 0x000000C7, { 0x000000E7,  0x00000000, 0, 0x00000000 } },
{ 0x000000C8, { 0x000000E8,  0x00000000, 0, 0x00000000 } },
{ 0x000000C9, { 0x000000E9,  0x00000000, 0, 0x00000000 } },
{ 0x000000CA, { 0x000000EA,  0x00000000, 0, 0x00000000 } },
{ 0x000000CB, { 0x000000EB,  0x00000000, 0, 0x00000000 } },
{ 0x000000CC, { 0x000000EC,  0x00000000, 0, 0x00000000 } },
{ 0x000000CD, { 0x000000ED,  0x00000000, 0, 0x00000000 } },
{ 0x000000CE, { 0x000000EE,  0x00000000, 0, 0x00000000 } },
{ 0x000000CF, { 0x000000EF,  0x00000000, 0, 0x00000000 } },
{ 0x000000D0, { 0x000000F0,  0x00000000, 0, 0x00000000 } },
{ 0x000000D1, { 0x000000F1,  0x00000000, 0, 0x00000000 } },
{ 0x000000D2, { 0x000000F2,  0x00000000, 0, 0x00000000 } },
{ 0x000000D3, { 0x000000F3,  0x00000000, 0, 0x00000000 } },
{ 0x000000D4, { 0x000000F4,  0x00000000, 0, 0x00000000 } },
{ 0x000000D5, { 0x000000F5,  0x00000000, 0, 0x00000000 } },
{ 0x000000D6, { 0x000000F6,  0x00000000, 0, 0x00000000 } },
{ 0x000000D8, { 0x000000F8,  0x00000000, 0, 0x00000000 } },
{ 0x000000D9, { 0x000000F9,  0x00000000, 0, 0x00000000 } },
{ 0x000000DA, { 0x000000FA,  0x00000000, 0, 0x00000000 } },
{ 0x000000DB, { 0x000000FB,  0x00000000, 0, 0x00000000 } },
{ 0x000000DC, { 0x000000FC,  0x00000000, 0, 0x00000000 } },
{ 0x000000DD, { 0x000000FD,  0x00000000, 0, 0x00000000 } },
{ 0x000000DE, { 0x000000FE,  0x00000000, 0, 0x00000000 } },
{ 0x000000DF, { 0x00000000,  0x00000000, U"\U00000073\U00000073", 0x00000000 } },
{ 0x00000100, { 0x00000101,  0x00000000, 0, 0x00000000 } },
{ 0x00000102, { 0x00000103,  0x00000000, 0, 0x00000000 } },
{ 0x00000104, { 0x00000105,  0x00000000, 0, 0x00000000 } },
{ 0x00000106, { 0x00000107,  0x00000000, 0, 0x00000000 } },
{ 0x00000108, { 0x00000109,  0x00000000, 0, 0x00000000 } },
{ 0x0000010A, { 0x0000010B,  0x00000000, 0, 0x00000000 } },
{ 0x0000010C, { 0x0000010D,  0x00000000, 0, 0x00000000 } },
{ 0x0000010E, { 0x0000010F,  0x00000000, 0, 0x00000000 } },
{ 0x00000110, { 0x00000111,  0x00000000, 0, 0x00000000 } },
{ 0x00000112, { 0x00000113,  0x00000000, 0, 0x00000000 } },
{ 0x00000114, { 0x00000115,  0x00000000, 0, 0x00000000 } },
{ 0x00000116, { 0x00000117,  0x00000000, 0, 0x00000000 } },
{ 0x00000118, { 0x00000119,  0x00000000, 0, 0x00000000 } },
{ 0x0000011A, { 0x0000011B,  0x00000000, 0, 0x00000000 } },
{ 0x0000011C, { 0x0000011D,  0x00000000, 0, 0x00000000 } },
{ 0x0000011E, { 0x0000011F,  0x00000000, 0, 0x00000000 } },
{ 0x00000120, { 0x00000121,  0x00000000, 0, 0x00000000 } },
{ 0x00000122, { 0x00000123,  0x00000000, 0, 0x00000000 } },
{ 0x00000124, { 0x00000125,  0x00000000, 0, 0x00000000 } },
{ 0x00000126, { 0x00000127,  0x00000000, 0, 0x00000000 } },
{ 0x00000128, { 0x00000129,  0x00000000, 0, 0x00000000 } },
{ 0x0000012A, { 0x0000012B,  0x00000000, 0, 0x00000000 } },
{ 0x0000212B, { 0x000000E5,  0x00000000, 0, 0x00000000 } },
{ 0x0000012C, { 0x0000012D,  0x00000000, 0, 0x00000000 } },
{ 0x0000012E, { 0x0000012F,  0x00000000, 0, 0x00000000 } },
{ 0x00000130, { 0x00000000,  0x00000000, U"\U00000069\U00000307", 0x00000069 } },
{ 0x00000132, { 0x00000133,  0x00000000, 0, 0x00000000 } },
{ 0x00000134, { 0x00000135,  0x00000000, 0, 0x00000000 } },
{ 0x00000136, { 0x00000137,  0x00000000, 0, 0x00000000 } },
{ 0x00000139, { 0x0000013A,  0x00000000, 0, 0x00000000 } },
{ 0x0000013B, { 0x0000013C,  0x00000000, 0, 0x00000000 } },
{ 0x0000013D, { 0x0000013E,  0x00000000, 0, 0x00000000 } },
{ 0x0000013F, { 0x00000140,  0x00000000, 0, 0x00000000 } },
{ 0x00000141, { 0x00000142,  0x00000000, 0, 0x00000000 } },
{ 0x00000143, { 0x00000144,  0x00000000, 0, 0x00000000 } },
{ 0x00000145, { 0x00000146,  0x00000000, 0, 0x00000000 } },
{ 0x00000147, { 0x00000148,  0x00000000, 0, 0x00000000 } },
{ 0x00000149, { 0x00000000,  0x00000000, U"\U000002BC\U0000006E", 0x00000000 } },
{ 0x0000014A, { 0x0000014B,  0x00000000, 0, 0x00000000 } },
{ 0x0000014C, { 0x0000014D,  0x00000000, 0, 0x00000000 } },
{ 0x0000014E, { 0x0000014F,  0x00000000, 0, 0x00000000 } },
{ 0x00000150, { 0x00000151,  0x00000000, 0, 0x00000000 } },
{ 0x00000152, { 0x00000153,  0x00000000, 0, 0x00000000 } },
{ 0x00000154, { 0x00000155,  0x00000000, 0, 0x00000000 } },
{ 0x00000156, { 0x00000157,  0x00000000, 0, 0x00000000 } },
{ 0x00000158, { 0x00000159,  0x00000000, 0, 0x00000000 } },
{ 0x0000015A, { 0x0000015B,  0x00000000, 0, 0x00000000 } },
{ 0x0000015C, { 0x0000015D,  0x00000000, 0, 0x00000000 } },
{ 0x0000015E, { 0x0000015F,  0x00000000, 0, 0x00000000 } },
{ 0x00000160, { 0x00000161,  0x00000000, 0, 0x00000000 } },
{ 0x00002161, { 0x00002171,  0x00000000, 0, 0x00000000 } },
{ 0x00000162, { 0x00000163,  0x00000000, 0, 0x00000000 } },
{ 0x00002163, { 0x00002173,  0x00000000, 0, 0x00000000 } },
{ 0x00000164, { 0x00000165,  0x00000000, 0, 0x00000000 } },
{ 0x00002165, { 0x00002175,  0x00000000, 0, 0x00000000 } },
{ 0x00002166, { 0x00002176,  0x00000000, 0, 0x00000000 } },
{ 0x00002167, { 0x00002177,  0x00000000, 0, 0x00000000 } },
{ 0x00002168, { 0x00002178,  0x00000000, 0, 0x00000000 } },
{ 0x00002169, { 0x00002179,  0x00000000, 0, 0x00000000 } },
{ 0x0000216A, { 0x0000217A,  0x00000000, 0, 0x00000000 } },
{ 0x0000216B, { 0x0000217B,  0x00000000, 0, 0x00000000 } },
{ 0x0000216C, { 0x0000217C,  0x00000000, 0, 0x00000000 } },
{ 0x0000216D, { 0x0000217D,  0x00000000, 0, 0x00000000 } },
{ 0x0000216E, { 0x0000217E,  0x00000000, 0, 0x00000000 } },
{ 0x0000216F, { 0x0000217F,  0x00000000, 0, 0x00000000 } },
{ 0x00000170, { 0x00000171,  0x00000000, 0, 0x00000000 } },
{ 0x00000172, { 0x00000173,  0x00000000, 0, 0x00000000 } },
{ 0x00000174, { 0x00000175,  0x00000000, 0, 0x00000000 } },
{ 0x00000176, { 0x00000177,  0x00000000, 0, 0x00000000 } },
{ 0x00000178, { 0x000000FF,  0x00000000, 0, 0x00000000 } },
{ 0x00000179, { 0x0000017A,  0x00000000, 0, 0x00000000 } },
{ 0x0000017B, { 0x0000017C,  0x00000000, 0, 0x00000000 } },
{ 0x0000017D, { 0x0000017E,  0x00000000, 0, 0x00000000 } },
{ 0x0000017F, { 0x00000073,  0x00000000, 0, 0x00000000 } },
{ 0x00000181, { 0x00000253,  0x00000000, 0, 0x00000000 } },
{ 0x00000182, { 0x00000183,  0x00000000, 0, 0x00000000 } },
{ 0x00002183, { 0x00002184,  0x00000000, 0, 0x00000000 } },
{ 0x00000184, { 0x00000185,  0x00000000, 0, 0x00000000 } },
{ 0x00000186, { 0x00000254,  0x00000000, 0, 0x00000000 } },
{ 0x00000187, { 0x00000188,  0x00000000, 0, 0x00000000 } },
{ 0x00000189, { 0x00000256,  0x00000000, 0, 0x00000000 } },
{ 0x0000018A, { 0x00000257,  0x00000000, 0, 0x00000000 } },
{ 0x0000018B, { 0x0000018C,  0x00000000, 0, 0x00000000 } },
{ 0x0000018E, { 0x000001DD,  0x00000000, 0, 0x00000000 } },
{ 0x0000018F, { 0x00000259,  0x00000000, 0, 0x00000000 } },
{ 0x00000190, { 0x0000025B,  0x00000000, 0, 0x00000000 } },
{ 0x00000191, { 0x00000192,  0x00000000, 0, 0x00000000 } },
{ 0x00000193, { 0x00000260,  0x00000000, 0, 0x00000000 } },
{ 0x00000194, { 0x00000263,  0x00000000, 0, 0x00000000 } },
{ 0x00000196, { 0x00000269,  0x00000000, 0, 0x00000000 } },
{ 0x00000197, { 0x00000268,  0x00000000, 0, 0x00000000 } },
{ 0x00000198, { 0x00000199,  0x00000000, 0, 0x00000000 } },
{ 0x0000019C, { 0x0000026F,  0x00000000, 0, 0x00000000 } },
{ 0x0000019D, { 0x00000272,  0x00000000, 0, 0x00000000 } },
{ 0x0000019F, { 0x00000275,  0x00000000, 0, 0x00000000 } },
{ 0x000001A0, { 0x000001A1,  0x00000000, 0, 0x00000000 } },
{ 0x000001A2, { 0x000001A3,  0x00000000, 0, 0x00000000 } },
{ 0x000001A4, { 0x000001A5,  0x00000000, 0, 0x00000000 } },
{ 0x000001A6, { 0x00000280,  0x00000000, 0, 0x00000000 } },
{ 0x000001A7, { 0x000001A8,  0x00000000, 0, 0x00000000 } },
{ 0x000001A9, { 0x00000283,  0x00000000, 0, 0x00000000 } },
{ 0x000001AC, { 0x000001AD,  0x00000000, 0, 0x00000000 } },
{ 0x000001AE, { 0x00000288,  0x00000000, 0, 0x00000000 } },
{ 0x000001AF, { 0x000001B0,  0x00000000, 0, 0x00000000 } },
{ 0x000001B1, { 0x0000028A,  0x00000000, 0, 0x00000000 } },
{ 0x000001B2, { 0x0000028B,  0x00000000, 0, 0x00000000 } },
{ 0x000001B3, { 0x000001B4,  0x00000000, 0, 0x00000000 } },
{ 0x000001B5, { 0x000001B6,  0x00000000, 0, 0x00000000 } },
{ 0x000001B7, { 0x00000292,  0x00000000, 0, 0x00000000 } },
{ 0x000001B8, { 0x000001B9,  0x00000000, 0, 0x00000000 } },
{ 0x000001BC, { 0x000001BD,  0x00000000, 0, 0x00000000 } },
{ 0x000001C4, { 0x000001C6,  0x00000000, 0, 0x00000000 } },
{ 0x000001C5, { 0x000001C6,  0x00000000, 0, 0x00000000 } },
{ 0x000001C7, { 0x000001C9,  0x00000000, 0, 0x00000000 } },
{ 0x000001C8, { 0x000001C9,  0x00000000, 0, 0x00000000 } },
{ 0x000001CA, { 0x000001CC,  0x00000000, 0, 0x00000000 } },
{ 0x000001CB, { 0x000001CC,  0x00000000, 0, 0x00000000 } },
{ 0x000001CD, { 0x000001CE,  0x00000000, 0, 0x00000000 } },
{ 0x000001CF, { 0x000001D0,  0x00000000, 0, 0x00000000 } },
{ 0x000001D1, { 0x000001D2,  0x00000000, 0, 0x00000000 } },
{ 0x000001D3, { 0x000001D4,  0x00000000, 0, 0x00000000 } },
{ 0x000001D5, { 0x000001D6,  0x00000000, 0, 0x00000000 } },
{ 0x000001D7, { 0x000001D8,  0x00000000, 0, 0x00000000 } },
{ 0x000001D9, { 0x000001DA,  0x00000000, 0, 0x00000000 } },
{ 0x000001DB, { 0x000001DC,  0x00000000, 0, 0x00000000 } },
{ 0x000001DE, { 0x000001DF,  0x00000000, 0, 0x00000000 } },
{ 0x000001E0, { 0x000001E1,  0x00000000, 0, 0x00000000 } },
{ 0x000001E2, { 0x000001E3,  0x00000000, 0, 0x00000000 } },
{ 0x000001E4, { 0x000001E5,  0x00000000, 0, 0x00000000 } },
{ 0x000001E6, { 0x000001E7,  0x00000000, 0, 0x00000000 } },
{ 0x000001E8, { 0x000001E9,  0x00000000, 0, 0x00000000 } },
{ 0x000001EA, { 0x000001EB,  0x00000000, 0, 0x00000000 } },
{ 0x000001EC, { 0x000001ED,  0x00000000, 0, 0x00000000 } },
{ 0x000001EE, { 0x000001EF,  0x00000000, 0, 0x00000000 } },
{ 0x000001F0, { 0x00000000,  0x00000000, U"\U0000006A\U0000030C", 0x00000000 } },
{ 0x000001F1, { 0x000001F3,  0x00000000, 0, 0x00000000 } },
{ 0x000001F2, { 0x000001F3,  0x00000000, 0, 0x00000000 } },
{ 0x000001F4, { 0x000001F5,  0x00000000, 0, 0x00000000 } },
{ 0x000001F6, { 0x00000195,  0x00000000, 0, 0x00000000 } },
{ 0x000001F7, { 0x000001BF,  0x00000000, 0, 0x00000000 } },
{ 0x000001F8, { 0x000001F9,  0x00000000, 0, 0x00000000 } },
{ 0x000001FA, { 0x000001FB,  0x00000000, 0, 0x00000000 } },
{ 0x000001FC, { 0x000001FD,  0x00000000, 0, 0x00000000 } },
{ 0x000001FE, { 0x000001FF,  0x00000000, 0, 0x00000000 } },
{ 0x00000200, { 0x00000201,  0x00000000, 0, 0x00000000 } },
{ 0x00000202, { 0x00000203,  0x00000000, 0, 0x00000000 } },
{ 0x00000204, { 0x00000205,  0x00000000, 0, 0x00000000 } },
{ 0x00000206, { 0x00000207,  0x00000000, 0, 0x00000000 } },
{ 0x00000208, { 0x00000209,  0x00000000, 0, 0x00000000 } },
{ 0x0000020A, { 0x0000020B,  0x00000000, 0, 0x00000000 } },
{ 0x0000020C, { 0x0000020D,  0x00000000, 0, 0x00000000 } },
{ 0x0000020E, { 0x0000020F,  0x00000000, 0, 0x00000000 } },
{ 0x00000210, { 0x00000211,  0x00000000, 0, 0x00000000 } },
{ 0x00000212, { 0x00000213,  0x00000000, 0, 0x00000000 } },
{ 0x00000214, { 0x00000215,  0x00000000, 0, 0x00000000 } },
{ 0x00000216, { 0x00000217,  0x00000000, 0, 0x00000000 } },
{ 0x00000218, { 0x00000219,  0x00000000, 0, 0x00000000 } },
{ 0x0000021A, { 0x0000021B,  0x00000000, 0, 0x00000000 } },
{ 0x0000021C, { 0x0000021D,  0x00000000, 0, 0x00000000 } },
{ 0x0000021E, { 0x0000021F,  0x00000000, 0, 0x00000000 } },
{ 0x00000220, { 0x0000019E,  0x00000000, 0, 0x00000000 } },
{ 0x00000222, { 0x00000223,  0x00000000, 0, 0x00000000 } },
{ 0x00000224, { 0x00000225,  0x00000000, 0, 0x00000000 } },
{ 0x00000226, { 0x00000227,  0x00000000, 0, 0x00000000 } },
{ 0x00000228, { 0x00000229,  0x00000000, 0, 0x00000000 } },
{ 0x0000022A, { 0x0000022B,  0x00000000, 0, 0x00000000 } },
{ 0x0000022C, { 0x0000022D,  0x00000000, 0, 0x00000000 } },
{ 0x0000022E, { 0x0000022F,  0x00000000, 0, 0x00000000 } },
{ 0x00000230, { 0x00000231,  0x00000000, 0, 0x00000000 } },
{ 0x00000232, { 0x00000233,  0x00000000, 0, 0x00000000 } },
{ 0x0000023A, { 0x00002C65,  0x00000000, 0, 0x00000000 } },
{ 0x0000023B, { 0x0000023C,  0x00000000, 0, 0x00000000 } },
{ 0x0000023D, { 0x0000019A,  0x00000000, 0, 0x00000000 } },
{ 0x0000023E, { 0x00002C66,  0x00000000, 0, 0x00000000 } },
{ 0x00000241, { 0x00000242,  0x00000000, 0, 0x00000000 } },
{ 0x00000243, { 0x00000180,  0x00000000, 0, 0x00000000 } },
{ 0x00000244, { 0x00000289,  0x00000000, 0, 0x00000000 } },
{ 0x00000245, { 0x0000028C,  0x00000000, 0, 0x00000000 } },
{ 0x00000246, { 0x00000247,  0x00000000, 0, 0x00000000 } },
{ 0x00000248, { 0x00000249,  0x00000000, 0, 0x00000000 } },
{ 0x0000024A, { 0x0000024B,  0x00000000, 0, 0x00000000 } },
{ 0x0000024C, { 0x0000024D,  0x00000000, 0, 0x00000000 } },
{ 0x0000024E, { 0x0000024F,  0x00000000, 0, 0x00000000 } },
{ 0x00010C9A, { 0x00010CDA,  0x00000000, 0, 0x00000000 } },
{ 0x00010C9C, { 0x00010CDC,  0x00000000, 0, 0x00000000 } },
{ 0x00010C9E, { 0x00010CDE,  0x00000000, 0, 0x00000000 } },
{ 0x00000345, { 0x000003B9,  0x00000000, 0, 0x00000000 } },
{ 0x00000370, { 0x00000371,  0x00000000, 0, 0x00000000 } },
{ 0x00000372, { 0x00000373,  0x00000000, 0, 0x00000000 } },
{ 0x00000376, { 0x00000377,  0x00000000, 0, 0x00000000 } },
{ 0x0000037F, { 0x000003F3,  0x00000000, 0, 0x00000000 } },
{ 0x00000386, { 0x000003AC,  0x00000000, 0, 0x00000000 } },
{ 0x00000388, { 0x000003AD,  0x00000000, 0, 0x00000000 } },
{ 0x00000389, { 0x000003AE,  0x00000000, 0, 0x00000000 } },
{ 0x0000038A, { 0x000003AF,  0x00000000, 0, 0x00000000 } },
{ 0x0000038C, { 0x000003CC,  0x00000000, 0, 0x00000000 } },
{ 0x0000038E, { 0x000003CD,  0x00000000, 0, 0x00000000 } },
{ 0x0000038F, { 0x000003CE,  0x00000000, 0, 0x00000000 } },
{ 0x00000390, { 0x00000000,  0x00000000, U"\U000003B9\U00000308\U00000301", 0x00000000 } },
{ 0x00000391, { 0x000003B1,  0x00000000, 0, 0x00000000 } },
{ 0x00000392, { 0x000003B2,  0x00000000, 0, 0x00000000 } },
{ 0x00000393, { 0x000003B3,  0x00000000, 0, 0x00000000 } },
{ 0x00000394, { 0x000003B4,  0x00000000, 0, 0x00000000 } },
{ 0x00000395, { 0x000003B5,  0x00000000, 0, 0x00000000 } },
{ 0x00000396, { 0x000003B6,  0x00000000, 0, 0x00000000 } },
{ 0x00000397, { 0x000003B7,  0x00000000, 0, 0x00000000 } },
{ 0x00000398, { 0x000003B8,  0x00000000, 0, 0x00000000 } },
{ 0x00000399, { 0x000003B9,  0x00000000, 0, 0x00000000 } },
{ 0x0000039A, { 0x000003BA,  0x00000000, 0, 0x00000000 } },
{ 0x0000039B, { 0x000003BB,  0x00000000, 0, 0x00000000 } },
{ 0x0000039C, { 0x000003BC,  0x00000000, 0, 0x00000000 } },
{ 0x0000039D, { 0x000003BD,  0x00000000, 0, 0x00000000 } },
{ 0x0000039E, { 0x000003BE,  0x00000000, 0, 0x00000000 } },
{ 0x0000039F, { 0x000003BF,  0x00000000, 0, 0x00000000 } },
{ 0x000003A0, { 0x000003C0,  0x00000000, 0, 0x00000000 } },
{ 0x000003A1, { 0x000003C1,  0x00000000, 0, 0x00000000 } },
{ 0x000003A3, { 0x000003C3,  0x00000000, 0, 0x00000000 } },
{ 0x000003A4, { 0x000003C4,  0x00000000, 0, 0x00000000 } },
{ 0x000003A5, { 0x000003C5,  0x00000000, 0, 0x00000000 } },
{ 0x000003A6, { 0x000003C6,  0x00000000, 0, 0x00000000 } },
{ 0x000003A7, { 0x000003C7,  0x00000000, 0, 0x00000000 } },
{ 0x000003A8, { 0x000003C8,  0x00000000, 0, 0x00000000 } },
{ 0x000003A9, { 0x000003C9,  0x00000000, 0, 0x00000000 } },
{ 0x000003AA, { 0x000003CA,  0x00000000, 0, 0x00000000 } },
{ 0x000003AB, { 0x000003CB,  0x00000000, 0, 0x00000000 } },
{ 0x000003B0, { 0x00000000,  0x00000000, U"\U000003C5\U00000308\U00000301", 0x00000000 } },
{ 0x000003C2, { 0x000003C3,  0x00000000, 0, 0x00000000 } },
{ 0x000003CF, { 0x000003D7,  0x00000000, 0, 0x00000000 } },
{ 0x000003D0, { 0x000003B2,  0x00000000, 0, 0x00000000 } },
{ 0x000003D1, { 0x000003B8,  0x00000000, 0, 0x00000000 } },
{ 0x000003D5, { 0x000003C6,  0x00000000, 0, 0x00000000 } },
{ 0x000003D6, { 0x000003C0,  0x00000000, 0, 0x00000000 } },
{ 0x000003D8, { 0x000003D9,  0x00000000, 0, 0x00000000 } },
{ 0x000003DA, { 0x000003DB,  0x00000000, 0, 0x00000000 } },
{ 0x000003DC, { 0x000003DD,  0x00000000, 0, 0x00000000 } },
{ 0x000003DE, { 0x000003DF,  0x00000000, 0, 0x00000000 } },
{ 0x000003E0, { 0x000003E1,  0x00000000, 0, 0x00000000 } },
{ 0x000003E2, { 0x000003E3,  0x00000000, 0, 0x00000000 } },
{ 0x000003E4, { 0x000003E5,  0x00000000, 0, 0x00000000 } },
{ 0x000003E6, { 0x000003E7,  0x00000000, 0, 0x00000000 } },
{ 0x000003E8, { 0x000003E9,  0x00000000, 0, 0x00000000 } },
{ 0x000003EA, { 0x000003EB,  0x00000000, 0, 0x00000000 } },
{ 0x000003EC, { 0x000003ED,  0x00000000, 0, 0x00000000 } },
{ 0x000003EE, { 0x000003EF,  0x00000000, 0, 0x00000000 } },
{ 0x000003F0, { 0x000003BA,  0x00000000, 0, 0x00000000 } },
{ 0x000003F1, { 0x000003C1,  0x00000000, 0, 0x00000000 } },
{ 0x000003F4, { 0x000003B8,  0x00000000, 0, 0x00000000 } },
{ 0x000003F5, { 0x000003B5,  0x00000000, 0, 0x00000000 } },
{ 0x000003F7, { 0x000003F8,  0x00000000, 0, 0x00000000 } },
{ 0x000003F9, { 0x000003F2,  0x00000000, 0, 0x00000000 } },
{ 0x000003FA, { 0x000003FB,  0x00000000, 0, 0x00000000 } },
{ 0x000003FD, { 0x0000037B,  0x00000000, 0, 0x00000000 } },
{ 0x000003FE, { 0x0000037C,  0x00000000, 0, 0x00000000 } },
{ 0x000003FF, { 0x0000037D,  0x00000000, 0, 0x00000000 } },
{ 0x00010400, { 0x00010428,  0x00000000, 0, 0x00000000 } },
{ 0x00000401, { 0x00000451,  0x00000000, 0, 0x00000000 } },
{ 0x00010402, { 0x0001042A,  0x00000000, 0, 0x00000000 } },
{ 0x00010403, { 0x0001042B,  0x00000000, 0, 0x00000000 } },
{ 0x00010404, { 0x0001042C,  0x00000000, 0, 0x00000000 } },
{ 0x00010405, { 0x0001042D,  0x00000000, 0, 0x00000000 } },
{ 0x00010406, { 0x0001042E,  0x00000000, 0, 0x00000000 } },
{ 0x00010407, { 0x0001042F,  0x00000000, 0, 0x00000000 } },
{ 0x00000408, { 0x00000458,  0x00000000, 0, 0x00000000 } },
{ 0x00010409, { 0x00010431,  0x00000000, 0, 0x00000000 } },
{ 0x0001040A, { 0x00010432,  0x00000000, 0, 0x00000000 } },
{ 0x0001040B, { 0x00010433,  0x00000000, 0, 0x00000000 } },
{ 0x0001040C, { 0x00010434,  0x00000000, 0, 0x00000000 } },
{ 0x0000040D, { 0x0000045D,  0x00000000, 0, 0x00000000 } },
{ 0x0000040E, { 0x0000045E,  0x00000000, 0, 0x00000000 } },
{ 0x0001040F, { 0x00010437,  0x00000000, 0, 0x00000000 } },
{ 0x00010410, { 0x00010438,  0x00000000, 0, 0x00000000 } },
{ 0x00000411, { 0x00000431,  0x00000000, 0, 0x00000000 } },
{ 0x00010412, { 0x0001043A,  0x00000000, 0, 0x00000000 } },
{ 0x00010413, { 0x0001043B,  0x00000000, 0, 0x00000000 } },
{ 0x00010414, { 0x0001043C,  0x00000000, 0, 0x00000000 } },
{ 0x00010415, { 0x0001043D,  0x00000000, 0, 0x00000000 } },
{ 0x00000416, { 0x00000436,  0x00000000, 0, 0x00000000 } },
{ 0x00010417, { 0x0001043F,  0x00000000, 0, 0x00000000 } },
{ 0x00000418, { 0x00000438,  0x00000000, 0, 0x00000000 } },
{ 0x00000419, { 0x00000439,  0x00000000, 0, 0x00000000 } },
{ 0x0001041A, { 0x00010442,  0x00000000, 0, 0x00000000 } },
{ 0x0000041B, { 0x0000043B,  0x00000000, 0, 0x00000000 } },
{ 0x0001041C, { 0x00010444,  0x00000000, 0, 0x00000000 } },
{ 0x0001041D, { 0x00010445,  0x00000000, 0, 0x00000000 } },
{ 0x0001041E, { 0x00010446,  0x00000000, 0, 0x00000000 } },
{ 0x0001041F, { 0x00010447,  0x00000000, 0, 0x00000000 } },
{ 0x00010420, { 0x00010448,  0x00000000, 0, 0x00000000 } },
{ 0x00000421, { 0x00000441,  0x00000000, 0, 0x00000000 } },
{ 0x00000422, { 0x00000442,  0x00000000, 0, 0x00000000 } },
{ 0x00000423, { 0x00000443,  0x00000000, 0, 0x00000000 } },
{ 0x00000424, { 0x00000444,  0x00000000, 0, 0x00000000 } },
{ 0x00000425, { 0x00000445,  0x00000000, 0, 0x00000000 } },
{ 0x00000426, { 0x00000446,  0x00000000, 0, 0x00000000 } },
{ 0x00000427, { 0x00000447,  0x00000000, 0, 0x00000000 } },
{ 0x00000428, { 0x00000448,  0x00000000, 0, 0x00000000 } },
{ 0x00000429, { 0x00000449,  0x00000000, 0, 0x00000000 } },
{ 0x0000042A, { 0x0000044A,  0x00000000, 0, 0x00000000 } },
{ 0x0000042B, { 0x0000044B,  0x00000000, 0, 0x00000000 } },
{ 0x0000042C, { 0x0000044C,  0x00000000, 0, 0x00000000 } },
{ 0x0000042D, { 0x0000044D,  0x00000000, 0, 0x00000000 } },
{ 0x0000042E, { 0x0000044E,  0x00000000, 0, 0x00000000 } },
{ 0x0000042F, { 0x0000044F,  0x00000000, 0, 0x00000000 } },
{ 0x00000460, { 0x00000461,  0x00000000, 0, 0x00000000 } },
{ 0x00000462, { 0x00000463,  0x00000000, 0, 0x00000000 } },
{ 0x00000464, { 0x00000465,  0x00000000, 0, 0x00000000 } },
{ 0x00000466, { 0x00000467,  0x00000000, 0, 0x00000000 } },
{ 0x00000468, { 0x00000469,  0x00000000, 0, 0x00000000 } },
{ 0x0000046A, { 0x0000046B,  0x00000000, 0, 0x00000000 } },
{ 0x0000046C, { 0x0000046D,  0x00000000, 0, 0x00000000 } },
{ 0x0000046E, { 0x0000046F,  0x00000000, 0, 0x00000000 } },
{ 0x00000470, { 0x00000471,  0x00000000, 0, 0x00000000 } },
{ 0x00000472, { 0x00000473,  0x00000000, 0, 0x00000000 } },
{ 0x00000474, { 0x00000475,  0x00000000, 0, 0x00000000 } },
{ 0x00000476, { 0x00000477,  0x00000000, 0, 0x00000000 } },
{ 0x00000478, { 0x00000479,  0x00000000, 0, 0x00000000 } },
{ 0x0000047A, { 0x0000047B,  0x00000000, 0, 0x00000000 } },
{ 0x0000047C, { 0x0000047D,  0x00000000, 0, 0x00000000 } },
{ 0x0000047E, { 0x0000047F,  0x00000000, 0, 0x00000000 } },
{ 0x00000480, { 0x00000481,  0x00000000, 0, 0x00000000 } },
{ 0x0000048A, { 0x0000048B,  0x00000000, 0, 0x00000000 } },
{ 0x0000048C, { 0x0000048D,  0x00000000, 0, 0x00000000 } },
{ 0x0000048E, { 0x0000048F,  0x00000000, 0, 0x00000000 } },
{ 0x00000490, { 0x00000491,  0x00000000, 0, 0x00000000 } },
{ 0x00000492, { 0x00000493,  0x00000000, 0, 0x00000000 } },
{ 0x00010CAC, { 0x00010CEC,  0x00000000, 0, 0x00000000 } },
{ 0x00000494, { 0x00000495,  0x00000000, 0, 0x00000000 } },
{ 0x00000496, { 0x00000497,  0x00000000, 0, 0x00000000 } },
{ 0x00000498, { 0x00000499,  0x00000000, 0, 0x00000000 } },
{ 0x0000049A, { 0x0000049B,  0x00000000, 0, 0x00000000 } },
{ 0x0000049C, { 0x0000049D,  0x00000000, 0, 0x00000000 } },
{ 0x0000049E, { 0x0000049F,  0x00000000, 0, 0x00000000 } },
{ 0x000004A0, { 0x000004A1,  0x00000000, 0, 0x00000000 } },
{ 0x000004A2, { 0x000004A3,  0x00000000, 0, 0x00000000 } },
{ 0x000004A4, { 0x000004A5,  0x00000000, 0, 0x00000000 } },
{ 0x000004A6, { 0x000004A7,  0x00000000, 0, 0x00000000 } },
{ 0x000004A8, { 0x000004A9,  0x00000000, 0, 0x00000000 } },
{ 0x000004AA, { 0x000004AB,  0x00000000, 0, 0x00000000 } },
{ 0x000004AC, { 0x000004AD,  0x00000000, 0, 0x00000000 } },
{ 0x000004AE, { 0x000004AF,  0x00000000, 0, 0x00000000 } },
{ 0x000104B0, { 0x000104D8,  0x00000000, 0, 0x00000000 } },
{ 0x000104B1, { 0x000104D9,  0x00000000, 0, 0x00000000 } },
{ 0x000004B2, { 0x000004B3,  0x00000000, 0, 0x00000000 } },
{ 0x000104B3, { 0x000104DB,  0x00000000, 0, 0x00000000 } },
{ 0x000104B4, { 0x000104DC,  0x00000000, 0, 0x00000000 } },
{ 0x000104B5, { 0x000104DD,  0x00000000, 0, 0x00000000 } },
{ 0x000104B6, { 0x000104DE,  0x00000000, 0, 0x00000000 } },
{ 0x000024B7, { 0x000024D1,  0x00000000, 0, 0x00000000 } },
{ 0x000104B8, { 0x000104E0,  0x00000000, 0, 0x00000000 } },
{ 0x000104B9, { 0x000104E1,  0x00000000, 0, 0x00000000 } },
{ 0x000104BA, { 0x000104E2,  0x00000000, 0, 0x00000000 } },
{ 0x000104BB, { 0x000104E3,  0x00000000, 0, 0x00000000 } },
{ 0x000024BC, { 0x000024D6,  0x00000000, 0, 0x00000000 } },
{ 0x000104BD, { 0x000104E5,  0x00000000, 0, 0x00000000 } },
{ 0x000024BE, { 0x000024D8,  0x00000000, 0, 0x00000000 } },
{ 0x000104BF, { 0x000104E7,  0x00000000, 0, 0x00000000 } },
{ 0x000104C0, { 0x000104E8,  0x00000000, 0, 0x00000000 } },
{ 0x000104C1, { 0x000104E9,  0x00000000, 0, 0x00000000 } },
{ 0x000104C2, { 0x000104EA,  0x00000000, 0, 0x00000000 } },
{ 0x000024C3, { 0x000024DD,  0x00000000, 0, 0x00000000 } },
{ 0x000104C4, { 0x000104EC,  0x00000000, 0, 0x00000000 } },
{ 0x000104C5, { 0x000104ED,  0x00000000, 0, 0x00000000 } },
{ 0x000104C6, { 0x000104EE,  0x00000000, 0, 0x00000000 } },
{ 0x000024C7, { 0x000024E1,  0x00000000, 0, 0x00000000 } },
{ 0x000024C8, { 0x000024E2,  0x00000000, 0, 0x00000000 } },
{ 0x000024C9, { 0x000024E3,  0x00000000, 0, 0x00000000 } },
{ 0x000024CA, { 0x000024E4,  0x00000000, 0, 0x00000000 } },
{ 0x000104CB, { 0x000104F3,  0x00000000, 0, 0x00000000 } },
{ 0x000104CC, { 0x000104F4,  0x00000000, 0, 0x00000000 } },
{ 0x000104CD, { 0x000104F5,  0x00000000, 0, 0x00000000 } },
{ 0x000104CE, { 0x000104F6,  0x00000000, 0, 0x00000000 } },
{ 0x000104CF, { 0x000104F7,  0x00000000, 0, 0x00000000 } },
{ 0x000104D0, { 0x000104F8,  0x00000000, 0, 0x00000000 } },
{ 0x000104D1, { 0x000104F9,  0x00000000, 0, 0x00000000 } },
{ 0x000104D2, { 0x000104FA,  0x00000000, 0, 0x00000000 } },
{ 0x000104D3, { 0x000104FB,  0x00000000, 0, 0x00000000 } },
{ 0x000004D4, { 0x000004D5,  0x00000000, 0, 0x00000000 } },
{ 0x000004D6, { 0x000004D7,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB79, { 0x000013A9,  0x00000000, 0, 0x00000000 } },
{ 0x000004D8, { 0x000004D9,  0x00000000, 0, 0x00000000 } },
{ 0x000004DA, { 0x000004DB,  0x00000000, 0, 0x00000000 } },
{ 0x000004DC, { 0x000004DD,  0x00000000, 0, 0x00000000 } },
{ 0x000004DE, { 0x000004DF,  0x00000000, 0, 0x00000000 } },
{ 0x000004E0, { 0x000004E1,  0x00000000, 0, 0x00000000 } },
{ 0x000004E2, { 0x000004E3,  0x00000000, 0, 0x00000000 } },
{ 0x000004E4, { 0x000004E5,  0x00000000, 0, 0x00000000 } },
{ 0x000004E6, { 0x000004E7,  0x00000000, 0, 0x00000000 } },
{ 0x000004E8, { 0x000004E9,  0x00000000, 0, 0x00000000 } },
{ 0x000004EA, { 0x000004EB,  0x00000000, 0, 0x00000000 } },
{ 0x000004EC, { 0x000004ED,  0x00000000, 0, 0x00000000 } },
{ 0x000004EE, { 0x000004EF,  0x00000000, 0, 0x00000000 } },
{ 0x000004F0, { 0x000004F1,  0x00000000, 0, 0x00000000 } },
{ 0x000004F2, { 0x000004F3,  0x00000000, 0, 0x00000000 } },
{ 0x000004F4, { 0x000004F5,  0x00000000, 0, 0x00000000 } },
{ 0x000004F6, { 0x000004F7,  0x00000000, 0, 0x00000000 } },
{ 0x000004F8, { 0x000004F9,  0x00000000, 0, 0x00000000 } },
{ 0x000004FA, { 0x000004FB,  0x00000000, 0, 0x00000000 } },
{ 0x000004FC, { 0x000004FD,  0x00000000, 0, 0x00000000 } },
{ 0x000004FE, { 0x000004FF,  0x00000000, 0, 0x00000000 } },
{ 0x00000500, { 0x00000501,  0x00000000, 0, 0x00000000 } },
{ 0x00000502, { 0x00000503,  0x00000000, 0, 0x00000000 } },
{ 0x00000504, { 0x00000505,  0x00000000, 0, 0x00000000 } },
{ 0x00000506, { 0x00000507,  0x00000000, 0, 0x00000000 } },
{ 0x00000508, { 0x00000509,  0x00000000, 0, 0x00000000 } },
{ 0x0000050A, { 0x0000050B,  0x00000000, 0, 0x00000000 } },
{ 0x00010CB0, { 0x00010CF0,  0x00000000, 0, 0x00000000 } },
{ 0x0000050C, { 0x0000050D,  0x00000000, 0, 0x00000000 } },
{ 0x0000050E, { 0x0000050F,  0x00000000, 0, 0x00000000 } },
{ 0x00000510, { 0x00000511,  0x00000000, 0, 0x00000000 } },
{ 0x00000512, { 0x00000513,  0x00000000, 0, 0x00000000 } },
{ 0x00000514, { 0x00000515,  0x00000000, 0, 0x00000000 } },
{ 0x00000516, { 0x00000517,  0x00000000, 0, 0x00000000 } },
{ 0x00000518, { 0x00000519,  0x00000000, 0, 0x00000000 } },
{ 0x0000051A, { 0x0000051B,  0x00000000, 0, 0x00000000 } },
{ 0x0000051C, { 0x0000051D,  0x00000000, 0, 0x00000000 } },
{ 0x0000051E, { 0x0000051F,  0x00000000, 0, 0x00000000 } },
{ 0x00000520, { 0x00000521,  0x00000000, 0, 0x00000000 } },
{ 0x00000522, { 0x00000523,  0x00000000, 0, 0x00000000 } },
{ 0x00000524, { 0x00000525,  0x00000000, 0, 0x00000000 } },
{ 0x00000526, { 0x00000527,  0x00000000, 0, 0x00000000 } },
{ 0x00000528, { 0x00000529,  0x00000000, 0, 0x00000000 } },
{ 0x0000052A, { 0x0000052B,  0x00000000, 0, 0x00000000 } },
{ 0x0000052C, { 0x0000052D,  0x00000000, 0, 0x00000000 } },
{ 0x0000052E, { 0x0000052F,  0x00000000, 0, 0x00000000 } },
{ 0x00000531, { 0x00000561,  0x00000000, 0, 0x00000000 } },
{ 0x00000532, { 0x00000562,  0x00000000, 0, 0x00000000 } },
{ 0x00000533, { 0x00000563,  0x00000000, 0, 0x00000000 } },
{ 0x00000534, { 0x00000564,  0x00000000, 0, 0x00000000 } },
{ 0x00000535, { 0x00000565,  0x00000000, 0, 0x00000000 } },
{ 0x00000536, { 0x00000566,  0x00000000, 0, 0x00000000 } },
{ 0x00000537, { 0x00000567,  0x00000000, 0, 0x00000000 } },
{ 0x00000538, { 0x00000568,  0x00000000, 0, 0x00000000 } },
{ 0x00000539, { 0x00000569,  0x00000000, 0, 0x00000000 } },
{ 0x0000053A, { 0x0000056A,  0x00000000, 0, 0x00000000 } },
{ 0x0000053B, { 0x0000056B,  0x00000000, 0, 0x00000000 } },
{ 0x0000053C, { 0x0000056C,  0x00000000, 0, 0x00000000 } },
{ 0x0000053D, { 0x0000056D,  0x00000000, 0, 0x00000000 } },
{ 0x0000053E, { 0x0000056E,  0x00000000, 0, 0x00000000 } },
{ 0x0000053F, { 0x0000056F,  0x00000000, 0, 0x00000000 } },
{ 0x00000540, { 0x00000570,  0x00000000, 0, 0x00000000 } },
{ 0x00000541, { 0x00000571,  0x00000000, 0, 0x00000000 } },
{ 0x00000542, { 0x00000572,  0x00000000, 0, 0x00000000 } },
{ 0x00000543, { 0x00000573,  0x00000000, 0, 0x00000000 } },
{ 0x00000544, { 0x00000574,  0x00000000, 0, 0x00000000 } },
{ 0x00000545, { 0x00000575,  0x00000000, 0, 0x00000000 } },
{ 0x00000546, { 0x00000576,  0x00000000, 0, 0x00000000 } },
{ 0x00000547, { 0x00000577,  0x00000000, 0, 0x00000000 } },
{ 0x00000548, { 0x00000578,  0x00000000, 0, 0x00000000 } },
{ 0x00000549, { 0x00000579,  0x00000000, 0, 0x00000000 } },
{ 0x0000054A, { 0x0000057A,  0x00000000, 0, 0x00000000 } },
{ 0x0000054B, { 0x0000057B,  0x00000000, 0, 0x00000000 } },
{ 0x0000054C, { 0x0000057C,  0x00000000, 0, 0x00000000 } },
{ 0x0000054D, { 0x0000057D,  0x00000000, 0, 0x00000000 } },
{ 0x0000054E, { 0x0000057E,  0x00000000, 0, 0x00000000 } },
{ 0x0000054F, { 0x0000057F,  0x00000000, 0, 0x00000000 } },
{ 0x00000550, { 0x00000580,  0x00000000, 0, 0x00000000 } },
{ 0x00000551, { 0x00000581,  0x00000000, 0, 0x00000000 } },
{ 0x00000552, { 0x00000582,  0x00000000, 0, 0x00000000 } },
{ 0x00000553, { 0x00000583,  0x00000000, 0, 0x00000000 } },
{ 0x00000554, { 0x00000584,  0x00000000, 0, 0x00000000 } },
{ 0x00000555, { 0x00000585,  0x00000000, 0, 0x00000000 } },
{ 0x00000556, { 0x00000586,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB91, { 0x000013C1,  0x00000000, 0, 0x00000000 } },
{ 0x00000587, { 0x00000000,  0x00000000, U"\U00000565\U00000582", 0x00000000 } },
{ 0x0000A640, { 0x0000A641,  0x00000000, 0, 0x00000000 } },
{ 0x0000A642, { 0x0000A643,  0x00000000, 0, 0x00000000 } },
{ 0x0000A644, { 0x0000A645,  0x00000000, 0, 0x00000000 } },
{ 0x0000A646, { 0x0000A647,  0x00000000, 0, 0x00000000 } },
{ 0x0000A648, { 0x0000A649,  0x00000000, 0, 0x00000000 } },
{ 0x0000A64A, { 0x0000A64B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A64C, { 0x0000A64D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A64E, { 0x0000A64F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A650, { 0x0000A651,  0x00000000, 0, 0x00000000 } },
{ 0x0000A652, { 0x0000A653,  0x00000000, 0, 0x00000000 } },
{ 0x0000A654, { 0x0000A655,  0x00000000, 0, 0x00000000 } },
{ 0x0000A656, { 0x0000A657,  0x00000000, 0, 0x00000000 } },
{ 0x0000A658, { 0x0000A659,  0x00000000, 0, 0x00000000 } },
{ 0x0000A65A, { 0x0000A65B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A65C, { 0x0000A65D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A65E, { 0x0000A65F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A660, { 0x0000A661,  0x00000000, 0, 0x00000000 } },
{ 0x0000A662, { 0x0000A663,  0x00000000, 0, 0x00000000 } },
{ 0x0000A664, { 0x0000A665,  0x00000000, 0, 0x00000000 } },
{ 0x0000A666, { 0x0000A667,  0x00000000, 0, 0x00000000 } },
{ 0x0000A668, { 0x0000A669,  0x00000000, 0, 0x00000000 } },
{ 0x0000A66A, { 0x0000A66B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A66C, { 0x0000A66D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A680, { 0x0000A681,  0x00000000, 0, 0x00000000 } },
{ 0x0000A682, { 0x0000A683,  0x00000000, 0, 0x00000000 } },
{ 0x0000A684, { 0x0000A685,  0x00000000, 0, 0x00000000 } },
{ 0x0000A686, { 0x0000A687,  0x00000000, 0, 0x00000000 } },
{ 0x0000A688, { 0x0000A689,  0x00000000, 0, 0x00000000 } },
{ 0x0000A68A, { 0x0000A68B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A68C, { 0x0000A68D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A68E, { 0x0000A68F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A690, { 0x0000A691,  0x00000000, 0, 0x00000000 } },
{ 0x0000A692, { 0x0000A693,  0x00000000, 0, 0x00000000 } },
{ 0x0000A694, { 0x0000A695,  0x00000000, 0, 0x00000000 } },
{ 0x0000A696, { 0x0000A697,  0x00000000, 0, 0x00000000 } },
{ 0x0000A698, { 0x0000A699,  0x00000000, 0, 0x00000000 } },
{ 0x0000A69A, { 0x0000A69B,  0x00000000, 0, 0x00000000 } },
{ 0x00002126, { 0x000003C9,  0x00000000, 0, 0x00000000 } },
{ 0x0000212A, { 0x0000006B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A722, { 0x0000A723,  0x00000000, 0, 0x00000000 } },
{ 0x0000A724, { 0x0000A725,  0x00000000, 0, 0x00000000 } },
{ 0x0000A726, { 0x0000A727,  0x00000000, 0, 0x00000000 } },
{ 0x0000A728, { 0x0000A729,  0x00000000, 0, 0x00000000 } },
{ 0x0000A72A, { 0x0000A72B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A72C, { 0x0000A72D,  0x00000000, 0, 0x00000000 } },
{ 0x00002132, { 0x0000214E,  0x00000000, 0, 0x00000000 } },
{ 0x0000A72E, { 0x0000A72F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A732, { 0x0000A733,  0x00000000, 0, 0x00000000 } },
{ 0x0000A734, { 0x0000A735,  0x00000000, 0, 0x00000000 } },
{ 0x0000A736, { 0x0000A737,  0x00000000, 0, 0x00000000 } },
{ 0x0000A738, { 0x0000A739,  0x00000000, 0, 0x00000000 } },
{ 0x0000A73A, { 0x0000A73B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A73C, { 0x0000A73D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A73E, { 0x0000A73F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A740, { 0x0000A741,  0x00000000, 0, 0x00000000 } },
{ 0x0000A742, { 0x0000A743,  0x00000000, 0, 0x00000000 } },
{ 0x0000A744, { 0x0000A745,  0x00000000, 0, 0x00000000 } },
{ 0x0000A746, { 0x0000A747,  0x00000000, 0, 0x00000000 } },
{ 0x0000A748, { 0x0000A749,  0x00000000, 0, 0x00000000 } },
{ 0x0000A74A, { 0x0000A74B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A74C, { 0x0000A74D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A74E, { 0x0000A74F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A750, { 0x0000A751,  0x00000000, 0, 0x00000000 } },
{ 0x0000A752, { 0x0000A753,  0x00000000, 0, 0x00000000 } },
{ 0x0000A754, { 0x0000A755,  0x00000000, 0, 0x00000000 } },
{ 0x0000A756, { 0x0000A757,  0x00000000, 0, 0x00000000 } },
{ 0x0000A758, { 0x0000A759,  0x00000000, 0, 0x00000000 } },
{ 0x0000A75A, { 0x0000A75B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A75C, { 0x0000A75D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A75E, { 0x0000A75F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A760, { 0x0000A761,  0x00000000, 0, 0x00000000 } },
{ 0x0000A762, { 0x0000A763,  0x00000000, 0, 0x00000000 } },
{ 0x0000A764, { 0x0000A765,  0x00000000, 0, 0x00000000 } },
{ 0x0000A766, { 0x0000A767,  0x00000000, 0, 0x00000000 } },
{ 0x0000A768, { 0x0000A769,  0x00000000, 0, 0x00000000 } },
{ 0x0000A76A, { 0x0000A76B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A76C, { 0x0000A76D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A76E, { 0x0000A76F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A779, { 0x0000A77A,  0x00000000, 0, 0x00000000 } },
{ 0x0000A77B, { 0x0000A77C,  0x00000000, 0, 0x00000000 } },
{ 0x0000A77D, { 0x00001D79,  0x00000000, 0, 0x00000000 } },
{ 0x0000A77E, { 0x0000A77F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A780, { 0x0000A781,  0x00000000, 0, 0x00000000 } },
{ 0x0000A782, { 0x0000A783,  0x00000000, 0, 0x00000000 } },
{ 0x0000A784, { 0x0000A785,  0x00000000, 0, 0x00000000 } },
{ 0x0000A786, { 0x0000A787,  0x00000000, 0, 0x00000000 } },
{ 0x0000A78B, { 0x0000A78C,  0x00000000, 0, 0x00000000 } },
{ 0x0000A78D, { 0x00000265,  0x00000000, 0, 0x00000000 } },
{ 0x0000A790, { 0x0000A791,  0x00000000, 0, 0x00000000 } },
{ 0x0000A792, { 0x0000A793,  0x00000000, 0, 0x00000000 } },
{ 0x0000A796, { 0x0000A797,  0x00000000, 0, 0x00000000 } },
{ 0x0000A798, { 0x0000A799,  0x00000000, 0, 0x00000000 } },
{ 0x0000A79A, { 0x0000A79B,  0x00000000, 0, 0x00000000 } },
{ 0x0000A79C, { 0x0000A79D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A79E, { 0x0000A79F,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7A0, { 0x0000A7A1,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7A2, { 0x0000A7A3,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7A4, { 0x0000A7A5,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7A6, { 0x0000A7A7,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7A8, { 0x0000A7A9,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7AA, { 0x00000266,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7AB, { 0x0000025C,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7AC, { 0x00000261,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7AD, { 0x0000026C,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7AE, { 0x0000026A,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7B0, { 0x0000029E,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7B1, { 0x00000287,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7B2, { 0x0000029D,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7B3, { 0x0000AB53,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7B4, { 0x0000A7B5,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7B6, { 0x0000A7B7,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7B8, { 0x0000A7B9,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7BA, { 0x0000A7BB,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7BC, { 0x0000A7BD,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7BE, { 0x0000A7BF,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7C2, { 0x0000A7C3,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7C4, { 0x0000A794,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7C5, { 0x00000282,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7C6, { 0x00001D8E,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7C7, { 0x0000A7C8,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7C9, { 0x0000A7CA,  0x00000000, 0, 0x00000000 } },
{ 0x0000A7F5, { 0x0000A7F6,  0x00000000, 0, 0x00000000 } },
{ 0x00001F2B, { 0x00001F23,  0x00000000, 0, 0x00000000 } },
{ 0x00002160, { 0x00002170,  0x00000000, 0, 0x00000000 } },
{ 0x00002162, { 0x00002172,  0x00000000, 0, 0x00000000 } },
{ 0x00002164, { 0x00002174,  0x00000000, 0, 0x00000000 } },
{ 0x00000166, { 0x00000167,  0x00000000, 0, 0x00000000 } },
{ 0x00000168, { 0x00000169,  0x00000000, 0, 0x00000000 } },
{ 0x0000016A, { 0x0000016B,  0x00000000, 0, 0x00000000 } },
{ 0x0000016C, { 0x0000016D,  0x00000000, 0, 0x00000000 } },
{ 0x0000016E, { 0x0000016F,  0x00000000, 0, 0x00000000 } },
{ 0x0001E900, { 0x0001E922,  0x00000000, 0, 0x00000000 } },
{ 0x0001E901, { 0x0001E923,  0x00000000, 0, 0x00000000 } },
{ 0x0001E902, { 0x0001E924,  0x00000000, 0, 0x00000000 } },
{ 0x0001E903, { 0x0001E925,  0x00000000, 0, 0x00000000 } },
{ 0x0001E904, { 0x0001E926,  0x00000000, 0, 0x00000000 } },
{ 0x0001E905, { 0x0001E927,  0x00000000, 0, 0x00000000 } },
{ 0x0001E906, { 0x0001E928,  0x00000000, 0, 0x00000000 } },
{ 0x0001E907, { 0x0001E929,  0x00000000, 0, 0x00000000 } },
{ 0x0001E908, { 0x0001E92A,  0x00000000, 0, 0x00000000 } },
{ 0x0001E909, { 0x0001E92B,  0x00000000, 0, 0x00000000 } },
{ 0x0001E90A, { 0x0001E92C,  0x00000000, 0, 0x00000000 } },
{ 0x0001E90B, { 0x0001E92D,  0x00000000, 0, 0x00000000 } },
{ 0x0001E90C, { 0x0001E92E,  0x00000000, 0, 0x00000000 } },
{ 0x0001E90D, { 0x0001E92F,  0x00000000, 0, 0x00000000 } },
{ 0x0001E90E, { 0x0001E930,  0x00000000, 0, 0x00000000 } },
{ 0x0001E90F, { 0x0001E931,  0x00000000, 0, 0x00000000 } },
{ 0x0001E910, { 0x0001E932,  0x00000000, 0, 0x00000000 } },
{ 0x0001E911, { 0x0001E933,  0x00000000, 0, 0x00000000 } },
{ 0x0001E912, { 0x0001E934,  0x00000000, 0, 0x00000000 } },
{ 0x0001E913, { 0x0001E935,  0x00000000, 0, 0x00000000 } },
{ 0x0001E914, { 0x0001E936,  0x00000000, 0, 0x00000000 } },
{ 0x0001E915, { 0x0001E937,  0x00000000, 0, 0x00000000 } },
{ 0x0001E916, { 0x0001E938,  0x00000000, 0, 0x00000000 } },
{ 0x0001E917, { 0x0001E939,  0x00000000, 0, 0x00000000 } },
{ 0x0001E918, { 0x0001E93A,  0x00000000, 0, 0x00000000 } },
{ 0x0001E919, { 0x0001E93B,  0x00000000, 0, 0x00000000 } },
{ 0x0001E91A, { 0x0001E93C,  0x00000000, 0, 0x00000000 } },
{ 0x0001E91B, { 0x0001E93D,  0x00000000, 0, 0x00000000 } },
{ 0x0001E91C, { 0x0001E93E,  0x00000000, 0, 0x00000000 } },
{ 0x0001E91D, { 0x0001E93F,  0x00000000, 0, 0x00000000 } },
{ 0x0001E91E, { 0x0001E940,  0x00000000, 0, 0x00000000 } },
{ 0x0001E91F, { 0x0001E941,  0x00000000, 0, 0x00000000 } },
{ 0x0001E920, { 0x0001E942,  0x00000000, 0, 0x00000000 } },
{ 0x0001E921, { 0x0001E943,  0x00000000, 0, 0x00000000 } },
{ 0x00010C96, { 0x00010CD6,  0x00000000, 0, 0x00000000 } },
{ 0x00010C80, { 0x00010CC0,  0x00000000, 0, 0x00000000 } },
{ 0x00002C82, { 0x00002C83,  0x00000000, 0, 0x00000000 } },
{ 0x00002C84, { 0x00002C85,  0x00000000, 0, 0x00000000 } },
{ 0x00002C86, { 0x00002C87,  0x00000000, 0, 0x00000000 } },
{ 0x00001C87, { 0x00000463,  0x00000000, 0, 0x00000000 } },
{ 0x00002C88, { 0x00002C89,  0x00000000, 0, 0x00000000 } },
{ 0x00002C8A, { 0x00002C8B,  0x00000000, 0, 0x00000000 } },
{ 0x00010C8C, { 0x00010CCC,  0x00000000, 0, 0x00000000 } },
{ 0x00010C8E, { 0x00010CCE,  0x00000000, 0, 0x00000000 } },
{ 0x00002C90, { 0x00002C91,  0x00000000, 0, 0x00000000 } },
{ 0x00002C92, { 0x00002C93,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB70, { 0x000013A0,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB71, { 0x000013A1,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB72, { 0x000013A2,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB73, { 0x000013A3,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB74, { 0x000013A4,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB75, { 0x000013A5,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB76, { 0x000013A6,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB77, { 0x000013A7,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB78, { 0x000013A8,  0x00000000, 0, 0x00000000 } },
{ 0x00002C94, { 0x00002C95,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB7A, { 0x000013AA,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB7B, { 0x000013AB,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB7C, { 0x000013AC,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB7D, { 0x000013AD,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB7E, { 0x000013AE,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB7F, { 0x000013AF,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB80, { 0x000013B0,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB81, { 0x000013B1,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB82, { 0x000013B2,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB83, { 0x000013B3,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB84, { 0x000013B4,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB85, { 0x000013B5,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB86, { 0x000013B6,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB87, { 0x000013B7,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB88, { 0x000013B8,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB89, { 0x000013B9,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB8A, { 0x000013BA,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB8B, { 0x000013BB,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB8C, { 0x000013BC,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB8D, { 0x000013BD,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB8E, { 0x000013BE,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB8F, { 0x000013BF,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB90, { 0x000013C0,  0x00000000, 0, 0x00000000 } },
{ 0x00010C98, { 0x00010CD8,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB92, { 0x000013C2,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB93, { 0x000013C3,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB94, { 0x000013C4,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB95, { 0x000013C5,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB96, { 0x000013C6,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB97, { 0x000013C7,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB98, { 0x000013C8,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB99, { 0x000013C9,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB9A, { 0x000013CA,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB9B, { 0x000013CB,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB9C, { 0x000013CC,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB9D, { 0x000013CD,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB9E, { 0x000013CE,  0x00000000, 0, 0x00000000 } },
{ 0x0000AB9F, { 0x000013CF,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA0, { 0x000013D0,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA1, { 0x000013D1,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA2, { 0x000013D2,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA3, { 0x000013D3,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA4, { 0x000013D4,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA5, { 0x000013D5,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA6, { 0x000013D6,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA7, { 0x000013D7,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA8, { 0x000013D8,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABA9, { 0x000013D9,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABAA, { 0x000013DA,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABAB, { 0x000013DB,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABAC, { 0x000013DC,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABAD, { 0x000013DD,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABAE, { 0x000013DE,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABAF, { 0x000013DF,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB0, { 0x000013E0,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB1, { 0x000013E1,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB2, { 0x000013E2,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB3, { 0x000013E3,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB4, { 0x000013E4,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB5, { 0x000013E5,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB6, { 0x000013E6,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB7, { 0x000013E7,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB8, { 0x000013E8,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABB9, { 0x000013E9,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABBA, { 0x000013EA,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABBB, { 0x000013EB,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABBC, { 0x000013EC,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABBD, { 0x000013ED,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABBE, { 0x000013EE,  0x00000000, 0, 0x00000000 } },
{ 0x0000ABBF, { 0x000013EF,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA0, { 0x00010CE0,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA2, { 0x00010CE2,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA4, { 0x00010CE4,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA6, { 0x00010CE6,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA8, { 0x00010CE8,  0x00000000, 0, 0x00000000 } },
{ 0x00010CAA, { 0x00010CEA,  0x00000000, 0, 0x00000000 } },
{ 0x00002C00, { 0x00002C30,  0x00000000, 0, 0x00000000 } },
{ 0x00002C01, { 0x00002C31,  0x00000000, 0, 0x00000000 } },
{ 0x00002C02, { 0x00002C32,  0x00000000, 0, 0x00000000 } },
{ 0x00002C03, { 0x00002C33,  0x00000000, 0, 0x00000000 } },
{ 0x00002C04, { 0x00002C34,  0x00000000, 0, 0x00000000 } },
{ 0x00002C05, { 0x00002C35,  0x00000000, 0, 0x00000000 } },
{ 0x00002C06, { 0x00002C36,  0x00000000, 0, 0x00000000 } },
{ 0x00002C07, { 0x00002C37,  0x00000000, 0, 0x00000000 } },
{ 0x00002C08, { 0x00002C38,  0x00000000, 0, 0x00000000 } },
{ 0x00002C09, { 0x00002C39,  0x00000000, 0, 0x00000000 } },
{ 0x00002C0A, { 0x00002C3A,  0x00000000, 0, 0x00000000 } },
{ 0x00002C0B, { 0x00002C3B,  0x00000000, 0, 0x00000000 } },
{ 0x00002C0C, { 0x00002C3C,  0x00000000, 0, 0x00000000 } },
{ 0x00002C0D, { 0x00002C3D,  0x00000000, 0, 0x00000000 } },
{ 0x00002C0E, { 0x00002C3E,  0x00000000, 0, 0x00000000 } },
{ 0x00002C0F, { 0x00002C3F,  0x00000000, 0, 0x00000000 } },
{ 0x00002C10, { 0x00002C40,  0x00000000, 0, 0x00000000 } },
{ 0x00002C11, { 0x00002C41,  0x00000000, 0, 0x00000000 } },
{ 0x00002C12, { 0x00002C42,  0x00000000, 0, 0x00000000 } },
{ 0x00002C13, { 0x00002C43,  0x00000000, 0, 0x00000000 } },
{ 0x00002C14, { 0x00002C44,  0x00000000, 0, 0x00000000 } },
{ 0x00002C15, { 0x00002C45,  0x00000000, 0, 0x00000000 } },
{ 0x00002C16, { 0x00002C46,  0x00000000, 0, 0x00000000 } },
{ 0x00002C17, { 0x00002C47,  0x00000000, 0, 0x00000000 } },
{ 0x00002C18, { 0x00002C48,  0x00000000, 0, 0x00000000 } },
{ 0x00002C19, { 0x00002C49,  0x00000000, 0, 0x00000000 } },
{ 0x00002C1A, { 0x00002C4A,  0x00000000, 0, 0x00000000 } },
{ 0x00002C1B, { 0x00002C4B,  0x00000000, 0, 0x00000000 } },
{ 0x00002C1C, { 0x00002C4C,  0x00000000, 0, 0x00000000 } },
{ 0x00002C1D, { 0x00002C4D,  0x00000000, 0, 0x00000000 } },
{ 0x00002C1E, { 0x00002C4E,  0x00000000, 0, 0x00000000 } },
{ 0x00002C1F, { 0x00002C4F,  0x00000000, 0, 0x00000000 } },
{ 0x00002C20, { 0x00002C50,  0x00000000, 0, 0x00000000 } },
{ 0x00002C21, { 0x00002C51,  0x00000000, 0, 0x00000000 } },
{ 0x00002C22, { 0x00002C52,  0x00000000, 0, 0x00000000 } },
{ 0x00002C23, { 0x00002C53,  0x00000000, 0, 0x00000000 } },
{ 0x00002C24, { 0x00002C54,  0x00000000, 0, 0x00000000 } },
{ 0x00002C25, { 0x00002C55,  0x00000000, 0, 0x00000000 } },
{ 0x00002C26, { 0x00002C56,  0x00000000, 0, 0x00000000 } },
{ 0x00002C27, { 0x00002C57,  0x00000000, 0, 0x00000000 } },
{ 0x00002C28, { 0x00002C58,  0x00000000, 0, 0x00000000 } },
{ 0x00002C29, { 0x00002C59,  0x00000000, 0, 0x00000000 } },
{ 0x00002C2A, { 0x00002C5A,  0x00000000, 0, 0x00000000 } },
{ 0x00002C2B, { 0x00002C5B,  0x00000000, 0, 0x00000000 } },
{ 0x00002C2C, { 0x00002C5C,  0x00000000, 0, 0x00000000 } },
{ 0x00002C2D, { 0x00002C5D,  0x00000000, 0, 0x00000000 } },
{ 0x00002C2E, { 0x00002C5E,  0x00000000, 0, 0x00000000 } },
{ 0x00002C60, { 0x00002C61,  0x00000000, 0, 0x00000000 } },
{ 0x00002C62, { 0x0000026B,  0x00000000, 0, 0x00000000 } },
{ 0x00002C63, { 0x00001D7D,  0x00000000, 0, 0x00000000 } },
{ 0x00002C64, { 0x0000027D,  0x00000000, 0, 0x00000000 } },
{ 0x00002C67, { 0x00002C68,  0x00000000, 0, 0x00000000 } },
{ 0x00002C69, { 0x00002C6A,  0x00000000, 0, 0x00000000 } },
{ 0x00002C6B, { 0x00002C6C,  0x00000000, 0, 0x00000000 } },
{ 0x00002C6D, { 0x00000251,  0x00000000, 0, 0x00000000 } },
{ 0x00002C6E, { 0x00000271,  0x00000000, 0, 0x00000000 } },
{ 0x00002C6F, { 0x00000250,  0x00000000, 0, 0x00000000 } },
{ 0x00002C70, { 0x00000252,  0x00000000, 0, 0x00000000 } },
{ 0x00002C72, { 0x00002C73,  0x00000000, 0, 0x00000000 } },
{ 0x00002C75, { 0x00002C76,  0x00000000, 0, 0x00000000 } },
{ 0x00002C7E, { 0x0000023F,  0x00000000, 0, 0x00000000 } },
{ 0x00002C7F, { 0x00000240,  0x00000000, 0, 0x00000000 } },
{ 0x00002C80, { 0x00002C81,  0x00000000, 0, 0x00000000 } },
{ 0x00010C81, { 0x00010CC1,  0x00000000, 0, 0x00000000 } },
{ 0x00010C82, { 0x00010CC2,  0x00000000, 0, 0x00000000 } },
{ 0x00010C83, { 0x00010CC3,  0x00000000, 0, 0x00000000 } },
{ 0x00010C84, { 0x00010CC4,  0x00000000, 0, 0x00000000 } },
{ 0x00010C85, { 0x00010CC5,  0x00000000, 0, 0x00000000 } },
{ 0x00010C86, { 0x00010CC6,  0x00000000, 0, 0x00000000 } },
{ 0x00010C87, { 0x00010CC7,  0x00000000, 0, 0x00000000 } },
{ 0x00010C88, { 0x00010CC8,  0x00000000, 0, 0x00000000 } },
{ 0x00010C89, { 0x00010CC9,  0x00000000, 0, 0x00000000 } },
{ 0x00010C8A, { 0x00010CCA,  0x00000000, 0, 0x00000000 } },
{ 0x00010C8B, { 0x00010CCB,  0x00000000, 0, 0x00000000 } },
{ 0x00002C8C, { 0x00002C8D,  0x00000000, 0, 0x00000000 } },
{ 0x00010C8D, { 0x00010CCD,  0x00000000, 0, 0x00000000 } },
{ 0x00002C8E, { 0x00002C8F,  0x00000000, 0, 0x00000000 } },
{ 0x00010C8F, { 0x00010CCF,  0x00000000, 0, 0x00000000 } },
{ 0x00010C90, { 0x00010CD0,  0x00000000, 0, 0x00000000 } },
{ 0x00010C91, { 0x00010CD1,  0x00000000, 0, 0x00000000 } },
{ 0x00010C92, { 0x00010CD2,  0x00000000, 0, 0x00000000 } },
{ 0x00010C93, { 0x00010CD3,  0x00000000, 0, 0x00000000 } },
{ 0x00010C94, { 0x00010CD4,  0x00000000, 0, 0x00000000 } },
{ 0x00010C95, { 0x00010CD5,  0x00000000, 0, 0x00000000 } },
{ 0x00002C96, { 0x00002C97,  0x00000000, 0, 0x00000000 } },
{ 0x00010C97, { 0x00010CD7,  0x00000000, 0, 0x00000000 } },
{ 0x00002C98, { 0x00002C99,  0x00000000, 0, 0x00000000 } },
{ 0x00010C99, { 0x00010CD9,  0x00000000, 0, 0x00000000 } },
{ 0x00002C9A, { 0x00002C9B,  0x00000000, 0, 0x00000000 } },
{ 0x00010C9B, { 0x00010CDB,  0x00000000, 0, 0x00000000 } },
{ 0x00002C9C, { 0x00002C9D,  0x00000000, 0, 0x00000000 } },
{ 0x00010C9D, { 0x00010CDD,  0x00000000, 0, 0x00000000 } },
{ 0x00002C9E, { 0x00002C9F,  0x00000000, 0, 0x00000000 } },
{ 0x00010C9F, { 0x00010CDF,  0x00000000, 0, 0x00000000 } },
{ 0x00002CA0, { 0x00002CA1,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA1, { 0x00010CE1,  0x00000000, 0, 0x00000000 } },
{ 0x00002CA2, { 0x00002CA3,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA3, { 0x00010CE3,  0x00000000, 0, 0x00000000 } },
{ 0x00002CA4, { 0x00002CA5,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA5, { 0x00010CE5,  0x00000000, 0, 0x00000000 } },
{ 0x00002CA6, { 0x00002CA7,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA7, { 0x00010CE7,  0x00000000, 0, 0x00000000 } },
{ 0x00002CA8, { 0x00002CA9,  0x00000000, 0, 0x00000000 } },
{ 0x00010CA9, { 0x00010CE9,  0x00000000, 0, 0x00000000 } },
{ 0x00002CAA, { 0x00002CAB,  0x00000000, 0, 0x00000000 } },
{ 0x00010CAB, { 0x00010CEB,  0x00000000, 0, 0x00000000 } },
{ 0x00002CAC, { 0x00002CAD,  0x00000000, 0, 0x00000000 } },
{ 0x00010CAD, { 0x00010CED,  0x00000000, 0, 0x00000000 } },
{ 0x00002CAE, { 0x00002CAF,  0x00000000, 0, 0x00000000 } },
{ 0x00010CAF, { 0x00010CEF,  0x00000000, 0, 0x00000000 } },
{ 0x00002CB0, { 0x00002CB1,  0x00000000, 0, 0x00000000 } },
{ 0x00010CB1, { 0x00010CF1,  0x00000000, 0, 0x00000000 } },
{ 0x00002CB2, { 0x00002CB3,  0x00000000, 0, 0x00000000 } },
{ 0x00002CB4, { 0x00002CB5,  0x00000000, 0, 0x00000000 } },
{ 0x00002CB6, { 0x00002CB7,  0x00000000, 0, 0x00000000 } },
{ 0x00002CB8, { 0x00002CB9,  0x00000000, 0, 0x00000000 } },
{ 0x00002CBA, { 0x00002CBB,  0x00000000, 0, 0x00000000 } },
{ 0x00002CBC, { 0x00002CBD,  0x00000000, 0, 0x00000000 } },
{ 0x00002CBE, { 0x00002CBF,  0x00000000, 0, 0x00000000 } },
{ 0x00002CC0, { 0x00002CC1,  0x00000000, 0, 0x00000000 } },
{ 0x00002CC2, { 0x00002CC3,  0x00000000, 0, 0x00000000 } },
{ 0x00002CC4, { 0x00002CC5,  0x00000000, 0, 0x00000000 } },
{ 0x00002CC6, { 0x00002CC7,  0x00000000, 0, 0x00000000 } },
{ 0x00002CC8, { 0x00002CC9,  0x00000000, 0, 0x00000000 } },
{ 0x00002CCA, { 0x00002CCB,  0x00000000, 0, 0x00000000 } },
{ 0x00002CCC, { 0x00002CCD,  0x00000000, 0, 0x00000000 } },
{ 0x00002CCE, { 0x00002CCF,  0x00000000, 0, 0x00000000 } },
{ 0x00002CD0, { 0x00002CD1,  0x00000000, 0, 0x00000000 } },
{ 0x00002CD2, { 0x00002CD3,  0x00000000, 0, 0x00000000 } },
{ 0x00002CD4, { 0x00002CD5,  0x00000000, 0, 0x00000000 } },
{ 0x00002CD6, { 0x00002CD7,  0x00000000, 0, 0x00000000 } },
{ 0x00002CD8, { 0x00002CD9,  0x00000000, 0, 0x00000000 } },
{ 0x00002CDA, { 0x00002CDB,  0x00000000, 0, 0x00000000 } },
{ 0x00002CDC, { 0x00002CDD,  0x00000000, 0, 0x00000000 } },
{ 0x00002CDE, { 0x00002CDF,  0x00000000, 0, 0x00000000 } },
{ 0x00002CE0, { 0x00002CE1,  0x00000000, 0, 0x00000000 } },
{ 0x00002CE2, { 0x00002CE3,  0x00000000, 0, 0x00000000 } },
{ 0x00002CEB, { 0x00002CEC,  0x00000000, 0, 0x00000000 } },
{ 0x00002CED, { 0x00002CEE,  0x00000000, 0, 0x00000000 } },
{ 0x00002CF2, { 0x00002CF3,  0x00000000, 0, 0x00000000 } },
{ 0x00016E40, { 0x00016E60,  0x00000000, 0, 0x00000000 } },
{ 0x00016E41, { 0x00016E61,  0x00000000, 0, 0x00000000 } },
{ 0x00016E42, { 0x00016E62,  0x00000000, 0, 0x00000000 } },
{ 0x00016E43, { 0x00016E63,  0x00000000, 0, 0x00000000 } },
{ 0x00016E44, { 0x00016E64,  0x00000000, 0, 0x00000000 } },
{ 0x00016E45, { 0x00016E65,  0x00000000, 0, 0x00000000 } },
{ 0x00016E46, { 0x00016E66,  0x00000000, 0, 0x00000000 } },
{ 0x00016E47, { 0x00016E67,  0x00000000, 0, 0x00000000 } },
{ 0x00016E48, { 0x00016E68,  0x00000000, 0, 0x00000000 } },
{ 0x00016E49, { 0x00016E69,  0x00000000, 0, 0x00000000 } },
{ 0x00016E4A, { 0x00016E6A,  0x00000000, 0, 0x00000000 } },
{ 0x00016E4B, { 0x00016E6B,  0x00000000, 0, 0x00000000 } },
{ 0x00016E4C, { 0x00016E6C,  0x00000000, 0, 0x00000000 } },
{ 0x00016E4D, { 0x00016E6D,  0x00000000, 0, 0x00000000 } },
{ 0x00016E4E, { 0x00016E6E,  0x00000000, 0, 0x00000000 } },
{ 0x00016E4F, { 0x00016E6F,  0x00000000, 0, 0x00000000 } },
{ 0x00016E50, { 0x00016E70,  0x00000000, 0, 0x00000000 } },
{ 0x00016E51, { 0x00016E71,  0x00000000, 0, 0x00000000 } },
{ 0x00016E52, { 0x00016E72,  0x00000000, 0, 0x00000000 } },
{ 0x00016E53, { 0x00016E73,  0x00000000, 0, 0x00000000 } },
{ 0x00016E54, { 0x00016E74,  0x00000000, 0, 0x00000000 } },
{ 0x00016E55, { 0x00016E75,  0x00000000, 0, 0x00000000 } },
{ 0x00016E56, { 0x00016E76,  0x00000000, 0, 0x00000000 } },
{ 0x00016E57, { 0x00016E77,  0x00000000, 0, 0x00000000 } },
{ 0x00016E58, { 0x00016E78,  0x00000000, 0, 0x00000000 } },
{ 0x00016E59, { 0x00016E79,  0x00000000, 0, 0x00000000 } },
{ 0x00016E5A, { 0x00016E7A,  0x00000000, 0, 0x00000000 } },
{ 0x00016E5B, { 0x00016E7B,  0x00000000, 0, 0x00000000 } },
{ 0x00016E5C, { 0x00016E7C,  0x00000000, 0, 0x00000000 } },
{ 0x00016E5D, { 0x00016E7D,  0x00000000, 0, 0x00000000 } },
{ 0x00016E5E, { 0x00016E7E,  0x00000000, 0, 0x00000000 } },
{ 0x00016E5F, { 0x00016E7F,  0x00000000, 0, 0x00000000 } },
{ 0x000004BE, { 0x000004BF,  0x00000000, 0, 0x00000000 } },
{ 0x000024B6, { 0x000024D0,  0x00000000, 0, 0x00000000 } },
{ 0x000004C0, { 0x000004CF,  0x00000000, 0, 0x00000000 } },
{ 0x000024B8, { 0x000024D2,  0x00000000, 0, 0x00000000 } },
{ 0x000004C1, { 0x000004C2,  0x00000000, 0, 0x00000000 } },
{ 0x000024BA, { 0x000024D4,  0x00000000, 0, 0x00000000 } },
{ 0x000004C3, { 0x000004C4,  0x00000000, 0, 0x00000000 } },
{ 0x000004C5, { 0x000004C6,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF2E, { 0x0000FF4E,  0x00000000, 0, 0x00000000 } },
{ 0x000024C0, { 0x000024DA,  0x00000000, 0, 0x00000000 } },
{ 0x000004C9, { 0x000004CA,  0x00000000, 0, 0x00000000 } },
{ 0x000004CB, { 0x000004CC,  0x00000000, 0, 0x00000000 } },
{ 0x000024C4, { 0x000024DE,  0x00000000, 0, 0x00000000 } },
{ 0x000024C5, { 0x000024DF,  0x00000000, 0, 0x00000000 } },
{ 0x000024C6, { 0x000024E0,  0x00000000, 0, 0x00000000 } },
{ 0x000010A0, { 0x00002D00,  0x00000000, 0, 0x00000000 } },
{ 0x000010A1, { 0x00002D01,  0x00000000, 0, 0x00000000 } },
{ 0x000010A2, { 0x00002D02,  0x00000000, 0, 0x00000000 } },
{ 0x000010A3, { 0x00002D03,  0x00000000, 0, 0x00000000 } },
{ 0x000010A4, { 0x00002D04,  0x00000000, 0, 0x00000000 } },
{ 0x000010A5, { 0x00002D05,  0x00000000, 0, 0x00000000 } },
{ 0x000010A6, { 0x00002D06,  0x00000000, 0, 0x00000000 } },
{ 0x000010A7, { 0x00002D07,  0x00000000, 0, 0x00000000 } },
{ 0x000010A8, { 0x00002D08,  0x00000000, 0, 0x00000000 } },
{ 0x000010A9, { 0x00002D09,  0x00000000, 0, 0x00000000 } },
{ 0x000010AA, { 0x00002D0A,  0x00000000, 0, 0x00000000 } },
{ 0x000010AB, { 0x00002D0B,  0x00000000, 0, 0x00000000 } },
{ 0x000010AC, { 0x00002D0C,  0x00000000, 0, 0x00000000 } },
{ 0x000010AD, { 0x00002D0D,  0x00000000, 0, 0x00000000 } },
{ 0x000010AE, { 0x00002D0E,  0x00000000, 0, 0x00000000 } },
{ 0x000010AF, { 0x00002D0F,  0x00000000, 0, 0x00000000 } },
{ 0x000010B0, { 0x00002D10,  0x00000000, 0, 0x00000000 } },
{ 0x000010B1, { 0x00002D11,  0x00000000, 0, 0x00000000 } },
{ 0x000010B2, { 0x00002D12,  0x00000000, 0, 0x00000000 } },
{ 0x000010B3, { 0x00002D13,  0x00000000, 0, 0x00000000 } },
{ 0x000010B4, { 0x00002D14,  0x00000000, 0, 0x00000000 } },
{ 0x000010B5, { 0x00002D15,  0x00000000, 0, 0x00000000 } },
{ 0x000010B6, { 0x00002D16,  0x00000000, 0, 0x00000000 } },
{ 0x000010B7, { 0x00002D17,  0x00000000, 0, 0x00000000 } },
{ 0x000010B8, { 0x00002D18,  0x00000000, 0, 0x00000000 } },
{ 0x000010B9, { 0x00002D19,  0x00000000, 0, 0x00000000 } },
{ 0x000010BA, { 0x00002D1A,  0x00000000, 0, 0x00000000 } },
{ 0x000010BB, { 0x00002D1B,  0x00000000, 0, 0x00000000 } },
{ 0x000010BC, { 0x00002D1C,  0x00000000, 0, 0x00000000 } },
{ 0x000010BD, { 0x00002D1D,  0x00000000, 0, 0x00000000 } },
{ 0x000010BE, { 0x00002D1E,  0x00000000, 0, 0x00000000 } },
{ 0x000010BF, { 0x00002D1F,  0x00000000, 0, 0x00000000 } },
{ 0x000010C0, { 0x00002D20,  0x00000000, 0, 0x00000000 } },
{ 0x000010C1, { 0x00002D21,  0x00000000, 0, 0x00000000 } },
{ 0x000010C2, { 0x00002D22,  0x00000000, 0, 0x00000000 } },
{ 0x000010C3, { 0x00002D23,  0x00000000, 0, 0x00000000 } },
{ 0x000010C4, { 0x00002D24,  0x00000000, 0, 0x00000000 } },
{ 0x000010C5, { 0x00002D25,  0x00000000, 0, 0x00000000 } },
{ 0x000010C7, { 0x00002D27,  0x00000000, 0, 0x00000000 } },
{ 0x000010CD, { 0x00002D2D,  0x00000000, 0, 0x00000000 } },
{ 0x000024CD, { 0x000024E7,  0x00000000, 0, 0x00000000 } },
{ 0x000118A3, { 0x000118C3,  0x00000000, 0, 0x00000000 } },
{ 0x000118A9, { 0x000118C9,  0x00000000, 0, 0x00000000 } },
{ 0x000013F8, { 0x000013F0,  0x00000000, 0, 0x00000000 } },
{ 0x000013F9, { 0x000013F1,  0x00000000, 0, 0x00000000 } },
{ 0x000013FA, { 0x000013F2,  0x00000000, 0, 0x00000000 } },
{ 0x000013FB, { 0x000013F3,  0x00000000, 0, 0x00000000 } },
{ 0x000013FC, { 0x000013F4,  0x00000000, 0, 0x00000000 } },
{ 0x000013FD, { 0x000013F5,  0x00000000, 0, 0x00000000 } },
{ 0x000118AF, { 0x000118CF,  0x00000000, 0, 0x00000000 } },
{ 0x000118B5, { 0x000118D5,  0x00000000, 0, 0x00000000 } },
{ 0x000118BB, { 0x000118DB,  0x00000000, 0, 0x00000000 } },
{ 0x000104BC, { 0x000104E4,  0x00000000, 0, 0x00000000 } },
{ 0x000104C3, { 0x000104EB,  0x00000000, 0, 0x00000000 } },
{ 0x000104C7, { 0x000104EF,  0x00000000, 0, 0x00000000 } },
{ 0x000104C8, { 0x000104F0,  0x00000000, 0, 0x00000000 } },
{ 0x000104C9, { 0x000104F1,  0x00000000, 0, 0x00000000 } },
{ 0x000104CA, { 0x000104F2,  0x00000000, 0, 0x00000000 } },
{ 0x00000400, { 0x00000450,  0x00000000, 0, 0x00000000 } },
{ 0x00010401, { 0x00010429,  0x00000000, 0, 0x00000000 } },
{ 0x00000402, { 0x00000452,  0x00000000, 0, 0x00000000 } },
{ 0x00000403, { 0x00000453,  0x00000000, 0, 0x00000000 } },
{ 0x00000404, { 0x00000454,  0x00000000, 0, 0x00000000 } },
{ 0x00000405, { 0x00000455,  0x00000000, 0, 0x00000000 } },
{ 0x00000406, { 0x00000456,  0x00000000, 0, 0x00000000 } },
{ 0x00000407, { 0x00000457,  0x00000000, 0, 0x00000000 } },
{ 0x00010408, { 0x00010430,  0x00000000, 0, 0x00000000 } },
{ 0x00000409, { 0x00000459,  0x00000000, 0, 0x00000000 } },
{ 0x0000040A, { 0x0000045A,  0x00000000, 0, 0x00000000 } },
{ 0x0000040B, { 0x0000045B,  0x00000000, 0, 0x00000000 } },
{ 0x0000040C, { 0x0000045C,  0x00000000, 0, 0x00000000 } },
{ 0x0001040D, { 0x00010435,  0x00000000, 0, 0x00000000 } },
{ 0x0001040E, { 0x00010436,  0x00000000, 0, 0x00000000 } },
{ 0x0000040F, { 0x0000045F,  0x00000000, 0, 0x00000000 } },
{ 0x00000410, { 0x00000430,  0x00000000, 0, 0x00000000 } },
{ 0x00010411, { 0x00010439,  0x00000000, 0, 0x00000000 } },
{ 0x00000412, { 0x00000432,  0x00000000, 0, 0x00000000 } },
{ 0x00000413, { 0x00000433,  0x00000000, 0, 0x00000000 } },
{ 0x00000414, { 0x00000434,  0x00000000, 0, 0x00000000 } },
{ 0x00000415, { 0x00000435,  0x00000000, 0, 0x00000000 } },
{ 0x00010416, { 0x0001043E,  0x00000000, 0, 0x00000000 } },
{ 0x00000417, { 0x00000437,  0x00000000, 0, 0x00000000 } },
{ 0x00010418, { 0x00010440,  0x00000000, 0, 0x00000000 } },
{ 0x00010419, { 0x00010441,  0x00000000, 0, 0x00000000 } },
{ 0x0000041A, { 0x0000043A,  0x00000000, 0, 0x00000000 } },
{ 0x000118A0, { 0x000118C0,  0x00000000, 0, 0x00000000 } },
{ 0x000118A1, { 0x000118C1,  0x00000000, 0, 0x00000000 } },
{ 0x000118A2, { 0x000118C2,  0x00000000, 0, 0x00000000 } },
{ 0x0001041B, { 0x00010443,  0x00000000, 0, 0x00000000 } },
{ 0x000118A4, { 0x000118C4,  0x00000000, 0, 0x00000000 } },
{ 0x000118A5, { 0x000118C5,  0x00000000, 0, 0x00000000 } },
{ 0x000118A6, { 0x000118C6,  0x00000000, 0, 0x00000000 } },
{ 0x000118A7, { 0x000118C7,  0x00000000, 0, 0x00000000 } },
{ 0x000118A8, { 0x000118C8,  0x00000000, 0, 0x00000000 } },
{ 0x0000041C, { 0x0000043C,  0x00000000, 0, 0x00000000 } },
{ 0x000118AA, { 0x000118CA,  0x00000000, 0, 0x00000000 } },
{ 0x000118AB, { 0x000118CB,  0x00000000, 0, 0x00000000 } },
{ 0x000118AC, { 0x000118CC,  0x00000000, 0, 0x00000000 } },
{ 0x000118AD, { 0x000118CD,  0x00000000, 0, 0x00000000 } },
{ 0x000118AE, { 0x000118CE,  0x00000000, 0, 0x00000000 } },
{ 0x0000041D, { 0x0000043D,  0x00000000, 0, 0x00000000 } },
{ 0x000118B0, { 0x000118D0,  0x00000000, 0, 0x00000000 } },
{ 0x000118B1, { 0x000118D1,  0x00000000, 0, 0x00000000 } },
{ 0x000118B2, { 0x000118D2,  0x00000000, 0, 0x00000000 } },
{ 0x000118B3, { 0x000118D3,  0x00000000, 0, 0x00000000 } },
{ 0x000118B4, { 0x000118D4,  0x00000000, 0, 0x00000000 } },
{ 0x0000041E, { 0x0000043E,  0x00000000, 0, 0x00000000 } },
{ 0x000118B6, { 0x000118D6,  0x00000000, 0, 0x00000000 } },
{ 0x000118B7, { 0x000118D7,  0x00000000, 0, 0x00000000 } },
{ 0x000118B8, { 0x000118D8,  0x00000000, 0, 0x00000000 } },
{ 0x000118B9, { 0x000118D9,  0x00000000, 0, 0x00000000 } },
{ 0x000118BA, { 0x000118DA,  0x00000000, 0, 0x00000000 } },
{ 0x0000041F, { 0x0000043F,  0x00000000, 0, 0x00000000 } },
{ 0x000118BC, { 0x000118DC,  0x00000000, 0, 0x00000000 } },
{ 0x000118BD, { 0x000118DD,  0x00000000, 0, 0x00000000 } },
{ 0x000118BE, { 0x000118DE,  0x00000000, 0, 0x00000000 } },
{ 0x000118BF, { 0x000118DF,  0x00000000, 0, 0x00000000 } },
{ 0x00000420, { 0x00000440,  0x00000000, 0, 0x00000000 } },
{ 0x00010421, { 0x00010449,  0x00000000, 0, 0x00000000 } },
{ 0x00010422, { 0x0001044A,  0x00000000, 0, 0x00000000 } },
{ 0x00010423, { 0x0001044B,  0x00000000, 0, 0x00000000 } },
{ 0x00010424, { 0x0001044C,  0x00000000, 0, 0x00000000 } },
{ 0x00010425, { 0x0001044D,  0x00000000, 0, 0x00000000 } },
{ 0x00010426, { 0x0001044E,  0x00000000, 0, 0x00000000 } },
{ 0x00010427, { 0x0001044F,  0x00000000, 0, 0x00000000 } },
{ 0x00010CAE, { 0x00010CEE,  0x00000000, 0, 0x00000000 } },
{ 0x00010CB2, { 0x00010CF2,  0x00000000, 0, 0x00000000 } },
{ 0x00001F28, { 0x00001F20,  0x00000000, 0, 0x00000000 } },
{ 0x00001F29, { 0x00001F21,  0x00000000, 0, 0x00000000 } },
{ 0x00001F2A, { 0x00001F22,  0x00000000, 0, 0x00000000 } },
{ 0x0000FB00, { 0x00000000,  0x00000000, U"\U00000066\U00000066", 0x00000000 } },
{ 0x0000FB01, { 0x00000000,  0x00000000, U"\U00000066\U00000069", 0x00000000 } },
{ 0x0000FB02, { 0x00000000,  0x00000000, U"\U00000066\U0000006C", 0x00000000 } },
{ 0x0000FB03, { 0x00000000,  0x00000000, U"\U00000066\U00000066\U00000069", 0x00000000 } },
{ 0x0000FB04, { 0x00000000,  0x00000000, U"\U00000066\U00000066\U0000006C", 0x00000000 } },
{ 0x0000FB05, { 0x00000000,  0x00000000, U"\U00000073\U00000074", 0x00000000 } },
{ 0x0000FB06, { 0x00000000,  0x00000000, U"\U00000073\U00000074", 0x00000000 } },
{ 0x00001F2C, { 0x00001F24,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF2D, { 0x0000FF4D,  0x00000000, 0, 0x00000000 } },
{ 0x0000FB13, { 0x00000000,  0x00000000, U"\U00000574\U00000576", 0x00000000 } },
{ 0x0000FB14, { 0x00000000,  0x00000000, U"\U00000574\U00000565", 0x00000000 } },
{ 0x0000FB15, { 0x00000000,  0x00000000, U"\U00000574\U0000056B", 0x00000000 } },
{ 0x0000FB16, { 0x00000000,  0x00000000, U"\U0000057E\U00000576", 0x00000000 } },
{ 0x0000FB17, { 0x00000000,  0x00000000, U"\U00000574\U0000056D", 0x00000000 } },
{ 0x00001F2F, { 0x00001F27,  0x00000000, 0, 0x00000000 } },
{ 0x00001F38, { 0x00001F30,  0x00000000, 0, 0x00000000 } },
{ 0x00001F39, { 0x00001F31,  0x00000000, 0, 0x00000000 } },
{ 0x00001F3A, { 0x00001F32,  0x00000000, 0, 0x00000000 } },
{ 0x000004B0, { 0x000004B1,  0x00000000, 0, 0x00000000 } },
{ 0x000104B2, { 0x000104DA,  0x00000000, 0, 0x00000000 } },
{ 0x000004B4, { 0x000004B5,  0x00000000, 0, 0x00000000 } },
{ 0x000004B6, { 0x000004B7,  0x00000000, 0, 0x00000000 } },
{ 0x000104B7, { 0x000104DF,  0x00000000, 0, 0x00000000 } },
{ 0x000004B8, { 0x000004B9,  0x00000000, 0, 0x00000000 } },
{ 0x000024B9, { 0x000024D3,  0x00000000, 0, 0x00000000 } },
{ 0x000004BA, { 0x000004BB,  0x00000000, 0, 0x00000000 } },
{ 0x000024BB, { 0x000024D5,  0x00000000, 0, 0x00000000 } },
{ 0x000004BC, { 0x000004BD,  0x00000000, 0, 0x00000000 } },
{ 0x000024BD, { 0x000024D7,  0x00000000, 0, 0x00000000 } },
{ 0x000104BE, { 0x000104E6,  0x00000000, 0, 0x00000000 } },
{ 0x000024BF, { 0x000024D9,  0x00000000, 0, 0x00000000 } },
{ 0x00001C80, { 0x00000432,  0x00000000, 0, 0x00000000 } },
{ 0x00001C81, { 0x00000434,  0x00000000, 0, 0x00000000 } },
{ 0x00001C82, { 0x0000043E,  0x00000000, 0, 0x00000000 } },
{ 0x00001C83, { 0x00000441,  0x00000000, 0, 0x00000000 } },
{ 0x00001C84, { 0x00000442,  0x00000000, 0, 0x00000000 } },
{ 0x00001C85, { 0x00000442,  0x00000000, 0, 0x00000000 } },
{ 0x00001C86, { 0x0000044A,  0x00000000, 0, 0x00000000 } },
{ 0x000024C1, { 0x000024DB,  0x00000000, 0, 0x00000000 } },
{ 0x00001C88, { 0x0000A64B,  0x00000000, 0, 0x00000000 } },
{ 0x000024C2, { 0x000024DC,  0x00000000, 0, 0x00000000 } },
{ 0x00001C90, { 0x000010D0,  0x00000000, 0, 0x00000000 } },
{ 0x00001C91, { 0x000010D1,  0x00000000, 0, 0x00000000 } },
{ 0x00001C92, { 0x000010D2,  0x00000000, 0, 0x00000000 } },
{ 0x00001C93, { 0x000010D3,  0x00000000, 0, 0x00000000 } },
{ 0x00001C94, { 0x000010D4,  0x00000000, 0, 0x00000000 } },
{ 0x00001C95, { 0x000010D5,  0x00000000, 0, 0x00000000 } },
{ 0x00001C96, { 0x000010D6,  0x00000000, 0, 0x00000000 } },
{ 0x00001C97, { 0x000010D7,  0x00000000, 0, 0x00000000 } },
{ 0x00001C98, { 0x000010D8,  0x00000000, 0, 0x00000000 } },
{ 0x00001C99, { 0x000010D9,  0x00000000, 0, 0x00000000 } },
{ 0x00001C9A, { 0x000010DA,  0x00000000, 0, 0x00000000 } },
{ 0x00001C9B, { 0x000010DB,  0x00000000, 0, 0x00000000 } },
{ 0x00001C9C, { 0x000010DC,  0x00000000, 0, 0x00000000 } },
{ 0x00001C9D, { 0x000010DD,  0x00000000, 0, 0x00000000 } },
{ 0x00001C9E, { 0x000010DE,  0x00000000, 0, 0x00000000 } },
{ 0x00001C9F, { 0x000010DF,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA0, { 0x000010E0,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA1, { 0x000010E1,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA2, { 0x000010E2,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA3, { 0x000010E3,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA4, { 0x000010E4,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA5, { 0x000010E5,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA6, { 0x000010E6,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA7, { 0x000010E7,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA8, { 0x000010E8,  0x00000000, 0, 0x00000000 } },
{ 0x00001CA9, { 0x000010E9,  0x00000000, 0, 0x00000000 } },
{ 0x00001CAA, { 0x000010EA,  0x00000000, 0, 0x00000000 } },
{ 0x000004C7, { 0x000004C8,  0x00000000, 0, 0x00000000 } },
{ 0x00001CAC, { 0x000010EC,  0x00000000, 0, 0x00000000 } },
{ 0x00001CAD, { 0x000010ED,  0x00000000, 0, 0x00000000 } },
{ 0x00001CAE, { 0x000010EE,  0x00000000, 0, 0x00000000 } },
{ 0x00001CAF, { 0x000010EF,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB0, { 0x000010F0,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB1, { 0x000010F1,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB2, { 0x000010F2,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB3, { 0x000010F3,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB4, { 0x000010F4,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB5, { 0x000010F5,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB6, { 0x000010F6,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB7, { 0x000010F7,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB8, { 0x000010F8,  0x00000000, 0, 0x00000000 } },
{ 0x00001CB9, { 0x000010F9,  0x00000000, 0, 0x00000000 } },
{ 0x00001CBA, { 0x000010FA,  0x00000000, 0, 0x00000000 } },
{ 0x00001CBD, { 0x000010FD,  0x00000000, 0, 0x00000000 } },
{ 0x00001CBE, { 0x000010FE,  0x00000000, 0, 0x00000000 } },
{ 0x00001CBF, { 0x000010FF,  0x00000000, 0, 0x00000000 } },
{ 0x000024CB, { 0x000024E5,  0x00000000, 0, 0x00000000 } },
{ 0x000024CC, { 0x000024E6,  0x00000000, 0, 0x00000000 } },
{ 0x000004CD, { 0x000004CE,  0x00000000, 0, 0x00000000 } },
{ 0x000024CE, { 0x000024E8,  0x00000000, 0, 0x00000000 } },
{ 0x000024CF, { 0x000024E9,  0x00000000, 0, 0x00000000 } },
{ 0x000004D0, { 0x000004D1,  0x00000000, 0, 0x00000000 } },
{ 0x000004D2, { 0x000004D3,  0x00000000, 0, 0x00000000 } },
{ 0x00001CAB, { 0x000010EB,  0x00000000, 0, 0x00000000 } },
{ 0x00001E00, { 0x00001E01,  0x00000000, 0, 0x00000000 } },
{ 0x00001E02, { 0x00001E03,  0x00000000, 0, 0x00000000 } },
{ 0x00001E04, { 0x00001E05,  0x00000000, 0, 0x00000000 } },
{ 0x00001E06, { 0x00001E07,  0x00000000, 0, 0x00000000 } },
{ 0x00001E08, { 0x00001E09,  0x00000000, 0, 0x00000000 } },
{ 0x00001E0A, { 0x00001E0B,  0x00000000, 0, 0x00000000 } },
{ 0x00001E0C, { 0x00001E0D,  0x00000000, 0, 0x00000000 } },
{ 0x00001E0E, { 0x00001E0F,  0x00000000, 0, 0x00000000 } },
{ 0x00001E10, { 0x00001E11,  0x00000000, 0, 0x00000000 } },
{ 0x00001E12, { 0x00001E13,  0x00000000, 0, 0x00000000 } },
{ 0x00001E14, { 0x00001E15,  0x00000000, 0, 0x00000000 } },
{ 0x00001E16, { 0x00001E17,  0x00000000, 0, 0x00000000 } },
{ 0x00001E18, { 0x00001E19,  0x00000000, 0, 0x00000000 } },
{ 0x00001E1A, { 0x00001E1B,  0x00000000, 0, 0x00000000 } },
{ 0x00001E1C, { 0x00001E1D,  0x00000000, 0, 0x00000000 } },
{ 0x00001E1E, { 0x00001E1F,  0x00000000, 0, 0x00000000 } },
{ 0x00001E20, { 0x00001E21,  0x00000000, 0, 0x00000000 } },
{ 0x00001E22, { 0x00001E23,  0x00000000, 0, 0x00000000 } },
{ 0x00001E24, { 0x00001E25,  0x00000000, 0, 0x00000000 } },
{ 0x00001E26, { 0x00001E27,  0x00000000, 0, 0x00000000 } },
{ 0x00001E28, { 0x00001E29,  0x00000000, 0, 0x00000000 } },
{ 0x00001E2A, { 0x00001E2B,  0x00000000, 0, 0x00000000 } },
{ 0x00001E2C, { 0x00001E2D,  0x00000000, 0, 0x00000000 } },
{ 0x00001E2E, { 0x00001E2F,  0x00000000, 0, 0x00000000 } },
{ 0x00001E30, { 0x00001E31,  0x00000000, 0, 0x00000000 } },
{ 0x00001E32, { 0x00001E33,  0x00000000, 0, 0x00000000 } },
{ 0x00001E34, { 0x00001E35,  0x00000000, 0, 0x00000000 } },
{ 0x00001E36, { 0x00001E37,  0x00000000, 0, 0x00000000 } },
{ 0x00001E38, { 0x00001E39,  0x00000000, 0, 0x00000000 } },
{ 0x00001E3A, { 0x00001E3B,  0x00000000, 0, 0x00000000 } },
{ 0x00001E3C, { 0x00001E3D,  0x00000000, 0, 0x00000000 } },
{ 0x00001E3E, { 0x00001E3F,  0x00000000, 0, 0x00000000 } },
{ 0x00001E40, { 0x00001E41,  0x00000000, 0, 0x00000000 } },
{ 0x00001E42, { 0x00001E43,  0x00000000, 0, 0x00000000 } },
{ 0x00001E44, { 0x00001E45,  0x00000000, 0, 0x00000000 } },
{ 0x00001E46, { 0x00001E47,  0x00000000, 0, 0x00000000 } },
{ 0x00001E48, { 0x00001E49,  0x00000000, 0, 0x00000000 } },
{ 0x00001E4A, { 0x00001E4B,  0x00000000, 0, 0x00000000 } },
{ 0x00001E4C, { 0x00001E4D,  0x00000000, 0, 0x00000000 } },
{ 0x00001E4E, { 0x00001E4F,  0x00000000, 0, 0x00000000 } },
{ 0x00001E50, { 0x00001E51,  0x00000000, 0, 0x00000000 } },
{ 0x00001E52, { 0x00001E53,  0x00000000, 0, 0x00000000 } },
{ 0x00001E54, { 0x00001E55,  0x00000000, 0, 0x00000000 } },
{ 0x00001E56, { 0x00001E57,  0x00000000, 0, 0x00000000 } },
{ 0x00001E58, { 0x00001E59,  0x00000000, 0, 0x00000000 } },
{ 0x00001E5A, { 0x00001E5B,  0x00000000, 0, 0x00000000 } },
{ 0x00001E5C, { 0x00001E5D,  0x00000000, 0, 0x00000000 } },
{ 0x00001E5E, { 0x00001E5F,  0x00000000, 0, 0x00000000 } },
{ 0x00001E60, { 0x00001E61,  0x00000000, 0, 0x00000000 } },
{ 0x00001E62, { 0x00001E63,  0x00000000, 0, 0x00000000 } },
{ 0x00001E64, { 0x00001E65,  0x00000000, 0, 0x00000000 } },
{ 0x00001E66, { 0x00001E67,  0x00000000, 0, 0x00000000 } },
{ 0x00001E68, { 0x00001E69,  0x00000000, 0, 0x00000000 } },
{ 0x00001E6A, { 0x00001E6B,  0x00000000, 0, 0x00000000 } },
{ 0x00001E6C, { 0x00001E6D,  0x00000000, 0, 0x00000000 } },
{ 0x00001E6E, { 0x00001E6F,  0x00000000, 0, 0x00000000 } },
{ 0x00001E70, { 0x00001E71,  0x00000000, 0, 0x00000000 } },
{ 0x00001E72, { 0x00001E73,  0x00000000, 0, 0x00000000 } },
{ 0x00001E74, { 0x00001E75,  0x00000000, 0, 0x00000000 } },
{ 0x00001E76, { 0x00001E77,  0x00000000, 0, 0x00000000 } },
{ 0x00001E78, { 0x00001E79,  0x00000000, 0, 0x00000000 } },
{ 0x00001E7A, { 0x00001E7B,  0x00000000, 0, 0x00000000 } },
{ 0x00001E7C, { 0x00001E7D,  0x00000000, 0, 0x00000000 } },
{ 0x00001E7E, { 0x00001E7F,  0x00000000, 0, 0x00000000 } },
{ 0x00001E80, { 0x00001E81,  0x00000000, 0, 0x00000000 } },
{ 0x00001E82, { 0x00001E83,  0x00000000, 0, 0x00000000 } },
{ 0x00001E84, { 0x00001E85,  0x00000000, 0, 0x00000000 } },
{ 0x00001E86, { 0x00001E87,  0x00000000, 0, 0x00000000 } },
{ 0x00001E88, { 0x00001E89,  0x00000000, 0, 0x00000000 } },
{ 0x00001E8A, { 0x00001E8B,  0x00000000, 0, 0x00000000 } },
{ 0x00001E8C, { 0x00001E8D,  0x00000000, 0, 0x00000000 } },
{ 0x00001E8E, { 0x00001E8F,  0x00000000, 0, 0x00000000 } },
{ 0x00001E90, { 0x00001E91,  0x00000000, 0, 0x00000000 } },
{ 0x00001E92, { 0x00001E93,  0x00000000, 0, 0x00000000 } },
{ 0x00001E94, { 0x00001E95,  0x00000000, 0, 0x00000000 } },
{ 0x00001E96, { 0x00000000,  0x00000000, U"\U00000068\U00000331", 0x00000000 } },
{ 0x00001E97, { 0x00000000,  0x00000000, U"\U00000074\U00000308", 0x00000000 } },
{ 0x00001E98, { 0x00000000,  0x00000000, U"\U00000077\U0000030A", 0x00000000 } },
{ 0x00001E99, { 0x00000000,  0x00000000, U"\U00000079\U0000030A", 0x00000000 } },
{ 0x00001E9A, { 0x00000000,  0x00000000, U"\U00000061\U000002BE", 0x00000000 } },
{ 0x00001E9B, { 0x00001E61,  0x00000000, 0, 0x00000000 } },
{ 0x00001E9E, { 0x00000000,  0x000000DF, U"\U00000073\U00000073", 0x00000000 } },
{ 0x00001EA0, { 0x00001EA1,  0x00000000, 0, 0x00000000 } },
{ 0x00001EA2, { 0x00001EA3,  0x00000000, 0, 0x00000000 } },
{ 0x00001EA4, { 0x00001EA5,  0x00000000, 0, 0x00000000 } },
{ 0x00001EA6, { 0x00001EA7,  0x00000000, 0, 0x00000000 } },
{ 0x00001EA8, { 0x00001EA9,  0x00000000, 0, 0x00000000 } },
{ 0x00001EAA, { 0x00001EAB,  0x00000000, 0, 0x00000000 } },
{ 0x00001EAC, { 0x00001EAD,  0x00000000, 0, 0x00000000 } },
{ 0x00001EAE, { 0x00001EAF,  0x00000000, 0, 0x00000000 } },
{ 0x00001EB0, { 0x00001EB1,  0x00000000, 0, 0x00000000 } },
{ 0x00001EB2, { 0x00001EB3,  0x00000000, 0, 0x00000000 } },
{ 0x00001EB4, { 0x00001EB5,  0x00000000, 0, 0x00000000 } },
{ 0x00001EB6, { 0x00001EB7,  0x00000000, 0, 0x00000000 } },
{ 0x00001EB8, { 0x00001EB9,  0x00000000, 0, 0x00000000 } },
{ 0x00001EBA, { 0x00001EBB,  0x00000000, 0, 0x00000000 } },
{ 0x00001EBC, { 0x00001EBD,  0x00000000, 0, 0x00000000 } },
{ 0x00001EBE, { 0x00001EBF,  0x00000000, 0, 0x00000000 } },
{ 0x00001EC0, { 0x00001EC1,  0x00000000, 0, 0x00000000 } },
{ 0x00001EC2, { 0x00001EC3,  0x00000000, 0, 0x00000000 } },
{ 0x00001EC4, { 0x00001EC5,  0x00000000, 0, 0x00000000 } },
{ 0x00001EC6, { 0x00001EC7,  0x00000000, 0, 0x00000000 } },
{ 0x00001EC8, { 0x00001EC9,  0x00000000, 0, 0x00000000 } },
{ 0x00001ECA, { 0x00001ECB,  0x00000000, 0, 0x00000000 } },
{ 0x00001ECC, { 0x00001ECD,  0x00000000, 0, 0x00000000 } },
{ 0x00001ECE, { 0x00001ECF,  0x00000000, 0, 0x00000000 } },
{ 0x00001ED0, { 0x00001ED1,  0x00000000, 0, 0x00000000 } },
{ 0x00001ED2, { 0x00001ED3,  0x00000000, 0, 0x00000000 } },
{ 0x00001ED4, { 0x00001ED5,  0x00000000, 0, 0x00000000 } },
{ 0x00001ED6, { 0x00001ED7,  0x00000000, 0, 0x00000000 } },
{ 0x00001ED8, { 0x00001ED9,  0x00000000, 0, 0x00000000 } },
{ 0x00001EDA, { 0x00001EDB,  0x00000000, 0, 0x00000000 } },
{ 0x00001EDC, { 0x00001EDD,  0x00000000, 0, 0x00000000 } },
{ 0x00001EDE, { 0x00001EDF,  0x00000000, 0, 0x00000000 } },
{ 0x00001EE0, { 0x00001EE1,  0x00000000, 0, 0x00000000 } },
{ 0x00001EE2, { 0x00001EE3,  0x00000000, 0, 0x00000000 } },
{ 0x00001EE4, { 0x00001EE5,  0x00000000, 0, 0x00000000 } },
{ 0x00001EE6, { 0x00001EE7,  0x00000000, 0, 0x00000000 } },
{ 0x00001EE8, { 0x00001EE9,  0x00000000, 0, 0x00000000 } },
{ 0x00001EEA, { 0x00001EEB,  0x00000000, 0, 0x00000000 } },
{ 0x00001EEC, { 0x00001EED,  0x00000000, 0, 0x00000000 } },
{ 0x00001EEE, { 0x00001EEF,  0x00000000, 0, 0x00000000 } },
{ 0x00001EF0, { 0x00001EF1,  0x00000000, 0, 0x00000000 } },
{ 0x00001EF2, { 0x00001EF3,  0x00000000, 0, 0x00000000 } },
{ 0x00001EF4, { 0x00001EF5,  0x00000000, 0, 0x00000000 } },
{ 0x00001EF6, { 0x00001EF7,  0x00000000, 0, 0x00000000 } },
{ 0x00001EF8, { 0x00001EF9,  0x00000000, 0, 0x00000000 } },
{ 0x00001EFA, { 0x00001EFB,  0x00000000, 0, 0x00000000 } },
{ 0x00001EFC, { 0x00001EFD,  0x00000000, 0, 0x00000000 } },
{ 0x00001EFE, { 0x00001EFF,  0x00000000, 0, 0x00000000 } },
{ 0x00001F08, { 0x00001F00,  0x00000000, 0, 0x00000000 } },
{ 0x00001F09, { 0x00001F01,  0x00000000, 0, 0x00000000 } },
{ 0x00001F0A, { 0x00001F02,  0x00000000, 0, 0x00000000 } },
{ 0x00001F0B, { 0x00001F03,  0x00000000, 0, 0x00000000 } },
{ 0x00001F0C, { 0x00001F04,  0x00000000, 0, 0x00000000 } },
{ 0x00001F0D, { 0x00001F05,  0x00000000, 0, 0x00000000 } },
{ 0x00001F0E, { 0x00001F06,  0x00000000, 0, 0x00000000 } },
{ 0x00001F0F, { 0x00001F07,  0x00000000, 0, 0x00000000 } },
{ 0x00001F18, { 0x00001F10,  0x00000000, 0, 0x00000000 } },
{ 0x00001F19, { 0x00001F11,  0x00000000, 0, 0x00000000 } },
{ 0x00001F1A, { 0x00001F12,  0x00000000, 0, 0x00000000 } },
{ 0x00001F1B, { 0x00001F13,  0x00000000, 0, 0x00000000 } },
{ 0x00001F1C, { 0x00001F14,  0x00000000, 0, 0x00000000 } },
{ 0x00001F1D, { 0x00001F15,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF21, { 0x0000FF41,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF22, { 0x0000FF42,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF23, { 0x0000FF43,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF24, { 0x0000FF44,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF25, { 0x0000FF45,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF26, { 0x0000FF46,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF27, { 0x0000FF47,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF28, { 0x0000FF48,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF29, { 0x0000FF49,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF2A, { 0x0000FF4A,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF2B, { 0x0000FF4B,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF2C, { 0x0000FF4C,  0x00000000, 0, 0x00000000 } },
{ 0x00001F2D, { 0x00001F25,  0x00000000, 0, 0x00000000 } },
{ 0x00001F2E, { 0x00001F26,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF2F, { 0x0000FF4F,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF30, { 0x0000FF50,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF31, { 0x0000FF51,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF32, { 0x0000FF52,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF33, { 0x0000FF53,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF34, { 0x0000FF54,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF35, { 0x0000FF55,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF36, { 0x0000FF56,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF37, { 0x0000FF57,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF38, { 0x0000FF58,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF39, { 0x0000FF59,  0x00000000, 0, 0x00000000 } },
{ 0x0000FF3A, { 0x0000FF5A,  0x00000000, 0, 0x00000000 } },
{ 0x00001F3B, { 0x00001F33,  0x00000000, 0, 0x00000000 } },
{ 0x00001F3C, { 0x00001F34,  0x00000000, 0, 0x00000000 } },
{ 0x00001F3D, { 0x00001F35,  0x00000000, 0, 0x00000000 } },
{ 0x00001F3E, { 0x00001F36,  0x00000000, 0, 0x00000000 } },
{ 0x00001F3F, { 0x00001F37,  0x00000000, 0, 0x00000000 } },
{ 0x00001F48, { 0x00001F40,  0x00000000, 0, 0x00000000 } },
{ 0x00001F49, { 0x00001F41,  0x00000000, 0, 0x00000000 } },
{ 0x00001F4A, { 0x00001F42,  0x00000000, 0, 0x00000000 } },
{ 0x00001F4B, { 0x00001F43,  0x00000000, 0, 0x00000000 } },
{ 0x00001F4C, { 0x00001F44,  0x00000000, 0, 0x00000000 } },
{ 0x00001F4D, { 0x00001F45,  0x00000000, 0, 0x00000000 } },
{ 0x00001F50, { 0x00000000,  0x00000000, U"\U000003C5\U00000313", 0x00000000 } },
{ 0x00001F52, { 0x00000000,  0x00000000, U"\U000003C5\U00000313\U00000300", 0x00000000 } },
{ 0x00001F54, { 0x00000000,  0x00000000, U"\U000003C5\U00000313\U00000301", 0x00000000 } },
{ 0x00001F56, { 0x00000000,  0x00000000, U"\U000003C5\U00000313\U00000342", 0x00000000 } },
{ 0x00001F59, { 0x00001F51,  0x00000000, 0, 0x00000000 } },
{ 0x00001F5B, { 0x00001F53,  0x00000000, 0, 0x00000000 } },
{ 0x00001F5D, { 0x00001F55,  0x00000000, 0, 0x00000000 } },
{ 0x00001F5F, { 0x00001F57,  0x00000000, 0, 0x00000000 } },
{ 0x00001F68, { 0x00001F60,  0x00000000, 0, 0x00000000 } },
{ 0x00001F69, { 0x00001F61,  0x00000000, 0, 0x00000000 } },
{ 0x00001F6A, { 0x00001F62,  0x00000000, 0, 0x00000000 } },
{ 0x00001F6B, { 0x00001F63,  0x00000000, 0, 0x00000000 } },
{ 0x00001F6C, { 0x00001F64,  0x00000000, 0, 0x00000000 } },
{ 0x00001F6D, { 0x00001F65,  0x00000000, 0, 0x00000000 } },
{ 0x00001F6E, { 0x00001F66,  0x00000000, 0, 0x00000000 } },
{ 0x00001F6F, { 0x00001F67,  0x00000000, 0, 0x00000000 } },
{ 0x00001F80, { 0x00000000,  0x00000000, U"\U00001F00\U000003B9", 0x00000000 } },
{ 0x00001F81, { 0x00000000,  0x00000000, U"\U00001F01\U000003B9", 0x00000000 } },
{ 0x00001F82, { 0x00000000,  0x00000000, U"\U00001F02\U000003B9", 0x00000000 } },
{ 0x00001F83, { 0x00000000,  0x00000000, U"\U00001F03\U000003B9", 0x00000000 } },
{ 0x00001F84, { 0x00000000,  0x00000000, U"\U00001F04\U000003B9", 0x00000000 } },
{ 0x00001F85, { 0x00000000,  0x00000000, U"\U00001F05\U000003B9", 0x00000000 } },
{ 0x00001F86, { 0x00000000,  0x00000000, U"\U00001F06\U000003B9", 0x00000000 } },
{ 0x00001F87, { 0x00000000,  0x00000000, U"\U00001F07\U000003B9", 0x00000000 } },
{ 0x00001F88, { 0x00000000,  0x00001F80, U"\U00001F00\U000003B9", 0x00000000 } },
{ 0x00001F89, { 0x00000000,  0x00001F81, U"\U00001F01\U000003B9", 0x00000000 } },
{ 0x00001F8A, { 0x00000000,  0x00001F82, U"\U00001F02\U000003B9", 0x00000000 } },
{ 0x00001F8B, { 0x00000000,  0x00001F83, U"\U00001F03\U000003B9", 0x00000000 } },
{ 0x00001F8C, { 0x00000000,  0x00001F84, U"\U00001F04\U000003B9", 0x00000000 } },
{ 0x00001F8D, { 0x00000000,  0x00001F85, U"\U00001F05\U000003B9", 0x00000000 } },
{ 0x00001F8E, { 0x00000000,  0x00001F86, U"\U00001F06\U000003B9", 0x00000000 } },
{ 0x00001F8F, { 0x00000000,  0x00001F87, U"\U00001F07\U000003B9", 0x00000000 } },
{ 0x00001F90, { 0x00000000,  0x00000000, U"\U00001F20\U000003B9", 0x00000000 } },
{ 0x00001F91, { 0x00000000,  0x00000000, U"\U00001F21\U000003B9", 0x00000000 } },
{ 0x00001F92, { 0x00000000,  0x00000000, U"\U00001F22\U000003B9", 0x00000000 } },
{ 0x00001F93, { 0x00000000,  0x00000000, U"\U00001F23\U000003B9", 0x00000000 } },
{ 0x00001F94, { 0x00000000,  0x00000000, U"\U00001F24\U000003B9", 0x00000000 } },
{ 0x00001F95, { 0x00000000,  0x00000000, U"\U00001F25\U000003B9", 0x00000000 } },
{ 0x00001F96, { 0x00000000,  0x00000000, U"\U00001F26\U000003B9", 0x00000000 } },
{ 0x00001F97, { 0x00000000,  0x00000000, U"\U00001F27\U000003B9", 0x00000000 } },
{ 0x00001F98, { 0x00000000,  0x00001F90, U"\U00001F20\U000003B9", 0x00000000 } },
{ 0x00001F99, { 0x00000000,  0x00001F91, U"\U00001F21\U000003B9", 0x00000000 } },
{ 0x00001F9A, { 0x00000000,  0x00001F92, U"\U00001F22\U000003B9", 0x00000000 } },
{ 0x00001F9B, { 0x00000000,  0x00001F93, U"\U00001F23\U000003B9", 0x00000000 } },
{ 0x00001F9C, { 0x00000000,  0x00001F94, U"\U00001F24\U000003B9", 0x00000000 } },
{ 0x00001F9D, { 0x00000000,  0x00001F95, U"\U00001F25\U000003B9", 0x00000000 } },
{ 0x00001F9E, { 0x00000000,  0x00001F96, U"\U00001F26\U000003B9", 0x00000000 } },
{ 0x00001F9F, { 0x00000000,  0x00001F97, U"\U00001F27\U000003B9", 0x00000000 } },
{ 0x00001FA0, { 0x00000000,  0x00000000, U"\U00001F60\U000003B9", 0x00000000 } },
{ 0x00001FA1, { 0x00000000,  0x00000000, U"\U00001F61\U000003B9", 0x00000000 } },
{ 0x00001FA2, { 0x00000000,  0x00000000, U"\U00001F62\U000003B9", 0x00000000 } },
{ 0x00001FA3, { 0x00000000,  0x00000000, U"\U00001F63\U000003B9", 0x00000000 } },
{ 0x00001FA4, { 0x00000000,  0x00000000, U"\U00001F64\U000003B9", 0x00000000 } },
{ 0x00001FA5, { 0x00000000,  0x00000000, U"\U00001F65\U000003B9", 0x00000000 } },
{ 0x00001FA6, { 0x00000000,  0x00000000, U"\U00001F66\U000003B9", 0x00000000 } },
{ 0x00001FA7, { 0x00000000,  0x00000000, U"\U00001F67\U000003B9", 0x00000000 } },
{ 0x00001FA8, { 0x00000000,  0x00001FA0, U"\U00001F60\U000003B9", 0x00000000 } },
{ 0x00001FA9, { 0x00000000,  0x00001FA1, U"\U00001F61\U000003B9", 0x00000000 } },
{ 0x00001FAA, { 0x00000000,  0x00001FA2, U"\U00001F62\U000003B9", 0x00000000 } },
{ 0x00001FAB, { 0x00000000,  0x00001FA3, U"\U00001F63\U000003B9", 0x00000000 } },
{ 0x00001FAC, { 0x00000000,  0x00001FA4, U"\U00001F64\U000003B9", 0x00000000 } },
{ 0x00001FAD, { 0x00000000,  0x00001FA5, U"\U00001F65\U000003B9", 0x00000000 } },
{ 0x00001FAE, { 0x00000000,  0x00001FA6, U"\U00001F66\U000003B9", 0x00000000 } },
{ 0x00001FAF, { 0x00000000,  0x00001FA7, U"\U00001F67\U000003B9", 0x00000000 } },
{ 0x00001FB2, { 0x00000000,  0x00000000, U"\U00001F70\U000003B9", 0x00000000 } },
{ 0x00001FB3, { 0x00000000,  0x00000000, U"\U000003B1\U000003B9", 0x00000000 } },
{ 0x00001FB4, { 0x00000000,  0x00000000, U"\U000003AC\U000003B9", 0x00000000 } },
{ 0x00001FB6, { 0x00000000,  0x00000000, U"\U000003B1\U00000342", 0x00000000 } },
{ 0x00001FB7, { 0x00000000,  0x00000000, U"\U000003B1\U00000342\U000003B9", 0x00000000 } },
{ 0x00001FB8, { 0x00001FB0,  0x00000000, 0, 0x00000000 } },
{ 0x00001FB9, { 0x00001FB1,  0x00000000, 0, 0x00000000 } },
{ 0x00001FBA, { 0x00001F70,  0x00000000, 0, 0x00000000 } },
{ 0x00001FBB, { 0x00001F71,  0x00000000, 0, 0x00000000 } },
{ 0x00001FBC, { 0x00000000,  0x00001FB3, U"\U000003B1\U000003B9", 0x00000000 } },
{ 0x00001FBE, { 0x000003B9,  0x00000000, 0, 0x00000000 } },
{ 0x00001FC2, { 0x00000000,  0x00000000, U"\U00001F74\U000003B9", 0x00000000 } },
{ 0x00001FC3, { 0x00000000,  0x00000000, U"\U000003B7\U000003B9", 0x00000000 } },
{ 0x00001FC4, { 0x00000000,  0x00000000, U"\U000003AE\U000003B9", 0x00000000 } },
{ 0x00001FC6, { 0x00000000,  0x00000000, U"\U000003B7\U00000342", 0x00000000 } },
{ 0x00001FC7, { 0x00000000,  0x00000000, U"\U000003B7\U00000342\U000003B9", 0x00000000 } },
{ 0x00001FC8, { 0x00001F72,  0x00000000, 0, 0x00000000 } },
{ 0x00001FC9, { 0x00001F73,  0x00000000, 0, 0x00000000 } },
{ 0x00001FCA, { 0x00001F74,  0x00000000, 0, 0x00000000 } },
{ 0x00001FCB, { 0x00001F75,  0x00000000, 0, 0x00000000 } },
{ 0x00001FCC, { 0x00000000,  0x00001FC3, U"\U000003B7\U000003B9", 0x00000000 } },
{ 0x00001FD2, { 0x00000000,  0x00000000, U"\U000003B9\U00000308\U00000300", 0x00000000 } },
{ 0x00001FD3, { 0x00000000,  0x00000000, U"\U000003B9\U00000308\U00000301", 0x00000000 } },
{ 0x00001FD6, { 0x00000000,  0x00000000, U"\U000003B9\U00000342", 0x00000000 } },
{ 0x00001FD7, { 0x00000000,  0x00000000, U"\U000003B9\U00000308\U00000342", 0x00000000 } },
{ 0x00001FD8, { 0x00001FD0,  0x00000000, 0, 0x00000000 } },
{ 0x00001FD9, { 0x00001FD1,  0x00000000, 0, 0x00000000 } },
{ 0x00001FDA, { 0x00001F76,  0x00000000, 0, 0x00000000 } },
{ 0x00001FDB, { 0x00001F77,  0x00000000, 0, 0x00000000 } },
{ 0x00001FE2, { 0x00000000,  0x00000000, U"\U000003C5\U00000308\U00000300", 0x00000000 } },
{ 0x00001FE3, { 0x00000000,  0x00000000, U"\U000003C5\U00000308\U00000301", 0x00000000 } },
{ 0x00001FE4, { 0x00000000,  0x00000000, U"\U000003C1\U00000313", 0x00000000 } },
{ 0x00001FE6, { 0x00000000,  0x00000000, U"\U000003C5\U00000342", 0x00000000 } },
{ 0x00001FE7, { 0x00000000,  0x00000000, U"\U000003C5\U00000308\U00000342", 0x00000000 } },
{ 0x00001FE8, { 0x00001FE0,  0x00000000, 0, 0x00000000 } },
{ 0x00001FE9, { 0x00001FE1,  0x00000000, 0, 0x00000000 } },
{ 0x00001FEA, { 0x00001F7A,  0x00000000, 0, 0x00000000 } },
{ 0x00001FEB, { 0x00001F7B,  0x00000000, 0, 0x00000000 } },
{ 0x00001FEC, { 0x00001FE5,  0x00000000, 0, 0x00000000 } },
{ 0x00001FF2, { 0x00000000,  0x00000000, U"\U00001F7C\U000003B9", 0x00000000 } },
{ 0x00001FF3, { 0x00000000,  0x00000000, U"\U000003C9\U000003B9", 0x00000000 } },
{ 0x00001FF4, { 0x00000000,  0x00000000, U"\U000003CE\U000003B9", 0x00000000 } },
{ 0x00001FF6, { 0x00000000,  0x00000000, U"\U000003C9\U00000342", 0x00000000 } },
{ 0x00001FF7, { 0x00000000,  0x00000000, U"\U000003C9\U00000342\U000003B9", 0x00000000 } },
{ 0x00001FF8, { 0x00001F78,  0x00000000, 0, 0x00000000 } },
{ 0x00001FF9, { 0x00001F79,  0x00000000, 0, 0x00000000 } },
{ 0x00001FFA, { 0x00001F7C,  0x00000000, 0, 0x00000000 } },
{ 0x00001FFB, { 0x00001F7D,  0x00000000, 0, 0x00000000 } },
{ 0x00001FFC, { 0x00000000,  0x00001FF3, U"\U000003C9\U000003B9", 0x00000000 } },
};
namespace _block_properties {
using T = Block;
const auto D = Block::Unassigned;
static const size_t _block_size = 128;
static const Block _4[] = { T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::LatinExtendedB,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions, };
static const Block _5[] = { T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::IPAExtensions,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters,T::SpacingModifierLetters, };
static const Block _6[] = { T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::CombiningDiacriticalMarks,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic,T::GreekAndCoptic, };
static const Block _10[] = { T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::CyrillicSupplement,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian, };
static const Block _11[] = { T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Armenian,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew,T::Hebrew, };
static const Block _14[] = { T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::Syriac,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement,T::ArabicSupplement, };
static const Block _15[] = { T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::Thaana,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo,T::NKo, };
static const Block _16[] = { T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Samaritan,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::Mandaic,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,T::SyriacSupplement,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const Block _17[] = { D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA,T::ArabicExtendedA, };
static const Block _33[] = { T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Myanmar,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian,T::Georgian, };
static const Block _39[] = { T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::EthiopicSupplement,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee,T::Cherokee, };
static const Block _45[] = { T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Ogham,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic,T::Runic, };
static const Block _46[] = { T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Tagalog,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Hanunoo,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Buhid,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa,T::Tagbanwa, };
static const Block _49[] = { T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::Mongolian,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended,T::UnifiedCanadianAboriginalSyllabicsExtended, };
static const Block _50[] = { T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::Limbu,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe,T::TaiLe, };
static const Block _51[] = { T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::NewTaiLue,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols,T::KhmerSymbols, };
static const Block _52[] = { T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::Buginese,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham, };
static const Block _53[] = { T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::TaiTham,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended,T::CombiningDiacriticalMarksExtended, };
static const Block _55[] = { T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Sundanese,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak,T::Batak, };
static const Block _56[] = { T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::Lepcha,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki,T::OlChiki, };
static const Block _57[] = { T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::CyrillicExtendedC,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::GeorgianExtended,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::SundaneseSupplement,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions,T::VedicExtensions, };
static const Block _59[] = { T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::PhoneticExtensionsSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement,T::CombiningDiacriticalMarksSupplement, };
static const Block _64[] = { T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::GeneralPunctuation,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts, };
static const Block _65[] = { T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::SuperscriptsAndSubscripts,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CurrencySymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols,T::CombiningDiacriticalMarksForSymbols, };
static const Block _66[] = { T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::LetterlikeSymbols,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms, };
static const Block _67[] = { T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::NumberForms,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows,T::Arrows, };
static const Block _72[] = { T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::ControlPictures,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::OpticalCharacterRecognition,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics,T::EnclosedAlphanumerics, };
static const Block _75[] = { T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::BlockElements,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes,T::GeometricShapes, };
static const Block _79[] = { T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::Dingbats,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::MiscellaneousMathematicalSymbolsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA,T::SupplementalArrowsA, };
static const Block _88[] = { T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::Glagolitic,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC,T::LatinExtendedC, };
static const Block _90[] = { T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::GeorgianSupplement,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh,T::Tifinagh, };
static const Block _91[] = { T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::EthiopicExtended,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA,T::CyrillicExtendedA, };
static const Block _95[] = { T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,T::KangxiRadicals,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters,T::IdeographicDescriptionCharacters, };
static const Block _96[] = { T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::CJKSymbolsAndPunctuation,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana, };
static const Block _97[] = { T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Hiragana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana,T::Katakana, };
static const Block _98[] = { T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::Bopomofo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo, };
static const Block _99[] = { T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::HangulCompatibilityJamo,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::Kanbun,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::BopomofoExtended,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::CJKStrokes,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions,T::KatakanaPhoneticExtensions, };
static const Block _155[] = { T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::CJKUnifiedIdeographsExtensionA,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols,T::YijingHexagramSymbols, };
static const Block _329[] = { T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiSyllables,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::YiRadicals,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu,T::Lisu, };
static const Block _332[] = { T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::Vai,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB, };
static const Block _333[] = { T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::CyrillicExtendedB,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum,T::Bamum, };
static const Block _334[] = { T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::ModifierToneLetters,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD,T::LatinExtendedD, };
static const Block _336[] = { T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::SylotiNagri,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::CommonIndicNumberForms,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa,T::PhagsPa, };
static const Block _337[] = { T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::Saurashtra,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended,T::DevanagariExtended, };
static const Block _338[] = { T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::KayahLi,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::Rejang,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA,T::HangulJamoExtendedA, };
static const Block _339[] = { T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::Javanese,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB,T::MyanmarExtendedB, };
static const Block _340[] = { T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::Cham,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA,T::MyanmarExtendedA, };
static const Block _341[] = { T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::TaiViet,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions,T::MeeteiMayekExtensions, };
static const Block _342[] = { T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::EthiopicExtendedA,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::LatinExtendedE,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement, };
static const Block _343[] = { T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::CherokeeSupplement,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek,T::MeeteiMayek, };
static const Block _431[] = { T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulSyllables,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB,T::HangulJamoExtendedB, };
static const Block _502[] = { T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::AlphabeticPresentationForms,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA,T::ArabicPresentationFormsA, };
static const Block _508[] = { T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VariationSelectors,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::VerticalForms,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CombiningHalfMarks,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::CJKCompatibilityForms,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::SmallFormVariants,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB,T::ArabicPresentationFormsB, };
static const Block _511[] = { T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::HalfwidthAndFullwidthForms,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials,T::Specials, };
static const Block _514[] = { T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AegeanNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers, };
static const Block _515[] = { T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientGreekNumbers,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::AncientSymbols,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc,T::PhaistosDisc, };
static const Block _517[] = { T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Lycian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::Carian,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers,T::CopticEpactNumbers, };
static const Block _518[] = { T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::OldItalic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::Gothic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic,T::OldPermic, };
static const Block _519[] = { T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::Ugaritic,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,T::OldPersian,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const Block _520[] = { T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Deseret,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian,T::Shavian, };
static const Block _521[] = { T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osmanya,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage,T::Osage, };
static const Block _522[] = { T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::Elbasan,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,T::CaucasianAlbanian,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const Block _528[] = { T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::CypriotSyllabary,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::ImperialAramaic,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene,T::Palmyrene, };
static const Block _529[] = { T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,T::Nabataean,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran,T::Hatran, };
static const Block _530[] = { T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Phoenician,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,T::Lydian,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D, };
static const Block _531[] = { T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticHieroglyphs,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::MeroiticCursive,T::Meroitic