/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#ifndef CNIC_H
#define CNIC_H

#define CNIC_DRV_MODULE_VERSION	"1.0.26.0"
#define CNIC_DRIVER_NAME  "qcnic"

#define CNIC_HEAP_INITIAL_SIZE ( 16 * 1024 * 1024 )
#define CNIC_HEAP_MAXIMUM_SIZE ( 128 * 1024 * 1024 )

#define QFLE3_DEV_IDENTIFIER		"com.qlogic.cnicDev"
#define QFLE3_LEGACY_DEV_IDENTIFIER	"com.qlogic.cnicDevL"
#define QFLE3I_DEV_IDENTIFIER		"com.qlogic.iscsiDev"
#define QFLE3F_DEV_IDENTIFIER		"com.qlogic.fcoeDev"

extern unsigned long cnic_debug;
extern struct cnic_driver_info_t cnic_driver_info;

enum CNIC_DEBUG_MODULE {
	CNIC_MSG_DRV        = 0x0001,
	CNIC_MSG_INIT       = 0x0002,
	CNIC_MSG_SESS       = 0x0010,
	CNIC_MSG_NW         = 0x400000,
	CNIC_MSG_EVENT      = 0x800000,
	CNIC_MSG_INTR       = 0x1000000,
	CNIC_MSG_MEM        = 0x2000000,
	CNIC_MSG_CTL        = 0x4000000,
	CNIC_MSG_DEBUGFS    = 0x8000000,
};

enum CNIC_DEBUG_LEVEL {
	CNIC_DBG_ALERT,
	CNIC_DBG_WARN,
	CNIC_DBG_INFO,
};

/* Key-Value vmkmgmt defines */
#define MAX_KV_STRING_LEN 4096

/*
 * ESX-6.5: XXX:
 * For IPV6, to be backward compatible, vmware asked us to use
 * old definition of AF_INET6 = 0xa.
 * This will replace native kernel's VMK_SOCKET_AF_INET6 usage.
 */
#define AF_INET6	0xa /* IPV6 family */

#define CNIC_IS_NAME_EMPTY(name_p)   (*(vmk_NameToString(name_p)) == '\0')

#define DRV_ALERT_MODLOAD_FAIL(msg, status)                   \
	vmk_AlertMessage(msg ": %s\n", vmk_StatusToString(status)); \
	vmk_AlertMessage("Could not load " CNIC_DRIVER_NAME " module.\n");

#define DRV_DEBUG(urgency, lvl, msg, args...)                                 \
	do {                                                                      \
		if (vmk_LogGetCurrentLogLevel(cnic_driver_info.cnic_vmkLog) >= lvl) { \
			vmk_LogLevel(urgency, cnic_driver_info.cnic_vmkLog,               \
					lvl, msg"\n", ##args);                                    \
		}                                                                     \
	}while (0);

#define DRV_INFO(dbg_mask, msg, args...)                                    \
	do {                                                                    \
		if (cnic_debug & dbg_mask) {                                        \
			DRV_DEBUG(VMK_LOG_URGENCY_NORMAL, CNIC_DBG_INFO, "%s:%d: " msg, \
					__func__, __LINE__, ##args)                             \
		}                                                                   \
	} while (0);

#define DRV_WARN(msg, args...)                                       \
	DRV_DEBUG(VMK_LOG_URGENCY_WARNING, CNIC_DBG_WARN, "%s:%d: " msg, \
			__func__, __LINE__, ##args)

#define DRV_ERR(msg, args...)                                       \
	DRV_DEBUG(VMK_LOG_URGENCY_ALERT, CNIC_DBG_ALERT, "%s:%d: " msg, \
			__func__, __LINE__, ##args)

#define DUMP_CHAR(fmt, args...)     \
		vmk_LogNoLevel(VMK_LOG_URGENCY_ALERT, fmt, ##args)

#if (VMK_PAGE_SHIFT > 14)
#define BNX2_PAGE_BITS  14
#else
#define BNX2_PAGE_BITS  VMK_PAGE_SHIFT
#endif
#define BNX2_PAGE_SIZE  (1 << BNX2_PAGE_BITS)

#if (CNIC_ISCSI_OOO_SUPPORT)
#define FLG_SLEEPABLE	0
#define FLG_ATOMIC		1

#define HC_INDEX_FWD_TX_CQ_CONS	HC_SP_INDEX_ETH_ISCSI_CQ_CONS
#define HC_INDEX_OOO_RX_CQ_CONS	HC_SP_INDEX_ETH_ISCSI_RX_CQ_CONS
#endif

enum cnic_lock_rank_t{
	CNIC_LOW_LOCK_RANK = 1,
	CNIC_HIGH_LOCK_RANK,
};

struct cnic_driver_info_t {
	vmk_ModuleID module_id;
	vmk_Name driver_name;
	vmk_Semaphore instance_lock;
	vmk_ListLinks adapters;
	vmk_HeapID heap_id;
	vmk_HeapID heap_id_dma;
	vmk_LogComponent cnic_vmkLog;

	/* Key-Value vmkmgmt */
	vmk_MgmtApiSignature kvSig;
	vmk_MgmtHandle kv_mgmt_handle;

	vmk_ListLinks cnic_dev_list;
	/* SV: XXX: Move this to per cnic_dev? */
	vmk_Semaphore cnic_lock;

	vmk_Driver cnic_driver;
	vmk_LockDomainID  lock_domain;	/* Lock Domain */
	vmk_TimerQueue timer_queue;		/* Timer Queue */
	vmk_DumpFileHandle dump_handler;
	vmk_Bool debugfs_initialized;
	vmk_Lock drv_lock;
	vmk_TimerQueue delayed_tq;
	ql_vmk_singlethread_workqueue_t *delayed_wq;

	/* SV: XXX: memory tracing */
	vmk_uint64 mem;
	vmk_uint64 count;
};

/* SV: XXX: New data structs. */
typedef struct cnic_fastpath {
	union {
		void							*gen;
		struct status_block_msix		*bnx2;
		struct host_hc_status_block_e1x	*qfle3_e1x;
		struct host_hc_status_block_e2	*qfle3_e2;
	} status_blk;
	vmk_uint32	status_blk_num;
	vmk_uint32	qfle3_igu_sb_id;
} cnic_fastpath_t;

typedef struct msix_handlers {
	void (*intr_handler)(void *, vmk_IntrCookie);
} msix_handlers_t;


/* CNIC Debug log macros. */
#define CNIC_INFO(cdev, dbg_mask, __fmt, __args...)                   \
	do {                                                              \
		if (cnic_debug & dbg_mask) {                                  \
			DRV_DEBUG(VMK_LOG_URGENCY_NORMAL, CNIC_DBG_INFO,          \
				"%s:%d:(%s): " __fmt,                                  \
				__func__, __LINE__, CNIC_IS_NAME_EMPTY(&cdev->name) ? \
				"?" : vmk_NameToString(&cdev->name), ##__args)        \
		}                                                             \
	} while (0)

#define CNIC_WARN(cdev, __fmt, __args...)                         \
	do {                                                          \
		DRV_DEBUG(VMK_LOG_URGENCY_WARNING, CNIC_DBG_WARN,         \
			"%s:%d:(%s): " __fmt,                                  \
			__func__, __LINE__, CNIC_IS_NAME_EMPTY(&cdev->name) ? \
				"?" : vmk_NameToString(&cdev->name), ##__args)    \
	} while (0)


#define CNIC_ERR(cdev, __fmt, __args...)                          \
	do {                                                          \
		DRV_DEBUG(VMK_LOG_URGENCY_ALERT, CNIC_DBG_ALERT,          \
			"%s:%d:(%s): " __fmt,                                  \
			__func__, __LINE__, CNIC_IS_NAME_EMPTY(&cdev->name) ? \
				"?" : vmk_NameToString(&cdev->name), ##__args)    \
	} while (0)


#ifndef DIV_ROUND_UP
#define DIV_ROUND_UP(n,d) (((n) + (d) - 1) / (d))
#endif

#ifndef ISCSI_DEF_FIRST_BURST_LEN
#define ISCSI_DEF_FIRST_BURST_LEN		65536
#endif

#ifndef ISCSI_DEF_MAX_RECV_SEG_LEN
#define ISCSI_DEF_MAX_RECV_SEG_LEN		8192
#endif

#ifndef ISCSI_DEF_MAX_BURST_LEN
#define ISCSI_DEF_MAX_BURST_LEN			262144
#endif

#define ISCSI_DEFAULT_MAX_OUTSTANDING_R2T 	(1)

/* Formerly Cstorm iSCSI EQ index (HC_INDEX_C_ISCSI_EQ_CONS) */
#define HC_INDEX_ISCSI_EQ_CONS			6

/* Formerly Ustorm iSCSI EQ index (HC_INDEX_U_FCOE_EQ_CONS) */
#define HC_INDEX_FCOE_EQ_CONS			3

#define C_SB_ETH_TX_CQ_INDEX			5

#define HC_SP_INDEX_ETH_ISCSI_CQ_CONS		5
#define HC_SP_INDEX_ETH_ISCSI_RX_CQ_CONS	1

#define KWQ_PAGE_CNT	4
#define KCQ_PAGE_CNT	16

#define KWQ_CID 		24
#define KCQ_CID 		25

/*
 *	krnlq_context definition
 */
#define L5_KRNLQ_FLAGS	0x00000000
#define L5_KRNLQ_SIZE	0x00000000
#define L5_KRNLQ_TYPE	0x00000000
#define KRNLQ_FLAGS_PG_SZ					(0xf<<0)
#define KRNLQ_FLAGS_PG_SZ_256					(0<<0)
#define KRNLQ_FLAGS_PG_SZ_512					(1<<0)
#define KRNLQ_FLAGS_PG_SZ_1K					(2<<0)
#define KRNLQ_FLAGS_PG_SZ_2K					(3<<0)
#define KRNLQ_FLAGS_PG_SZ_4K					(4<<0)
#define KRNLQ_FLAGS_PG_SZ_8K					(5<<0)
#define KRNLQ_FLAGS_PG_SZ_16K					(6<<0)
#define KRNLQ_FLAGS_PG_SZ_32K					(7<<0)
#define KRNLQ_FLAGS_PG_SZ_64K					(8<<0)
#define KRNLQ_FLAGS_PG_SZ_128K					(9<<0)
#define KRNLQ_FLAGS_PG_SZ_256K					(10<<0)
#define KRNLQ_FLAGS_PG_SZ_512K					(11<<0)
#define KRNLQ_FLAGS_PG_SZ_1M					(12<<0)
#define KRNLQ_FLAGS_PG_SZ_2M					(13<<0)
#define KRNLQ_FLAGS_QE_SELF_SEQ					(1<<15)
#define KRNLQ_SIZE_TYPE_SIZE	((((0x28 + 0x1f) & ~0x1f) / 0x20) << 16)
#define KRNLQ_TYPE_TYPE						(0xf<<28)
#define KRNLQ_TYPE_TYPE_EMPTY					(0<<28)
#define KRNLQ_TYPE_TYPE_KRNLQ					(6<<28)

#define L5_KRNLQ_HOST_QIDX		0x00000004
#define L5_KRNLQ_HOST_FW_QIDX		0x00000008
#define L5_KRNLQ_NX_QE_SELF_SEQ 	0x0000000c
#define L5_KRNLQ_QE_SELF_SEQ_MAX	0x0000000c
#define L5_KRNLQ_NX_QE_HADDR_HI 	0x00000010
#define L5_KRNLQ_NX_QE_HADDR_LO 	0x00000014
#define L5_KRNLQ_PGTBL_PGIDX		0x00000018
#define L5_KRNLQ_NX_PG_QIDX 		0x00000018
#define L5_KRNLQ_PGTBL_NPAGES		0x0000001c
#define L5_KRNLQ_QIDX_INCR		0x0000001c
#define L5_KRNLQ_PGTBL_HADDR_HI 	0x00000020
#define L5_KRNLQ_PGTBL_HADDR_LO 	0x00000024

#define BNX2_PG_CTX_MAP			0x1a0034
#define BNX2_ISCSI_CTX_MAP		0x1a0074

struct cnic_redirect_entry {
	struct dst_entry *old_dst;
	struct dst_entry *new_dst;
};

struct cnic_work_node {
	vmk_uint32			work_type;
#define	WORK_TYPE_KCQE			1
#define	WORK_TYPE_NEIGH_UPDATE		2
#define	WORK_TYPE_REDIRECT		3
	union {
		struct kcqe		kcqe;
		struct neighbour	*neigh;
		struct cnic_redirect_entry cnic_redir;
	} work_data;
};

#define NET_IF_NAME_SZ	32

#define MAX_CNIC_L5_CONTEXT	128

#define WORK_RING_SIZE		256
#define WORK_RING_SIZE_MASK	255
#define MAX_CM_SK_TBL_SZ	MAX_CNIC_L5_CONTEXT
#define MAX_COMPLETED_KCQE	64

#define MAX_ISCSI_TBL_SZ	128

#define KWQE_CNT (BNX2_PAGE_SIZE / sizeof(struct kwqe))
#define KCQE_CNT (BNX2_PAGE_SIZE / sizeof(struct kcqe))
#define MAX_KWQE_CNT (KWQE_CNT - 1)
#define MAX_KCQE_CNT (KCQE_CNT - 1)

#define MAX_KWQ_IDX	((KWQ_PAGE_CNT * KWQE_CNT) - 1)
#define MAX_KCQ_IDX	((KCQ_PAGE_CNT * KCQE_CNT) - 1)

#define KWQ_PG(x) (((x) & ~MAX_KWQE_CNT) >> (BNX2_PAGE_BITS - 5))
#define KWQ_IDX(x) ((x) & MAX_KWQE_CNT)

#define KCQ_PG(x) (((x) & ~MAX_KCQE_CNT) >> (BNX2_PAGE_BITS - 5))
#define KCQ_IDX(x) ((x) & MAX_KCQE_CNT)

#define QFLE3_NEXT_KCQE(x) (((x) & (MAX_KCQE_CNT - 1)) ==		\
		(MAX_KCQE_CNT - 1)) ?					\
		(x) + 2 : (x) + 1

#define QFLE3_KWQ_DATA_PG(cp, x) ((x) / (cp)->kwq_16_data_pp)
#define QFLE3_KWQ_DATA_IDX(cp, x) ((x) % (cp)->kwq_16_data_pp)
#define QFLE3_KWQ_DATA(cp, x)						\
	&(cp)->kwq_16_data[QFLE3_KWQ_DATA_PG(cp, x)][QFLE3_KWQ_DATA_IDX(cp, x)]

#define DEF_IPID_START		0x8000

#define DEF_KA_TIMEOUT		10000
#define DEF_KA_INTERVAL		300000
#define DEF_KA_MAX_PROBE_COUNT	3
#define DEF_TOS			0
#define DEF_TTL			0xfe
#define DEF_SND_SEQ_SCALE	0
#define DEF_RCV_BUF		0xffff
#define DEF_SND_BUF		0xffff
#define DEF_SEED		0
#define DEF_MAX_RT_TIME		500
#define DEF_MAX_DA_COUNT	2
#define DEF_SWS_TIMER		1000
#define DEF_MAX_CWND		0xffff

#define MAX_IOOO_BLOCK_SUPPORTED	(256)
#define MAX_OOO_RX_DESC_CNT		(BNX2_RX_DESC_CNT * 4)
#define MAX_OOO_TX_DESC_CNT		(BNX2_RX_DESC_CNT * 4)
#define MAX_BNX2_OOO_RX_DESC_CNT	(BNX2_RX_DESC_CNT * 2)
#define MAX_BNX2_OOO_TX_DESC_CNT	(BNX2_RX_DESC_CNT * 2)

#define MAX_RX_OOO_RING			(10)
#define MAX_TX_OOO_RING			(10)

#define BNX2_RXP_SCRATCH_OOO_RX_CID	(BNX2_RXP_SCRATCH + 0x31e0)
#define BNX2_RXP_SCRATCH_OOO_FLAGS	(BNX2_RXP_SCRATCH + 0x31e4)

#define RX_CATCHUP_CID			(RX_CID + 1)
#define TX_CATCHUP_CID			(TX_CID + 2)
#define BNX2_IOOO_FLAGS_ENABLE		(1<<0)
#define BNX2_IOOO_FLAGS_OVERRIDE	(1<<31)

#define TX_OOO_EST_NBD			8

#if (CNIC_ISCSI_OOO_SUPPORT)
/* packet descriptor for 1g, 10g uses skb  */
struct iooo_pkt_desc {
	vmk_ListLinks	list;
	vmk_uint32			pkt_len;
	void			*buf;
	vmk_IOA		mapping;
	vmk_PktHandle		*skb;	/* 10g */
};

struct iooo_block {
	vmk_ListLinks	list;
	vmk_uint32			id;
	vmk_uint32			pkt_cnt;
	struct iooo_pkt_desc	pd_head;
};

struct bnx2_ooo_fhdr {
	vmk_uint8			drop_blk_idx;
	vmk_uint8			drop_size;
	vmk_uint8			opcode;
	vmk_uint8			blk_idx;
	vmk_uint32			icid;
	vmk_uint16			vlan_tag;
	vmk_uint16			pkt_len;
	vmk_uint16			tcp_udp_xsum;
	vmk_uint16			ip_xsum;
};

struct qfle3_ooo_fhdr {
	vmk_uint8			qidx;
	vmk_uint8			pl_offset;
	vmk_uint8			status;
	vmk_uint8			error;
	vmk_uint32			rss_hash;
	vmk_uint16			pkt_len;
	vmk_uint16			vlan;
	vmk_uint16			flags;
	vmk_uint16			bd_len;
	vmk_uint32			cid;
	vmk_uint8			blk_idx;
	vmk_uint8			opcode;
	vmk_uint8			drop_size;
	vmk_uint8			drop_blk_idx;
};

struct iooo_tx_ring_info {
	vmk_uint32			tx_prod_bseq;
	vmk_uint16			tx_prod;
	vmk_uint32			tx_desc_cnt;
	vmk_uint32			tx_desc_cnt_max;

	vmk_uint16			*tx_cons_idx_ptr;
	vmk_uint32			tx_cid_addr;
	vmk_uint32			tx_bidx_addr;
	vmk_uint32			tx_bseq_addr;
	vmk_uint32			tx_buf_size;
	vmk_uint32			tx_max_ring;
	struct iooo_pkt_desc	tx_pend_pd_head;
	vmk_uint32			tx_pend_pd_cnt;
	vmk_uint32			tx_total_pkt_sent;

	struct bnx2_tx_bd	*tx_desc_ring[MAX_TX_OOO_RING];
	struct iooo_pkt_desc	*tx_pkt_desc[MAX_OOO_TX_DESC_CNT];

	vmk_uint16			tx_cons;
	vmk_uint16			hw_tx_cons;

	vmk_IOA		tx_desc_mapping[MAX_TX_OOO_RING];
};

struct iooo_rx_ring_info {
	vmk_uint32			rx_prod_bseq;
	vmk_uint16			rx_prod;
	vmk_uint16			rx_cons;

	vmk_uint16			*rx_cons_idx_ptr;
	vmk_uint32			rx_cid_addr;
	vmk_uint32			rx_bidx_addr;
	vmk_uint32			rx_bseq_addr;

	vmk_uint32			rx_max_ring;

	vmk_uint32			rx_desc_cnt;
	vmk_uint32			rx_desc_cnt_max;
	vmk_uint32			rx_buf_size;

	struct iooo_pkt_desc	*rx_pkt_desc[MAX_OOO_RX_DESC_CNT];
	struct bnx2_rx_bd	*rx_desc_ring[MAX_RX_OOO_RING];

	vmk_IOA		rx_desc_mapping[MAX_RX_OOO_RING];
};

struct iooo_hsi {
	vmk_uint32			iscsi_cid;
	vmk_uint8			blk_idx;
	vmk_uint8			opcode;
	vmk_uint8			drop_size;
	vmk_uint8			drop_blk_idx;
};

struct iooo_hsi_qfle3 {
	vmk_uint32			iscsi_cid;
	vmk_uint8			drop_blk_idx;
	vmk_uint8			drop_size;
	vmk_uint8			opcode;
	vmk_uint8			blk_idx;
};

struct iooo_mgmt {
	vmk_BitVector		*flags;
	/* Control flags */
#define IOOO_RESC_AVAIL         (0)
#define IOOO_START              (1)
	/* Runtime flags */
#define IOOO_BLK_FULL           (10)
#define IOOO_BLK_EMPTY          (11)
	/* - 1G specifics */
#define IOOO_START_HANDLER      (12)
#define IOOO_START_TX_FREE      (13)
	vmk_uint16			blk_cons;
	vmk_uint16			blk_prod;
	vmk_uint16			blk_alloc[MAX_IOOO_BLOCK_SUPPORTED];
	struct iooo_block	blk[MAX_IOOO_BLOCK_SUPPORTED];

	struct iooo_hsi		hsi;
	struct iooo_rx_ring_info rxr;
	struct iooo_tx_ring_info txr;
	vmk_uint32			pkt_buf_size;
	vmk_Lock              lock;
};

#endif

struct cnic_ctx {
	vmk_uint32		cid;
	void		*ctx;
	vmk_IOA	mapping;
};

#define BNX2_MAX_CID		0x2000

struct cnic_dma {
	int		num_pages;
	void		**pg_arr;
	vmk_IOA	*pg_map_arr;
	int		pgtbl_size;
	vmk_uint32		*pgtbl;
	vmk_IOA	pgtbl_map;
};

struct cnic_id_tbl {
	vmk_Lock		lock;
	vmk_uint32		start;
	vmk_uint32		max;
	vmk_uint32		next;
	vmk_uint32		count;
	vmk_BitVector	*table;
};

#define CNIC_KWQ16_DATA_SIZE	128

struct kwqe_16_data {
	vmk_uint8	data[CNIC_KWQ16_DATA_SIZE];
};

struct cnic_iscsi {
	struct cnic_dma		task_array_info;
	struct cnic_dma		r2tq_info;
	struct cnic_dma		hq_info;
#if (CNIC_ISCSI_OOO_SUPPORT)
	struct iooo_block	pen;
	vmk_uint32			blk_cnt;
#endif
};

struct cnic_context {
	vmk_uint32			cid;
	struct kwqe_16_data	*kwqe_data;
	vmk_IOA		kwqe_data_mapping;
	struct ql_vmk_cmpl waitq;
	int			wait_cond;
	unsigned long		timestamp;
	vmk_BitVector		*ctx_flags;
#define	CTX_FL_OFFLD_START	0
#define	CTX_FL_DELETE_WAIT	1
#define	CTX_FL_CID_ERROR	2
	vmk_uint8			ulp_proto_id;
	union {
		struct cnic_iscsi	*iscsi;
	} proto;
};

struct kcq_info {
	struct cnic_dma dma;
	struct kcqe	**kcq;

	vmk_uint16		*hw_prod_idx_ptr;
	vmk_uint16		sw_prod_idx;
	vmk_uint16		*status_idx_ptr;
	vmk_uint32		io_addr;

	vmk_uint16		(*next_idx)(vmk_uint16);
	vmk_uint16		(*hw_idx)(vmk_uint16);
};


struct l5cm_spe;

struct cnic_local {
	vmk_Lock cnic_ulp_lock;
	void *ulp_handle[MAX_CNIC_ULP_TYPE];
	vmk_BitVector *ulp_flags[MAX_CNIC_ULP_TYPE];
#define ULP_F_INIT	0
#define ULP_F_START	1
#define ULP_F_CALL_PENDING	2
	struct cnic_ulp_ops *ulp_ops[MAX_CNIC_ULP_TYPE];

	/* protected by ulp_lock */
	vmk_BitVector *cnic_local_flags;
#define	CNIC_LCL_FL_KWQ_INIT	0x00000001
#define CNIC_LCL_FL_IRQ_REQD    0x00000002
#define CNIC_LCL_FL_HW_START	0x00000003

	struct cnic_dev *dev;

	struct cnic_eth_dev *ethdev;
	struct iro const	*iro_arr;

	vmk_uint32 kwq_cid_addr;
	vmk_uint32 kcq_cid_addr;

	struct cnic_dma	kwq_info;
	struct kwqe		**kwq;

	struct cnic_dma	kwq_16_data_info;

	vmk_uint16		max_kwq_idx;

	vmk_uint16		kwq_prod_idx;
	vmk_uint32		kwq_io_addr;

	volatile vmk_uint16	*kwq_con_idx_ptr;
	vmk_uint16		kwq_con_idx;

	struct kcq_info	kcq1;
	struct kcq_info	kcq2;

	struct cnic_fastpath cfp[1];
	vmk_uint32			int_num;
	vmk_uint32			last_status_idx;
	//ql_vmk_tasklet_t	cnic_irq_task;

	void (*cnic_service_bh)(unsigned long);

	struct kcqe		*completed_kcq[MAX_COMPLETED_KCQE];

	struct cnic_sock 	*csk_tbl;
	struct cnic_id_tbl	csk_port_tbl;
	//vmk_uint32			next_tcp_port;

	struct cnic_dma		gbl_buf_info;

	struct cnic_iscsi	*iscsi_tbl;
	struct cnic_context	*ctx_tbl;
	struct cnic_id_tbl	cid_tbl;
	vmk_atomic64		iscsi_conn;
	vmk_uint32			iscsi_start_cid;

	vmk_uint32			fcoe_init_cid;
	vmk_uint32			fcoe_start_cid;
	struct cnic_id_tbl	fcoe_cid_tbl;

	vmk_uint32			max_cid_space;

	/* per connection parameters */
	int			num_iscsi_tasks;
	int			num_ccells;
	int			task_array_size;
	int			r2tq_size;
	int			hq_size;
	int			num_cqs;

	//struct notifier_block cm_nb;

	struct cnic_work_node	cnic_work_ring[WORK_RING_SIZE];
	int			cnic_wr_cons;
	int			cnic_wr_prod;
	vmk_Lock	wr_lock;

	ql_vmk_tasklet_t	cnic_task;

	ql_vmk_singlethread_delayed_workitem_t delete_task;

	struct cnic_ctx	*ctx_arr;
	int				ctx_blks;
	int				ctx_blk_size;
	unsigned long	ctx_align;
	int				cids_per_blk;

	vmk_uint32	chip_id;
	int			func;

	vmk_EthAddress	srcMACAddr;
	vmk_EthAddress	nextHopMACAddr;
	vmk_uint32		pmtu;
 	vmk_uint32		vlan_id;
	vmk_uint32		srcFamily;
	vmk_uint8		srcIPAddr[IPv6_ADDRLEN];  // IPv6 compatible
	vmk_uint8		src_IP4Addr[IPv4_ADDRLEN]; //IPv4 Compatible
	vmk_uint8		src_subnet_mask[IPv6_ADDRLEN];
	vmk_uint32 		cnic_local_port_min;
	vmk_uint32 		cnic_local_port_nr;

#if (CNIC_ISCSI_OOO_SUPPORT)
	struct iooo_mgmt	iooo_mgmr;
#endif

	struct cnicOps	*cnic_ops;
	struct nicOps	*nic_ops;

	int			(*start_hw)(struct cnic_dev *);
	void		(*stop_hw)(struct cnic_dev *);
	void		(*setup_pgtbl)(struct cnic_dev *, struct cnic_dma *);
	int			(*alloc_resc)(struct cnic_dev *);
	void		(*free_resc)(struct cnic_dev *);
	int			(*start_cm)(struct cnic_dev *);
	void		(*stop_cm)(struct cnic_dev *);
	void		(*enable_int)(struct cnic_dev *);
	void		(*disable_int_sync)(struct cnic_dev *);
	void		(*ack_int)(struct cnic_dev *);
	void		(*close_conn)(struct cnic_sock *, vmk_uint32 opcode);
#if (CNIC_ISCSI_OOO_SUPPORT)
	void		(*stop_ooo_hw)(struct cnic_dev *);
#endif
};

struct qfle3_bd_chain_next {
	vmk_uint32	addr_lo;
	vmk_uint32	addr_hi;
	vmk_uint8	reserved[8];
};

#define ISCSI_RAMROD_CMD_ID_UPDATE_CONN		(ISCSI_KCQE_OPCODE_UPDATE_CONN)
#define ISCSI_RAMROD_CMD_ID_INIT		(ISCSI_KCQE_OPCODE_INIT)

#define CDU_REGION_NUMBER_XCM_AG 2
#define CDU_REGION_NUMBER_UCM_AG 4

#define CDU_VALID_DATA(_cid, _region, _type)	\
	(((_cid) << 8) | (((_region)&0xf)<<4) | (((_type)&0xf)))

#define CDU_CRC8(_cid, _region, _type)	\
	(calc_crc8(CDU_VALID_DATA(_cid, _region, _type), 0xff))

#define CDU_RSRVD_VALUE_TYPE_A(_cid, _region, _type)	\
	(0x80 | ((CDU_CRC8(_cid, _region, _type)) & 0x7f))

#define QFLE3_ISCSI_START_CID		18
#define QFLE3_ISCSI_NUM_CONNECTIONS	128
#define QFLE3_ISCSI_TASK_CONTEXT_SIZE	128
#define QFLE3_ISCSI_CONTEXT_MEM_SIZE	1024
#define QFLE3_ISCSI_MAX_PENDING_R2TS	4
#define QFLE3_ISCSI_R2TQE_SIZE		8
#define QFLE3_ISCSI_HQ_BD_SIZE		64
#define QFLE3_ISCSI_CONN_BUF_SIZE	64
#define QFLE3_ISCSI_GLB_BUF_SIZE	64
#define QFLE3_ISCSI_PBL_NOT_CACHED	0xff
#define QFLE3_ISCSI_PDU_HEADER_NOT_CACHED	0xff

#define QFLE3_FCOE_L5_CID_BASE		MAX_ISCSI_TBL_SZ

#define QFLE3_CONTEXT_MEM_SIZE		1024

#define QFLE3_CHIP_IS_E2_PLUS(A) (CHIP_IS_E2(A) || CHIP_IS_E3(A))

#define QFLE3_SHMEM_MF_BLK_OFFSET	0x7e4

#define QFLE3_SHMEM_ADDR(base, field)	(base + \
		 vmk_offsetof(struct shmem_region, field))

#define QFLE3_SHMEM2_ADDR(base, field)	(base + \
		 vmk_offsetof(struct shmem2_region, field))

#define QFLE3_SHMEM2_HAS(base, field)				\
		((base) &&		 			\
		 (CNIC_RD(dev, QFLE3_SHMEM2_ADDR(base, size)) >	\
		  vmk_offsetof(struct shmem2_region, field)))

#define CNIC_FUNC(cp)			((cp)->func)

#if 0
//SV: XXX:
#define QFLE3_HW_CID(A, x)		((BP_PORT(A) << 23) | \
		 (BP_VN(A) << 17) | (x))
#endif

#define QFLE3_SW_CID(x)			((x) & 0x1ffff)

#define TCP_TSTORM_OOO_MASK			(3<<4)

#define QFLE3_DRIVER		1
#define QFLE3_LEGACY_DRIVER 0

/* SV: XXX: original value HZ/4, but we rounded it off to 1 second. */
#define QFLE3_RAMROD_TMO			(1)
#endif
